
#ifndef ESP8266_H
#define ESP8266_H

#include "ATParser.h"


class ESP8266
{
public:
    ESP8266(PinName tx, PinName rx, bool debug=false);

    /**
    * Startup the ESP8266
    *
    * @param mode of WIFI 1-client, 2-host, 3-both
    * @return true only if ESP8266 was setup correctly
    */
    bool startup(int mode);
    
    /**
    * Start the ESP8266 in server mode
    *
    * @param mode of WIFI 1-client, 2-host, 3-both
    * @param port    
    * @param timeout
    * @return true only if ESP8266 was setup correctly
    */  
    bool startServer(int mode,int port,int timeout);

    /**
    * Reset ESP8266
    *
    * @return true only if ESP8266 resets successfully
    */
    bool reset(void);
    
    /**
    * Get ESP8266 firmware version.
    *
    * @return firmware version.
    */   
    const char *getFirmware(void);

    /**
    * Enable/Disable DHCP
    *
    * @param enabled DHCP enabled when true
    * @param mode mode of DHCP 0-softAP, 1-station, 2-both
    * @return true only if ESP8266 enables/disables DHCP successfully
    */
    
    bool dhcp(bool enabled, int mode);

    /**
    * Connect ESP8266 to AP
    *
    * @param ap the name of the AP
    * @param passPhrase the password of AP
    * @return true only if ESP8266 is connected successfully
    */
    bool connect(const char *ap, const char *passPhrase);

    /**
    * Disconnect ESP8266 from AP
    *
    * @return true only if ESP8266 is disconnected successfully
    */
    bool disconnect(void);

    /**
    * Get the IP address of ESP8266
    *
    * @return null-teriminated IP address or null if no IP address is assigned
    */
    const char *getIPAddress(void);

    /**
    * Get the MAC address of ESP8266
    *
    * @return null-terminated MAC address or null if no MAC address is assigned
    */
    const char *getMACAddress(void);  

    /**
    * Check if ESP8266 is conenected
    *
    * @return true only if the chip has an IP address
    */
    bool isConnected(void);

    /**
    * Open a socketed connection
    *
    * @param type the type of socket to open "UDP" or "TCP"
    * @param id id to give the new socket, valid 0-4
    * @param port port to open connection with
    * @param addr the IP address of the destination
    * @return true only if socket opened successfully
    */
    bool open(const char *type, int id, const char* addr, int port);

    /**
    * Sends data to an open socket
    *
    * @param id id of socket to send to
    * @param data data to be sent
    * @param amount amount of data to be sent - max 2048
    * @return true only if data sent successfully
    */
    bool send(int id, const void *data, uint32_t amount);
    
    /**
    * Sends larger web page data to a connection
    * more controlled with comms failure recovery
    * 
    * @param id id of socket to send to
    * @param data web page data to be sent
    * @param amount amount of data to be sent.
    * No theoretical limit, use for large data pages.
    * Sends in 2048kbit chunks. 
    * @return true only if data sent successfully
    */    
    bool sendWebPage(int id, const char* page, uint32_t amount); 
    
    /**
    * Receives data from an open socket
    *
    * @param id id to receive from
    * @param data placeholder for returned information
    * @param amount number of bytes to be received
    * @return the number of bytes received
    */
    int32_t recv(int id, void *data, uint32_t amount);
    
    /**
    * Receives data from an open web socket
    *
    * @param id id to receive from
    * @param data placed in 4 containers
    * @param char requestType;
    * @param char request;
    * @param int linkId;
    * @param int ipdLen;
    * @return the number of bytes received
    */   
    const char *recvWebRequest(void);    

    /**
    * Closes a socket
    *
    * @param id id of socket to close, valid only 0-4
    * @return true only if socket is closed successfully
    */
    bool close(int id);
    
     /* Return RSSI for active connection
     *
     * @return      Measured RSSI
     */
    int8_t getRSSI();
    
    /**
    * Gets NTP time in seconds since 1970.
    *
    * @param NTPpool url for server e.g. "1.nl.pool.ntp.org"
    * @param tzoffset +/- seconds offset for required time zone e.g. 3600 to add one hour
    * @param setRTC If true, Set's MCU internal RTC to the received seconds since 1970 including offset 
    * @return seconds since 1970
    */   
    int32_t getNTP(char * NTPpool, int tzoffset, int setRTC);
    

    /**
    * Allows timeout to be changed between commands
    *
    * @param timeout_ms timeout of the connection
    */
    void setTimeout(uint32_t timeout_ms);

    /**
    * Checks if data is available
    */
    bool readable();

    /**
    * Checks if data can be written
    */
    bool writeable();
    
    char requestType[6];
    char request[50];
    int linkId;
    int ipdLen;
    int _con_status;
    char _ssid[32];
    char _APIP_buffer[16];
    char _APMAC_buffer[18];
    char _STAIP_buffer[16];
    char _STAMAC_buffer[18];
    char _firmware[200];
    

private:
    BufferedSerial _serial;
    ATParser _parser;
    Timer t;       
};

#endif
