/**
 * @author Serge Sozonoff 
 * @author Aaron Berk
 * Partially based on the work of Aaron Berk for the HMC6352
 *
 * @section LICENSE
 *
 * Copyright (c) 2010 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section DESCRIPTION
 *
 * Honeywell HMC6343 digital compass.
 *
 * Datasheet:
 *
 * http://www.ssec.honeywell.com/magnetic/datasheets/HMC6343.pdf
 */

#ifndef HMC6343_H
#define HMC6343_H

/**
 * Includes
 */
#include "mbed.h"

/**
 * Defines
 */
#define HMC6343_I2C_ADDRESS  0x32 >> 1 //7-bit address

//Commands.
#define HMC6343_EEPROM_WRITE     0xF1
#define HMC6343_EEPROM_READ      0xE1
#define HMC6343_ENTER_SLEEP      0x83
#define HMC6343_EXIT_SLEEP       0x84
#define HMC6343_ENTER_STANDBY    0x76
#define HMC6343_ENTER_RUN        0x75
#define HMC6343_RESET            0x82
#define HMC6343_ENTER_CALIB      0x71
#define HMC6343_EXIT_CALIB       0x7E
#define HMC6343_GET_ACCEL_DATA   0x40
#define HMC6343_GET_MAG_DATA     0x45
#define HMC6343_GET_HEADING_DATA 0x50
#define HMC6343_GET_TILT_DATA    0x55
#define HMC6343_GET_HEADING_DATA 0x50
#define HMC6343_SET_XFWD_PLS_ZUP 0x72
#define HMC6343_SET_XFWD_YUP     0x73
#define HMC6343_SET_ZFWD_NEG_XUP 0x74
#define HMC6343_GET_OPMODE       0x65

//EEPROM locations.
#define HMC6343_SLAVE_ADDR   0x00
#define HMC6343_OPMOD_REG1   0x04
#define HMC6343_OPMOD_REG2   0x05
#define HMC6343_SN_LSB       0x06
#define HMC6343_SN_MSB       0x07
#define HMC6343_DEV_LSB      0x0A
#define HMC6343_DEV_MSB      0x0B
#define HMC6343_VAR_LSB      0x0C
#define HMC6343_VAR_MSB      0x0D
#define HMC6343_XOFFSET_LSB  0x0E
#define HMC6343_XOFFSET_MSB  0x0F
#define HMC6343_YOFFSET_LSB  0x10
#define HMC6343_YOFFSET_MSB  0x11
#define HMC6343_ZOFFSET_LSB  0x12
#define HMC6343_ZOFFSET_MSB  0x13
#define HMC6343_IIRF_LSB     0x14
#define HMC6343_IIRF_MSB     0x15
#define HMC6343_SOFT_VER     0x02


#define HMC6343_X_AXIS 0x01
#define HMC6343_Y_AXIS 0x02
#define HMC6343_Z_AXIS 0x04


// Operation mode bit masks
// LSB
#define HMC6343_COMP    0x80
#define HMC6343_CAL     0x40
#define HMC6343_FILTER  0x20
#define HMC6343_RUN     0x10
#define HMC6343_STDBY   0x08
#define HMC6343_UF      0x04
#define HMC6343_UE      0x02
#define HMC6343_LEVEL   0x01


//Operational mode register masks.
#define HMC6343_CM_MR_1HZ    0x00
#define HMC6343_CM_MR_5HZ    0x100
#define HMC6343_CM_MR_10HZ   0x200


struct Heading {
    float heading;
    float pitch;
    float roll;
};

/**
 * Honeywell HMC6343 digital compass.
 */
class HMC6343 {

public:

    /**
     * Constructor.
     *
     * @param sda mbed pin to use for SDA line of I2C interface.
     * @param scl mbed pin to use for SCL line of I2C interface.
     */
    HMC6343(PinName sda, PinName scl);
    
    HMC6343(I2C& p_itc);

    /**
     * Sample the device and return the result.
     *
     * @return In heading output mode, the current heading as a number between
     *         0-3599, representing 0-359.9 degrees.
     *         In raw magnetometer X output mode, the raw output of the X-axis
     *         magnetometer.
     *         In raw magnetometer Y mode, the raw output of the Y-axis
     *         magnetometer.
     *         In magnetometer X mode, the corrected output of the X-axis
     *         magnetometer.
     *         In magnetometer Y mode, the corrected output of the Y-axis
     *         magnetometer.
     */
    void sampleHeading(Heading* p_heading);

    /**
     * Update bridge offsets.
     *
     * Performs a set/reset immediately.
     */
    void setReset(void);

    /**
     * Enter into or exit from calibration mode.
     *
     * @param enterOrExit 0x45 -> Exit
     *                    0x43 -> Enter
     */
    void setCalibrationMode(int enterOrExit);

    /**
     * Save the current operation mode byte to EEPROM.
     */
    void saveOpMode(void);

    /**
     * Read the memory location on the device which contains the slave address.
     *
     * @return The slave address of the device.
     */
    int getSlaveAddress(void);

    /**
     * Read the current offset for X or Y axis magnetometer.
     *
     * @param axis 0x01 -> X-axis
     *             0x02 -> Y-axis
     *             0x04 -> Z-axis
     * @return The current offset for the axis as a 16-bit number.
     */
    int getOffset(int axis);


    /**
     * Get the software version on the device.
     *
     * @return The software version number.
     */
    int getSoftwareVersion(void);

    /**
     * Get the current operation mode.
     *
     * @return 0x00 -> Standby mode
     *         0x01 -> Query mode
     *         0x02 -> Continuous mode
     */
    int getOpMode(void);

    /**
     * Set the operation mode.
     *
     * @param periodicSetReset 0x00 -> No periodic set/reset
     *                         0x01 -> Periodic set/reset
     */
    void setOpMode(int opMode);


    void setMagneticVariation(float var);
    float getMagneticVariation();
    void setMagneticDeviation(float data);
    float getMagneticDeviation();
    bool isOpModeFlagSet(int flag);
    void setIIRFilter(short data);
    short getIIRFilter();      
    void setMagOffset(int axis, int offset);
    int getMagOffset(int axis);  
    char getMeasurementRate();

private:

    I2C* i2c_;
    int  operationMode_;

    /**
     * Write to EEPROM on the device.
     *
     * @param address Address to write to.
     * @param data Data to write.
     */
    void writeEeprom(int address, int data);

    /**
     * Read EEPROM on the device.
     *
     * @param address Address to read from.
     * @return The contents of the memory address.
     */
    int readEeprom(int address);

    void writeShort(int lsb_address, short data);
    short readShort(int lsb_eprom_address);    
};

#endif /* HMC6343_H */
