/*
 * \file net_interface.h
 * \brief Add short description about this file!!!
 * Copyrigth ARM Ltd 2013
 *
 */

#ifndef NET_INTERFACE_H_
#define NET_INTERFACE_H_
#ifdef __cplusplus
extern "C" {
#endif
/** Platform is not valid for ZigBeeIP library */
#define ZIGBEE_IP_INVALID_PART -2


/**
 *  Socket event description:
 *
 *  8-bit variable where four MSB bits describes the event type and
 *  four LSB bits describes the socket that has received the event.
 *
 *  	Type   Socket ID
 *  	----   ----
 *  	xxxx   xxxx
 *
 */


/** Socket type exceptions */
/** Socket event Mask */
#define SOCKET_EVENT_MASK					0xF0
/** Data received*/
#define SOCKET_DATA							0
/** TCP connection ready */
#define SOCKET_BIND_DONE					1 << 4
/** TCP connection failed */
#define SOCKET_BIND_FAIL					2 << 4
/** TCP connection authentication failed */
#define SOCKET_BIND_AUTH_FAIL 				3 << 4
/** TCP state from listen to establishment */
#define SOCKET_SERVER_CONNECT_TO_CLIENT		4 << 4
/** Socket data send fail */
#define SOCKET_TX_FAIL						5 << 4
/** TCP connection closed */
#define SOCKET_CONNECT_CLOSED				6 << 4
/** TCP connection closed because no ACK received */
#define SOCKET_CONNECT_FAIL_CLOSED			7 << 4
/** No route available to the destination */
#define SOCKET_NO_ROUTE						8 << 4
/** Socket TX Done */
#define SOCKET_TX_DONE						9 << 4

/*!
 * \enum net_interface_type_e
 * \brief Interface type definition.
 */
typedef enum
{
	NET_INTERFACE_ETHERNET, 	/**< IPv4 or IPv6*/
	NET_INTERFACE_WIFI, 		/**< WIFI RF interface*/
	NET_INTERFACE_RF_6LOWPAN, 	/**< RF 6LoWPAN interface*/
	NET_INTERFACE_BLE_6LOWPAN, 	/**< BLE 6LoWPAN interface*/
	NET_INTERFACE_RF_ZIGBEEIP, 	/**< RF ZigBeeIP interface*/
	NET_INTERFACE_G3_6LOWPAN, 	/**< G3 6LoWPAN interface*/
	NET_INTERFACE_IEEE1901_2_6LOWPAN, /**< IEEE1901.2 6LoWPAN interface*/
	NET_INTERFACE_VIRTUAL, 	/**< IPv6 over anyserial interface */
} net_interface_type_e;

/*!
 * \enum net_security_t
 * Network Security Levels
 * */
typedef enum net_security_t
{
  NW_NO_SECURITY = 0,						/**< No Security*/
  NW_SECURITY_LEVEL_MIC32 = 1,				/**< 32-bit Mic verify no Encoding. */
  NW_SECURITY_LEVEL_MIC64 = 2,				/**< 64-bit Mic verify no Encoding. */
  NW_SECURITY_LEVEL_MIC128 = 3,				/**< 128-bit Mic verify no Encoding. */
  NW_SECURITY_LEVEL_ENC = 4,				/**< AES Encoding without MIC. */
  NW_SECURITY_LEVEL_ENC_MIC32 = 5,			/**< 32-bit Mic verify with Encoding. */
  NW_SECURITY_LEVEL_ENC_MIC64 = 6,			/**< 64-bit Mic verify with Encoding. */
  NW_SECURITY_LEVEL_ENC_MIC128 = 7			/**< 128-bit Mic verify with Encoding. */
}net_security_t;

/*!
 * \enum net_address_t
 * \brief addresses for arm_net_address_get().
 */
typedef enum net_address_t
{
  ADDR_IPV6_GP,				/**< Node Default Global address */
  ADDR_IPV6_GP_SEC,			/**< Node Secondary Global address */
  ADDR_IPV6_LL				/**< Node Default Link Local address */
}net_address_t;

typedef enum net_mac_address_t
{
  ADDR_MAC_SHORT16,				/**< Nodes 16-bit Short Address */
  ADDR_MAC_LONG64,				/**< IP layer EUID64 which based on MAC layer 64-bit long Addressafter  U/I -bit conversion  */
}net_mac_address_t;

/*!
 * \enum net_tls_cipher_e
 * \brief TLS cipher mode enumeration types.
 */
typedef enum
{
	NET_TLS_PSK_CIPHER, 		/**< Network Authentication support only PSK */
	NET_TLS_ECC_CIPHER, 		/**< Network Authentication support only ECC */
	NET_TLS_PSK_AND_ECC_CIPHER, /**< Network Authentication support PSK & ECC */
} net_tls_cipher_e;

/*!
 * \enum net_pana_session_mode_e
 * \brief PANA session cache support.
 */
typedef enum
{
	NET_PANA_SINGLE_SESSION, 		/**< Client keep track only 1 Pana session data, Default use case */
	NET_PANA_MULTI_SESSION, 		/**< Client support many Start network coordinator session data */
} net_pana_session_mode_e;

/*!
 * \enum net_6lowpan_link_layer_sec_mode_e
 * \brief 6Lowpan network Security & authentication modes.
 */
typedef enum
{
	NET_SEC_MODE_NO_LINK_SECURITY, 		/**< Security Disabled at Link Layer  DEFAULT */
	NET_SEC_MODE_PSK_LINK_SECURITY, 	/**< Link Security by PSK key*/
	NET_SEC_MODE_PANA_LINK_SECURITY, /**< PANA network authentication define link KEY */
} net_6lowpan_link_layer_sec_mode_e;


/*!
 * \enum net_6lowpan_mode_e
 * \brief 6LoWPAN bootsrap modes.
 */
typedef enum
{
	NET_6LOWPAN_BORDER_ROUTER, 	/**< Root device for 6LoWPAN ND */
	NET_6LOWPAN_ROUTER, 		/**< Router device */
	NET_6LOWPAN_HOST, 			/**< Host device  DEFAULT setting */
	NET_6LOWPAN_SLEEPY_HOST 	/**< Sleepy Host device */
} net_6lowpan_mode_e;

/*!
 * \struct link_layer_setups_s
 * \brief Network Cordinator Parameter List.
 * Structure is used to read network parameter for warm start.
 */
typedef struct link_layer_setups_s
{
	 uint16_t PANId;			/**< Network PAN-ID */
	 uint8_t LogicalChannel;	/**< Network Logical Channel */
	 net_mac_address_t addr_mode;	/**< Coordinator Address mode */
	 uint8_t address[8];		/**< Coordinator Address */
	 uint8_t sf;				/**< Network super frame setup */
}link_layer_setups_s;

/*!
 * \struct link_layer_address_s
 * \brief Network MAC address info.
 * Structure is used to read link layer Adress.
 */
typedef struct link_layer_address_s
{
	 uint16_t PANId;			/**< Network PAN-ID */
	 uint16_t mac_short;		/**< MAC short address if <0xffe then is valid */
	 uint8_t long_euid64[8];	/**< Long64-bit MAC address */
	 uint8_t euid64[8];	       /**< Long64-bit Network ID */
}link_layer_address_s;

/*!
 * \struct network_layer_address_s
 * \brief Network layer parent address info.

 */
typedef struct network_layer_address_s
{
	 uint8_t border_router[16];	/**< ND Border Router Address */
	 uint8_t prefix[8];	       /**< Long64-bit Network ID */
}network_layer_address_s;

/*!
 * \enum net_6lowpan_gp_address_mode_e
 * \brief 6LoWPAN stack address modes.
 */

typedef enum
{
	NET_6LOWPAN_GP64_ADDRESS, 		/**< Interface will register only GP64 Address  DEFAULT */
	NET_6LOWPAN_GP16_ADDRESS, 		/**< Interface will register only GP16 Address */
	NET_6LOWPAN_MULTI_GP_ADDRESS, 	/**< Interface will register GP16 & GP64 Addresss */
} net_6lowpan_gp_address_mode_e;


/*!
 * \struct net_tls_psk_info_s
 * \brief TLS PSK info structure.
 */
typedef struct net_tls_psk_info_s
{
	uint32_t key_id; 	/**< PSK Key ID can be 0x01-0xffff, storage size is intentionally 32 bits */
	uint8_t key[16];	/**< 128-bit PSK Key  */
} net_tls_psk_info_s;

/*!
 * \struct net_link_layer_psk_security_info_s
 * \brief NETWORK PSK link key structure.
 */
typedef struct
{
	uint8_t key_id; 			/**< Link Layer PSK Key ID can be 0x01-0xff */
	uint8_t security_key[16];	/**< Link layer 128-bit PSK Key */
} net_link_layer_psk_security_info_s;


/*!
 * \struct arm_certificate_chain_entry_s
 * \brief Certificate chain structure.
 */
typedef struct
{
	uint8_t chain_length;			/**< Certificate Chain length, indicate chain length */
	const uint8_t *cert_chain[4];	/**< Certificate Chain pointer List */
	uint16_t cert_len[4];			/**< Certificate length */
	const uint8_t *key_chain[4];	/**< Certificate private key */
} arm_certificate_chain_entry_s;

/**

* /struct ns_keys_t

* /brief Struct for the network keys  used by net_network_key_get

*/

typedef struct ns_keys_t

{
     uint8_t previous_active_network_key[16];    /**<  the key that is currently active when a new key is generated and activated */
     uint8_t previous_active_key_index;           /**<  the index associated to the current_active_network_key */
     uint8_t current_active_network_key[16];     /**<  last generated and activated key */
     uint8_t current_active_key_index;           /**<  the index associated to the current_active_network_key */
} ns_keys_t;


/*!
 * \struct border_router_setup_s
 * \brief 6LoWPAN Border Router information structure.
 */
typedef struct
{
	uint8_t channel;				/**< Channel 1-26 802.15.4 Radios */
	uint16_t mac_panid;				/**< Link Layer PAN-id accepts only < 0xfffe  */
	uint16_t mac_short_adr;			/**<  Define 802.15.4 short address if value is <0xffe that indicate that GP16 is activated at */
	uint8_t beacon_protocol_id;		/**< ZigBeeIP uses always 2 */
	uint8_t network_id[16];			/**< network id 16-bytes, will be used at Beacon payload */
    uint8_t lowpan_nd_prefix[8];	/**< Define ND Default prefix, ABRO, DODAG ID, GP address*/
	uint16_t ra_life_time;			/**< Define ND router lifetime in seconds Recommend value 180+. */
	uint16_t abro_version_num;		/**< ND ABRO version number (0 when starting new ND setup) */
} border_router_setup_s;

/**
  * \brief Init nanostack need set core idle call back.
  *
  * \param core_idle Function pointer for nanostack idle call back handler
  *
  * core_idle is a function pointer to the CPU idle time handler. CPU can be set to idle state or in sleep mode if supported by platform.
  * For skipping idle state handling, function can return null. Function first parameter can be as follows:
  *  1. EV_REAYDY_SLEEP then next second parameter tells max sleeptime in milliseconds then function should return sleep time end of the function
  *  2. EV_CORE_IDLE now cpu is safe to set to idle. Application will continue after next interrupt normally. Return anything is possible that case
  *
  * Reference Code for core idle call back function for :
  *
  * @code
  	uint32_t app_ns_core_idle(uint8_t event, uint32_t sleep_time_ms)
	{
		uint32_t returned_slept_time = 0;
		if(event == EV_READY_TO_SLEEP)
		{

			uint32_t wake_up_info = 0;
			uint8_t wakeup_bits = 0;
			//Hal sleep here
			//Returns time spent in sleep mode (milliseconds)
			returned_slept_time = hal_sleep(SLEEPMODE_WAKETIMER, sleep_time_ms, 1);
			//Initialize uart after sleeping
			debug_init(230400, 1);
			wake_up_info = (halGetWakeInfo());
			wakeup_bits = wake_up_info >> 24;
			 if(wakeup_bits & 0x80)
			 {
				 if(wakeup_bits & 1)
				 {
					 debug("Valid SLPCOMPA WU\r\n");
				 }
				 else
				 {
					 net_host_enter_sleep_state_set(0);
					 debug(" S1 wakeup disable sleeping state\r\n");
				 }
			 }
		}
		else // EV_CORE_IDLE
		{
			ATOMIC(halSleepWithOptions(SLEEPMODE_IDLE,0);)
		}
		return returned_slept_time;
	}
  * @endcode
  *
  * Reference Code for Border Router:
  *
  * @code
   uint32_t app_ns_core_idle(uint8_t event, uint32_t sleep_time)
   {
	 //CPU in idle
	 __WFI();
	 return 0;
   }
  * @endcode
  * \return 0, Init OK
  * \return -1, Null Pointer parameter
  * \return ZIGBEE_IP_INVALID_PART, Platform is not valid to Run ZigBeeIP Library
  */
extern int8_t net_init_core(void);



/**
 * \brief Create network interface base to IDLE state.
 * \param type Generates network interface type.
 * \param phy_driver_id Unique PHY device driver ID. PHY driver needs to be allocated first.
 * \param interface_name_ptr String pointer to interface name. Need to end to \0 character. Max length 32 characters including NULL at end.
 *
 * \return >=0 Interface ID (0-127). Application needs to save this information.
 * \return -1 Unsupported interface type.
 * \return -2 Driver is already associated to other interface.
 * \return -3 No memory for interface.
 */
extern int8_t arm_nwk_interface_init(net_interface_type_e type, uint8_t phy_driver_id, char * interface_name_ptr);

/**
 * \brief Set network interface bootsrap setup.
 *
 * \param nwk_interface_id Network interface ID
 * \param bootstrap_mode Selected Bootstrap mode:
 * 		* NET_6LOWPAN_BORDER_ROUTER, Initialize Border router basic setup
 * 		* NET_6LOWPAN_ROUTER, Enable normal 6LoWPAN ND and RPL to bootstrap
 * 		* NET_6LOWPAN_HOST, Enable normal 6LoWPAN ND only to bootstrap
 * 		* NET_6LOWPAN_SLEEPY_HOST, Enable normal 6LoWPAN ND only to bootstrap
 *
 * \param enable_mle_protocol Value 0x01 Enable MLE protocol & Value 0x00 Disable MLE for bootstrap
 * \param br_setup Pointer for border Router basic setup only checked when bootstrap_mode is NET_6LOWPAN_BORDER_ROUTER.
 *
 * \return >=0 Bootstrap mode set OK.
 * \return -1 Unknown network ID.
 * \return -2 Unsupported bootstrap type in this library.
 * \return -3 No Memory for 6LoWPAN stack.
 * \return -4 Null pointer parameter
 */
extern int8_t arm_nwk_interface_configure_6lowpan_bootstrap_set(int8_t nwk_interface_id, net_6lowpan_mode_e bootstrap_mode, uint8_t enable_mle_protocol);

/**
 * \brief Set configured network interface Global address mode (Border router bootstrap mode can't set this).
 *
 * \param nwk_interface_id Network interface ID
 * \param mode efine 6LoWPAN Global Address register mode::
 * 		* NET_6LOWPAN_GP64_ADDRESS, Interface register only GP64
 * 		* NET_6LOWPAN_GP16_ADDRESS, Interface register only GP16
 * 		* NET_6LOWPAN_MULTI_GP_ADDRESS, Interface register GP16 and GP64 addresss. GP16 is primary address and GP64 is secondary.
 *
 * \param short_address_base Short address base. If application defines value 0-0xffd 6LoWPAN try to register GP16 address using that address. 0xfffe and 0xffff will generate random 16-bit short address.
 * 			Border Router device will define here Short Address if it want to use that.
 * \param define_new_short_address_at_DAD This parameter is only checked when mode is not NET_6LOWPAN_GP64_ADDRESS and short_address_base is 0-0xffd. Recommend value is 1 that will enable automatic new address definition at Duplicate Address Detection(DAD). Value 0 will generate Duplicate Adreress Detection error for interface bootstrap.
Border Router Device will not check that part.
 *
 * \return >=0 Bootstrap mode set OK.
 * \return -1 Unknown network ID.
 * \return -2 Illegal for Border Router
 * \return -3 No Memory for 6LoWPAN stack.
 */
extern int8_t arm_nwk_6lowpan_gp_address_mode(int8_t nwk_interface_id, net_6lowpan_gp_address_mode_e mode, uint16_t short_address_base, uint8_t define_new_short_address_at_DAD);

extern int8_t arm_nwk_6lowpan_link_scan_paramameter_set(int8_t nwk_interface_id, uint32_t channel_mask, uint8_t scan_time);

extern int8_t arm_nwk_6lowpan_link_panid_filter_for_nwk_scan(int8_t nwk_interface_id, uint16_t pan_id_filter);

/**
  *  \brief Get current used channel.
  *
  * \return 11-26 = Active channel
  * \return 0 = Radio is closed
  */
extern uint8_t arm_net_get_current_channel(int8_t nwk_interface_id);

/**
 * \brief A function to read pan ID filter.
 * \return 16-bit value indicating a pan ID filter.
 */
extern uint16_t arm_net_get_nwk_pan_id_filter(int8_t nwk_interface_id);
/**
  * \brief Enable/Disable Network ID filter.
  *
  * \param nwk_interface_id Network interface ID.
  * \param nwk_id is a pointer a new network id filter NULL Disable filter.
  *
  * \return 0 on success.
  * \return -1 Unknown network ID.
  * \return -2 Interface Active.
  */
extern int8_t arm_nwk_6lowpan_link_nwk_id_filter_for_nwk_scan(int8_t nwk_interface_id, uint8_t *nwk_id_filter);
/**
  * \brief Enable/Disable Network protocol ID filter.
  *
  * \param nwk_interface_id Network interface ID.
  * \param protocol_ID is value which filter only supported network protocols (0= Zigbee1.x, 2= ZigBeeIP).
  *
  * \return 0 on success.
  * \return -1 Unknown network ID.
  * \return -2 Interface Active.
  */
extern int8_t arm_nwk_6lowpan_link_protocol_id_filter_for_nwk_scan(int8_t nwk_interface_id, uint8_t protocol_ID);

/**
  * \brief Init & Configure interface security mode.
  *
  * \param nwk_interface_id Network interface ID.
  * \param mode define Link layer security mode.
  *  NET_SEC_MODE_NO_LINK_SECURITY, No security
  *  NET_SEC_MODE_PSK_LINK_SECURITY, Predefined PSK link layer key and ID
  *  NET_SEC_MODE_PANA_LINK_SECURITY, Define Pana Bootstrap Network Authentication
  *
  * \param sec_level Define security level only will be checked when mode is not NET_SEC_MODE_NO_LINK_SECURITY
  * \param psk_key_info pointer for PSK link layer keys only checked when mode is NET_SEC_MODE_PSK_LINK_SECURITY
  *
  * \return 0 on success.
  */

extern int8_t arm_nwk_link_layer_security_mode(int8_t nwk_interface_id, net_6lowpan_link_layer_sec_mode_e  mode, uint8_t sec_level, net_link_layer_psk_security_info_s *psk_key_info);
/**
  * \brief Init & Configure interface PANA network Client.
  *
  * \param nwk_interface_id Network interface ID.
  * \param cipher_mode define TLS 1.2 Cipher mode PSK, ECC or both.
  * \param psk_key_id PSK KEY id for PSK Setup
  *
  * \return 0 on success.
  * \return -1 Unknown network ID.
  * \return -2 Interface Active.
  */
extern int8_t arm_pana_client_library_init(int8_t nwk_interface_id, net_tls_cipher_e cipher_mode, uint32_t psk_key_id);

/**
  * \brief Init & Configure interface PANA network Server.
  *
  * \param nwk_interface_id Network interface ID.
  * \param cipher_mode define TLS 1.2 Cipher mode PSK, ECC or both.
  * \param key_material pointer
  *
  * \return 0 on success.
  * \return -1 Unknown network ID.
  * \return -2 Interface Active.
  */
extern int8_t arm_pana_server_library_init(int8_t nwk_interface_id, net_tls_cipher_e cipher_mod, uint8_t *key_material, uint32_t time_period_before_activate_key);

/**
  * \brief Client Manually Pana client key Pull for test purpose.
  *
  * \param nwk_interface_id Network interface ID.
  *
  * \return 0 on success.
  * \return -1 Unknown network ID.
  */
extern int8_t arm_pana_client_key_pull(int8_t nwk_interface_id);


/**
 * \brief Trig Network Key Update process
 *
 * Function will deliver new network key to all ZigBee Routers which have registered GP address to server.
 * Function call always trig new key-id. Key delivery will be started between 300ms interval between nodes.
 * This function does not cause any traffic  if Server does not have any Routers device sessions.
 *
 *\param network_key_material pointer to new 128-bit key material or NULL generate random key.
 *
 * \return 0 Key Update process OK
 * \return -1 Pana server is not Initialized yet
 * \return -2 Old Key Update still active
 * \return -3 Memory allocation fail
 */
extern int8_t arm_pana_server_key_update(int8_t nwk_interface_id, uint8_t * network_key_material);

extern int8_t arm_pana_activate_new_key(int8_t nwk_interface_id);


/**
 * \brief Read Pana server security key material
 *
 * previous_active_network_key is information is only valid when current_active_key_index is bigger than 1.
 *
 *\param key pointer for store keymaterial information.
 *
 * \return 0 Key Read OK
 * \return -1 Pana server key material not available
 */
extern int8_t arm_network_key_get(int8_t interface_id, ns_keys_t *key);
/**
 * \brief Start network interface bootstrap.
 *
 * \param nwk_interface_id Network interface ID
 *
 *
 * \return >=0 Bootstrap Start OK.
 * \return -1 Unknown network ID.
 * \return -2 Not configured.
 * \return -3 Active.
 */
extern int8_t arm_nwk_interface_up(int8_t nwk_interface_id);
/**
 * \brief Stop and set interface to idle.
 *
 * \param nwk_interface_id Network interface ID
 *
 *
 * \return >=0 Process OK.
 * \return -1 Unknown network ID.
 * \return -3 Not ActiveActive.
 */
extern int8_t arm_nwk_interface_down(int8_t nwk_interface_id);

/**
 * \brief Define Border Router MAC & 6LoWPAN ND setup for selected interface
 *
 * \param nwk_interface_id Network interface ID
 * \param border_router_setup_ptr Pointer to MAC & 6LoWPAN ND setup
 *
 */
extern int8_t arm_nwk_6lowpan_border_router_init(int8_t nwk_interface_id, border_router_setup_s *border_router_setup_ptr);
/**
 * \brief Add context at 6Lowpan interface configure state
 *
 *
 * \param Network Interface ID
 * \param c_id_flags bit 5 indicate compress support and bit 0-3 context ID
 * \param context_len Context length in bits need to be 64-128
 * \param ttl contexts time to live value in minutes
 * \param context_ptr pointer to full 128-bit memory area.
 *
 * \return 0 context update OK
 * \return -1 No memory for new Context
 * \return -2 Border Router base not allocated
 * \return -3 Given parameter fail (c_id_flags > 0x1f or contex_len < 64)
 * \
 */
extern int8_t arm_nwk_6lowpan_border_router_context_update(int8_t nwk_interface_id, uint8_t c_id_flags, uint8_t context_len, uint16_t ttl, uint8_t *context_ptr);
/**
 * \brief Update Run time configured context
 *
 * Function can change contexts Compress state or Time to live value.
 * Function will trig New ABRO version number, which indicates that ND parameters are updated.
 *
 * \param Network Interface ID
 * \param c_id context ID stack will check first 4 bits , supported values 0-15
 * \param compress_mode 0=Compress disabled otherwise compress enabled
 * \param ttl contexts time to live value in minutes
 *
 * \return 0 Update OK
 * \return -1 Update fail by Router state
 *
 */
extern int8_t arm_nwk_6lowpan_border_router_context_parameter_update(int8_t nwk_interface_id, uint8_t c_id, uint8_t compress_mode, uint16_t ttl);
/**
 * \brief Delete Allocated Context by id
 *
 * \param Network Interface ID
 * \param c_id 4-bit context id which will be deleted
 *
 * \return 0 Context Delete OK.
 * \return -1 Delete process fail.
 */
extern int8_t arm_nwk_6lowpan_border_router_context_remove_by_id(int8_t nwk_interface_id, uint8_t c_id);
/**
 * \brief Update ND ABRO version number
 *
 * \param Network Interface ID
 *
 * \return 0 ABRO version update OK
 * \return -1 ABRO update fail( Interface is not Up yet or Border Router base is not allocated)
 */
extern int8_t arm_nwk_6lowpan_border_router_configure_push(int8_t nwk_interface_id);

extern int8_t arm_nwk_6lowpan_border_route_nd_default_prefix_timeout_set(int8_t nwk_interface_id, uint32_t time);

/**
 * \brief A function to read network layer configurations.
 * \param network_params is a pointer to the structure to where the network layer configs are written to.
 * \return 0 on success.
 * \return -1 if PAN coordinator is not known.
 */
int8_t arm_nwk_param_read(int8_t interface_id, link_layer_setups_s * network_params);


/**
 * \brief A function to read MAC PAN-ID, Short address & EUID64
 * \param mac_params is a pointer to the structure to where the mac address are written to.
 * \return 0 on success.
 * \return -1 .
 */
int8_t arm_nwk_mac_address_read(int8_t interface_id, link_layer_address_s * mac_params);


/**
 * \brief A function to read 6LoWPAN ND border router address and NWK prefix
 * \param mac_params is a pointer to the structure to where the mac address are written to.
 * \return 0 on success.
 * \return -1 .
 */
int8_t arm_nwk_nd_address_read(int8_t interface_id, network_layer_address_s * nd_params);


/**
 * \brief A function to read networking address informations.
 * \param addr_id identifies the address information type to be read.
 * \param address is a pointer to a structure to where the address information is written to.
 * \return 0 on success, -1 on failure.
 */
extern int8_t arm_net_address_get(int8_t nwk_interface_id, net_address_t addr_id, uint8_t *address);

/** Border Router ND NVM update types */
#define ND_PROXY_CONTEXT_NVM_UPDATE 		0 /* ND Context Update, 20  bytes data behind pointer */
#define ND_PROXY_CONTEXT_FLAGS_NVM_UPDATE 	1 /* ND Context Update flags update */
#define ND_PROXY_CONTEXT_NVM_REMOVE 		2 /* ND Context Removed */
#define ND_PROXY_PREFIX_NVM_UPDATE  		3 /* ND Prefix Update */
#define ND_PROXY_ABRO_VERSION_NVM_UPDATE  	4 /* ND ABRO version updated */
/**
 * \brief Border Router ND setup NVM update callback set
 *
 * \param passed_fptr function pointer to ND NVM update process
 *
 * \return 0 ND NVM operation Init OK.
 * \return -1 No memory for NVM buffer.
 */
extern int8_t border_router_nd_nvm_callback_set(void (*passed_fptr)(uint8_t ,uint8_t *));
/**
 * \brief Load context from NVM at ZigBeeIP interface configure state
 *
 *
 * \param Network Interface ID
 * \param context_data pointer to proper builded 20 bytes update array.
 *
 * \return 0 context reload OK
 * \return <0 Load fail.
 */
extern int8_t arm_nwk_6lowpan_border_router_nd_context_load(int8_t nwk_interface_id, uint8_t * contex_data); //NVM

extern int8_t arm_bootstrap_certificate_chain_set(arm_certificate_chain_entry_s *chain_info);
extern int8_t arm_network_certificate_chain_set(arm_certificate_chain_entry_s *chain_info);
extern int8_t arm_transport_certificate_chain_set(arm_certificate_chain_entry_s *chain_info);

/**
  * \brief Get the library version information.
  *
  * \param *ptr Pointer where the data is stored. Required size is 20 bytes.
  *
  *
  * The array containing the version information has the following structure.
  *
  *	 | Platform type | Version | Build ID |
  *	 | :-----------: | :----------------: |
  *  | 1 byte        | 	1 byte | 4 bytes  |
  *
  * Example how to read library version information is described below:
  *
  * @code
	uint8_t i;
	uint8_t app_version_info[6];
	net_get_version_information(app_version_info);
	debug("Platform Type: ");
	debug_hex(app_version_info[0]);
	debug(", Version: ");
	debug_hex(app_version_info[1]);
	debug(" ,Number:0x");
	for(i=2; i<6; i++)
	{
		debug_hex(app_version_info[i]);
	}
	debug("\r\n");
  * @endcode
  *
  */
extern void net_get_version_information(uint8_t *ptr);
#ifdef __cplusplus
}
#endif
#endif /* NET_INTERFACE_H_ */
