#ifndef _NS_SOCKET_API_H
#define _NS_SOCKET_API_H

/*
 * Copyright Sensinode Ltd 2011
 * socket_api.h
 */
 #ifdef __cplusplus
extern "C" {
#endif
/**
 * \file socket_api.h
 * \brief NanosStack 2.0 Library Socket API
 *
 * \section socket-com Common Socket API
 *	- socket_open(), A function to open a socket
 *	- socket_free(), A function to free a socket
 *	- socket_raw_open(), A function to open a raw ICMP socket
 *
 * \section socket-sec Set socket TLS chipher suite list (SOCKET_TCP_TLS socket Type)
 *  - sec_socket_set_chipher_suite_list(), Used to set opened socket TLS chiphersuite support
 *   * check socket_security.h more information
 *
 * \section socket-read Socket Read API at Callback
 *	- socket_read(), A function to read received data buffer from a socket
 *	- socket_read_info(), A function to read signal information and length info from a socket
 *	- socket_read_session_address(), A function to read a session info for TCP event
 *
 * \section socket-tx Socket TX API
 * - socket_send(), A function to write data buffer to a socket (TCP Client Only)
 * - socket_sendto(), A function to write data to socket to specific destination
 *
 * \section sock-connect TCP Socket Connection Handle
 *  - socket_listen(), A function to set the socket to the listening mode
 *  - socket_connect(), A function to connect to remote peer
 *  - socket_close(), A function to close a connection
 *
 * Sockets are common abstraction model for network communication and are used in most operating systems.
 * The NanoStack2.0 Library API follows BSD socket API conventions closely with some extensions necessitated by the event-based scheduling model.
 *
 * Library supports following socket types:
 * | Socket name    |	Socket description           |
 * | :------------: | :----------------------------: |
 * | SOCKET_UDP     |	UDP socket type              |
 * | SOCKET_TCP     |	TCP socket type              |
 * | SOCKET_TCP_TLS |	TCP socket with TLS security |
 *
 * \section socket-raw ICMP RAW Socket Instruction
 * ICMP RAW socket can be created by socket_raw_open(). When using ICMP sockets the minimum payload length is 8 bytes, where first 4 bytes form the ICMP header.
 * The first byte is for type, second for code and last two are for checksum which must always be set to zero.
 * Stack will calculate the checksum automatically before transmitting the packet. Header is followed by the payload.

 * | ICMP Type |	ICMP Code | Checksum  | Payload    | Notes           |
 * | :-------: | :----------: | :-------: | :--------: | :--------------:|
 * | 1         | 1            | 2         | n (min. 4) | Length in bytes |
 * | 0xXX      | 0xXX         | 0x00 0x00 | n bytes    | Transmit		 |
 * | 0xXX      | 0xXX         | 0xXX 0xXX | n bytes    | Receive         |
 *
 * ICMP echo request with 4 bytes of payload (ping6).
 *
 * | ICMP Type |	ICMP Code | Checksum  | Payload             |
 * | :-------: | :----------: | :-------: | :-----------------: |
 * | 0x80      | 0x00         | 0x00 0x00 | 0x00 0x01 0x02 0x03 |
 *
 * ICMP echo response for the message above.
 *
 * | ICMP Type |	ICMP Code | Checksum  | Payload             |
 * | :-------: | :----------: | :-------: | :-----------------: |
 * | 0x81      | 0x00         | 0xXX 0xXX | 0x00 0x01 0x02 0x03 |
 *
 * \section socket-receive Socket receiving
 * When there is data to read from the socket, a receive callback function is called with the socket event parameter from the library.
 * The state of the TCP socket then changes and the socket TX process is ready (SOCKET_TX_FAIL or SOCKET_TX_DONE).
 * Socket event has event_type and socket_id fields. The receive callback function must be defined when socket is opened using socket_open() API.
 *
 * All supported socket event types are listed here:
 *
 * | Event Type                 | Value | Description                                                         |
 * | :------------------------: | :---: | :-----------------------------------------------------------------: |
 * | SOCKET_EVENT_MASK          | 0xF0  | NC Socket event mask                                                |
 * | SOCKET_DATA                | 0x00  | Data received, Read data length from d_len field                    |
 * | SOCKET_BIND_DONE           | 0x10  | TCP connection ready                                                |
 * | SOCKET_TX_FAIL             | 0x50  | Socket data send failed                                             |
 * | SOCKET_CONNECT_CLOSED      | 0x60  | TCP connection closed                                               |
 * | SOCKET_CONNECT_FAIL_CLOSED | 0x70  | TCP connection closed  no ACK received                             |
 * | SOCKET_NO_ROUTER           | 0x80  | No route available to destination                					  |
 * | SOCKET_TX_DONE             | 0x90  | Last Socket TX process Done, at TCP case Whole TCP Process is ready |
 *
 * When socket call back is called application need to handle event and read data from the socket socket_read().
 * Read operation release memory from stack when read is called or after callback automatic.
 *
 * Example for UDP Socket Callback handler:
 * @code
 	void main_udp_receive(void * cb)
	{
		socket_callback_t * cb_res =0;
		int16_t length;
		cb_res = (socket_callback_t *) cb;

		if(cb_res->event_type == SOCKET_DATA)
		{
			uint8_t *payload = 0;
			//Read Data and Handle Dta
			//error = socket_read_info(ap_sock_id, &info);
			if ( cb_res->d_len > 0)
			{
				//Read data to the RX buffer
				payload = (uint8_t *) MEM_ALLOC(cb_res->d_len);
				if(payload)
				{
					//Read data to the RX buffer
					length = socket_read(cb_res->socket_id, &app_src, payload, cb_res->d_len);
					if(length)
					{

						uint8_t * ptr = payload ;
						//Handles data received in UDP
						debug("UDP Data From:");
						printf_ipv6_address(&(app_src.address[0]));

						if(app_src.identifier == 7)
						{
							debug("Echo Response::");
							printf_array(ptr , length);
						}
						else
						{
							debug("Data:");
							printf_string(ptr , length);
						}

					}
					MEM_FREE(payload);
				}

			}
			else
			{
				debug("No Data\r\n");
				socket_read(cb_res->socket_id, &app_src, NULL, 0);
			}
		}
		else
		{
			if(cb_res->event_type == SOCKET_TX_DONE)
			{
				debug("UDP TX Done\r\n");
			}
			else
			{
				//No Route to Packet Destination Address
				if(cb_res->event_type == SOCKET_NO_ROUTE)
				{
					debug("ND/RPL not ready\r\n");
				}
				//Link Layer TX Fail for socket packet
				else if(cb_res->event_type == SOCKET_TX_FAIL)
				{
					debug("Link Layer Tx fail\r\n");
				}

			}
		}
	}
 * @endcode
 *
 * Example for TCP Socket Callback handler:
 *
 * @code
 	 void main_tcp_receive(void * cb)
	{
		socket_callback_t * cb_res =0;
		int16_t length;
		cb_res = (socket_callback_t *) cb;

		if(cb_res->event_type == SOCKET_DATA)
		{
			uint8_t *payload = 0;
			//Read Data
			if ( cb_res->d_len > 0)
			{
				//Read data to the RX buffer
				payload = (uint8_t *) MEM_ALLOC(cb_res->d_len);
				if(payload)
				{
					//Read data to the RX buffer
					length = socket_read(cb_res->socket_id, &app_src, payload, cb_res->d_len);
					if(length)
					{

						uint8_t * ptr = payload ;
						//Handles data received in TCP socket

						printf_ipv6_address(&(app_src.address[0]));
						//...App spesific

					}
					MEM_FREE(payload);
				}
			}
			else
			{
				debug("No Data\r\n");
				socket_read(cb_res->socket_id, &app_src, NULL, 0);
			}
		}
		else
		{
			//TCP socket connection is ready
			if(cb_res->event_type == SOCKET_BIND_DONE)
			{
				debug("Connect Ready\r\n");
				if(socket_read_session_address(cb_res->socket_id, &app_src) == 0)
				{
					print_opened_port(app_src.identifier);
					debug(", ");
					printf_ipv6_address(&(app_src.address[0]));
				}
				//...App spesific
			}
			//TCP socket connection is closed
			else if(cb_res->event_type == SOCKET_CONNECT_CLOSED)
			{
				debug("Connection Closed\r\n");

			}
			else if(cb_res->event_type == SOCKET_CONNECT_FAIL_CLOSED)
			{
				//TCP socket connection lost
				debug("TCP Connect Fail:No ACK\r\n");
			}
			else if(cb_res->event_type == SOCKET_TX_DONE)
			{
				if(socket_read_session_address(cb_res->socket_id, &app_src) == 0)
				{
					//Close Connection
					socket_close(cb_res->socket_id, &app_src,0);
				}
			}
			else
			{
				//No Route to Packet Destination Address
				if(cb_res->event_type == SOCKET_NO_ROUTE)
				{
					debug("ND/RPL not ready\r\n");
				}
				//Link Layer TX Fail for socket packet
				else if(cb_res->event_type == SOCKET_TX_FAIL)
				{
					debug("Link Layer Tx fail\r\n");
				}
			}
		}
	}
 * @endcode
 *
 * \section socket-tcp How to Use TCP sockets
 *
 * | API                           | Socket Type   | Description                                                    |
 * | :---------------------------: | :-----------: | :------------------------------------------------------------: |
 * | socket_open()                 | Server/Client | Open socket to specific or dynamically port number.              |
 * | socket_close()                | Server/Client | Close opened TCP connection 					                  |
 * | socket_listen()               | Server        | Set server port to listen state 				                  |
 * | socket_connect()              | Client        | Connect client socket to specific destination 	              |
 * | socket_send()                 | Client        | Send data to session based destination 	                      |
 * | socket_sendto()               | Server/Client | Send data to specific destination 	                          |
 * | socket_read_session_address() | Server/Client | Function read socket TCP session address and port information. |
 *
 * When TCP socket is opened it is in closed state. It must be set either listen or to connect state before it can be used to receive or transmit data.
 * SOCKET_TCP_TLS socket type is for secured TCP communication. Socket default TLS cipher suite is PSK. Change supported cipher suites can be set by sec_socket_set_chipher_suite_list() function.
 *
 * Socket can be set to listen mode by calling socket_listen() function. After the call the socket can accept an incoming connection from a remote host.
 * Listen mode closes the connection automatically after server timeout or when client or application closes the connection manually by socket_close() function.
 *
 * TCP socket can be connected to a remote host by calling socket_connect() with correct arguments. After the function call (non-blocking) application must wait for the socket event to confirm the successful state change of the socket.
 * After the event of successful state change data can be sent using socket_send()  by client and socket_send() by server call.
 * Connection can be closed by calling socket_close() function or by server timeout.
 *
 * \section socket-udpicmp How to Use UDP and RAW Socket
 *
 * UDP socket is ready to receive and send data immediately after successful call of socket_open() and NET_READY event is received.
 * Data can be transmitted by using socket_sendto() function. ICMP socket works with same function call.
 *
 */

#include "ns_address.h"

/** Protocol IDs used when opened sockets*/
/** Doesn't reserve anything. Socket in free. */
#define SOCKET_NONE 	0
/** UDP socket type */
#define SOCKET_UDP 		17
/** Normal TCP socket type */
#define SOCKET_TCP 		6
/** TCP socket with TLS authentication */
#define SOCKET_TCP_TLS 	7

/**
 * \brief A function to open a raw ICMP socket.
 *
 * \param passed_fptr pointer to socket callback function
 *
 * \return positive socket opened
 * \return eFALSE no free sockets
 * \return eBUSY port reserved
 */
extern int8_t socket_raw_open(void (*passed_fptr)(void *));

#define SOCKET_ICMP		32
/** ICMP Socket instruction
 *
 * If used ICMP socket, minimum payload length is 8-bytes, where first 4-bytes form ICMP header.
 * First byte is for type, second for code and last two are for checksum witch must always be zero.
 * Stack will calculate the checksum automatically when send. Payload comes after the header.
 *
 * This applies for reading and sending.
 *
 * 		ICMP TYPE	ICMP Code	ICMP Checksum (2-bytes) 	Payload n-bytes (min. 4 bytes)
 * 		--------    --------	-------- --------			-------- -------- ....... -------
 * 		0xXX 		0xXX		0x00	0x00				0xXX	0xXX	......	0xXX
 *
 * Example data flow for ping:
 *
 * ICMP echo request with 4-bytes payload (Ping6).
 * 		0x80, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03
 *
 * ICMP echo response for the message above.
 * 		0x81, 0x00, 0xXX, 0xXX, 0x00, 0x01, 0x02, 0x03
 */

#ifndef SOCKET_CALLBACK_T
#define SOCKET_CALLBACK_T
 /*!
  * \struct socket_callback_t
  * \brief Socket Callback function structure type.
  */
typedef struct socket_callback_t
{
  uint8_t event_type; /**< Socket Callback Event check list below*/
  uint8_t socket_id; /** Socket id queue which socket cause call back */
  uint16_t	d_len;	/** Data length if event type is SOCKET_DATA */
  uint8_t LINK_LQI; /** LINK LQI info if interface cuold give */
} socket_callback_t;
#endif

/**
 * \brief Used to initialize a socket for communication.
 *
 * Note: Maximum of seven sockets at the same time is supported.
 *
 * \param protocol defines the protocol to use.
 * \param identifier is e.g. in case of UDP, the UDP port.
 * \param passed_fptr is a function pointer to a function that is called whenever a data frame is received to this socket.
 *
 * \return 0 or greater on success; return value is the socket ID.
 * \return -1 on failure.
 *
 * Note: PROTOCOL_TCP_TLS socket default TLS chiphersuite is PSK.
 */
extern int8_t socket_open(uint8_t protocol, uint16_t identifier, void (*passed_fptr)(void *));

/**
 * \brief A function to free a socket.
 *
 * \param socket ID of the socket to be released.
 *
 * \return 0 socket released.
 * \return -1 socket not released.
 */
extern int8_t socket_free(int8_t socket);

/**
 * \brief A function to set the socket to the listening mode.
 *
 * \param socket socket ID to bind
 * \return 0 on success.
 * \return -1 on failure.
 */
extern int8_t socket_listen(int8_t socket);

/**
 * \brief A function to connect to remote peer (TCP).
 *
 * Used by the application to send data.
 *
 * \param socket socket id
 * \param address address of remote peer
 * \param randomly_take_src_number 1=enable find next free random port number for current one
 *
 * \return eOK done
 * \return eFALSE invalid socket id
 */
extern int8_t socket_connect(int8_t socket, ns_address_t *address, uint8_t randomly_take_src_number);

/**
 * \brief A function to close a connection.
 *
 * \param socket refers to the socket ID to close.
 * \param address refers to the destination client address. When using as a client, a null pointer shall be passed.
 * \param release_security_session refers to the security session id.
 *
 * \return 0 on success.
 * \return -1 if a given socket ID is not found, if a socket type is wrong or tcp_close() returns a failure.
 * \return -2 if no active tcp session was found.
 * \return -3 if referred socket ID port is declared as a zero.
 *
 * Note: randomly_take_src_number is highly recommend to use always. Stack will generate new source port between 49152-65534.
 */
extern int8_t socket_close(int8_t socket,ns_address_t *address, uint8_t release_security_session);

/**
 * \brief Used to send data via a connected TCP socket by client.
 *
 * Note: Socket connection must be ready before using this function.
 * Stack will automatically use the address of the remote connected host as the destination address for the packet.
 *
 * \param socket socket id
 * \param buffer pointer to data
 * \param length data length
 *
 * \return 0 done
 * \return -1 invalid socket id
 * \return -2 Socket memory allocation fail
 * \return -3 TCP state not established
 * \return -4 Socket tx process busy
 * \return -5 TLS authentication not ready
 * \return -6 Packet too short
 *
 * Server need to use always specific address information when it close socket.
 * Client may set address pointer to NULL.
 *
 */
extern int8_t socket_send(int8_t socket, uint8_t *buffer, uint16_t length);

/**
 * \brief A function to read received data buffer from a socket.
 *
 * Used by the application to get data from a socket.
 *
 * \param socket refers to socket ID to be read from.
 * \param address is a pointer to the address structure, allocated by an application; source address is copied into this structure.
 * \param buffer is a pointer to an array to where the read data is written to.
 * \param length is the maximum length of the allocated buffer.
 *
 * \return 0 if no data is read.
 * \return greater than 0, indicating the length of the data copied to buffer.
 *
 */
extern int16_t socket_read(int8_t socket, ns_address_t *address, uint8_t *buffer, uint16_t length);

/**
 * \brief Used to send UDP, TCP or raw ICMP data via the socket.
 *
 * Used by the application to send data.
 *
 * \param socket refers to socket ID.
 * \param address is a pointer to the destination address information.
 * \param buffer is a pointer to a data to be sent.
 * \param length is a length of the data to be sent.
 * \return 0 on success.
 * \return -1 invalid socket id.
 * \return -2 Socket memory allocation fail.
 * \return -3 TCP state not established.
 * \return -4 Socket tx process busy.
 * \return -5 TLS authentication not ready.
 * \return -6 Packet too short.
 */
extern int8_t socket_sendto(int8_t socket, ns_address_t *address, uint8_t *buffer, uint16_t length);

/**
 * \brief A function to read signal information and length info from a socket.
 *
 * Used by the application to read signal info.
 *
 * \param socket socket id
 * \param info pointer to response data
 *
 * \return eOK done
 * \return eFALSE invalid socket id
 */
//extern int8_t socket_read_info(int8_t socket, rf_info_t *info);
/**
 * \brief A function to read a session info for TCP event.
 *
 *
 * \param socket refers to the ID to be read from.
 * \param address is a pointer to the address structure to where the session address information is read to.
 *
 * \return 0 on success.
 * \return -1 if no socket is found or TCP is not compiled into this project.
 * \return -2 if no session information is found.
 *
 * Note: Function should be called only at Socket call back call when socket event is SOCKET_BIND_DONE or SOCKET_TX_DONE.
 * The following sections introduce those functions.
 */
extern int8_t socket_read_session_address(int8_t socket, ns_address_t *address);

/** Protocol levels used for socket_setsockopt */
#define SOCKET_IPPROTO_IPV6			41

/** Option names for protocol level SOCKET_IPPROTO_IPV6 */
/** Specify traffic class for outgoing packets, as int16_t (RFC 3542 S6.5 says int); valid values 0-255, or -1 for system default */
#define SOCKET_IPV6_TCLASS			1

/**
 * \brief Set an option for a socket
 *
 * Used to specify miscellaneous options for a socket. Supported levels and
 * names defined above.
 *
 * \param socket socket id
 * \param level protocol level
 * \param opt_name option name (interpretation depends on level)
 * \param opt_value pointer to value for the specified option
 * \param opt_len size of the data pointed to by value
 *
 * \return 0 on success
 * \return -1 invalid socket id
 * \return -2 invalid/unsupported option
 * \return -3 invalid option value
 */
extern int8_t socket_setsockopt(int8_t socket, uint8_t level, uint8_t opt_name, const void *opt_value, uint16_t opt_len);
#ifdef __cplusplus
}
#endif
#endif /*_NS_SOCKET_H*/
