/* File: HCSR04.h
 * Author: Robert Abad  Copyright (c) 2013
 *
 * Desc: driver for HCSR04 Ultrasonic Range Finder.  The returned range
 *       is in units of meters.
 *
 *       To use this driver you must call the methods ::startMeas()
 *       and ::getMeas().  The HCSR04 requires a trigger time of
 *       10 usec (microseconds) which is initiated by ::startMeas().
 *       If a successful measurement is made, getMeas() will return
 *       RANGE_MEAS_VALID.  If unsuccessful, initiate a new measurement.
 *
 *       The datasheet for this device can be found here:
 *       http://www.elecfreaks.com/store/download/product/Sensor/HC-SR04/HC-SR04_Ultrasonic_Module_User_Guide.pdf
 *
 *       Below is some sample code:
 *
 *       #include "mbed.h"
 *       #include "HCSR04.h"
 *
 *       #define PIN_TRIGGER    (p14)
 *       #define PIN_ECHO       (p15)
 *
 *       int main(void)
 *       {
 *           HCSR04 rangeFinder( PIN_TRIGGER, PIN_ECHO );
 *           float range;
 *           
 *           while (1)
 *           {
 *               rangeFinder.startMeas();
 *               wait(0.1);
 *               if ( rangeFinder.getMeas(range) == RANGE_MEAS_VALID )
 *               {
 *                   printf("range = %f\n\r", range);
 *               }
 *           }
 *       }
 */

#ifndef __HCSR04_H__
#define __HCSR04_H__

#include "mbed.h"

#define MTRS_TO_INCH   (39.3701)

typedef enum
{
    RANGE_MEAS_INVALID,
    RANGE_MEAS_VALID
} etHCSR04_RANGE_STATUS;



class HCSR04
{
public:
    HCSR04( PinName pinTrigger, PinName pinEcho );
    void startMeas(void);
    etHCSR04_RANGE_STATUS getMeas(float &rRangeMeters);
    
    
private:
    DigitalOut trigger;
    Ticker triggerTicker;
    InterruptIn echo;
    Timer echoTimer;
    unsigned long measTimeStart_us;
    unsigned long measTimeStop_us;
    unsigned long dTime_us;
    etHCSR04_RANGE_STATUS status;
    
    void triggerTicker_cb(void);    // trigger ticker callback function
    void ISR_echoRising(void);      // ISR for rising edge
    void ISR_echoFalling(void);     // ISR for falling edge
};

#endif /* __HCSR04_H__ */
