#ifndef ROM_COMMANDS
#define ROM_COMMANDS

#include "OneWire.h"

class ROMCommands;

#define READ_ROM 0x33
#define SKIP_ROM 0xCC
#define MATCH_ROM 0x55
#define SEARCH_ROM 0xF0

// type used to represent a one-wire rom code
typedef unsigned long long OneWire_ROM;

// types used for the one-wire registers
typedef struct TransportRead{
    unsigned char read; // bits or rom read
    unsigned char crc8; // crc8 check
} TransportRead;

/** Persistant search structure
 *  Used to keep track of search state and results between executions of the search transport layer command
 */
typedef struct TransportSearchPersist{
    /** Rom code result of previous search operation
     *
     *  @note value should be ignored if a one-wire execution error occured
     */
    OneWire_ROM rom;
    
    signed char last; // used internally by the micro instruction system to keep track of state
    
    /** Used to tell when to end a search sequence
     *
     *  TRUE when the search has completed
     *
     *  FALSE when there are still rom codes left to search
     */
    bool done;
    
    /** Clears existing search state and results in preparation for a new search
     *
     *  This will reset search progress to start a search sequence from the beginning
     *
     *  @note Search sequences should be restarted if a one-wire execution error occured as persistant state values might be invalid
     */
    void clear();
} TransportSearchPersist;

typedef struct TransportSearch{
    unsigned char read; // bits or rom read
    unsigned char crc8; // crc8 check
    signed char marker; // last conflict marker
    unsigned char bit; // storage place current bit result
} TransportSearch;

/** One-Wire Transport layer class
 *  Used to set up a one-wire instruction with a transport layer
 */
class OWTransport{
public:
    /** Set a one-wire instruction to perform the read transport layer command
     *
     * @param inst One-wire instruction that will carry out the command
     * @param out One-wire rom that will be used to store the read rom code
     */
    static void read(OneWire_Instruction * inst, OneWire_ROM * out);
    
    /** Set a one-wire instruction to perform the search transport layer command
     *
     * @param inst One-wire instruction that will carry out the command
     * @param search Persistant search structure that will be used for search state and results
     */
    static void search(OneWire_Instruction * inst, TransportSearchPersist * search);
private:
    // private functions which carry out the one-wire logic
    static void _inst_readsetup(OneWire * which);
    static void _inst_read(OneWire * which);
    static void _inst_readhandler(OneWire * which, char bit);
    static void _inst_searchsetup(OneWire * which);
    static void _inst_search(OneWire * which);
    static void _inst_searchhandler(OneWire * which, char bit);
};

#endif