#ifndef ONE_WIRE
#define ONE_WIRE

#include "mbed.h"
#include "rtos.h"

class OneWire;

typedef void (* OneWire_Instruction_Func)(OneWire *);

typedef struct OneWire_InstCall{
    unsigned char code;
    OneWire_Instruction_Func inst;
    void * args;
} OneWire_InstCall;

typedef struct OneWire_Instruction{
    OneWire_InstCall network;
    OneWire_InstCall transport;
} OneWire_Instruction;

typedef void (* OneWire_ReadHandler)(OneWire * which, char bit);

typedef enum OneWire_Error {SUCCESS,
                            NO_PRESENCE,
                             ABORTED,
                             CRC_ERROR} OneWire_Error;

#define ONEWIRE_TIMESLOT 60 // 60 to 120
#define ONEWIRE_RECOVER 1 // 1 to whatever
#define ONEWIRE_PULSEHIGH 30 // 15 to 30
#define ONEWIRE_PULSELOW 120 // 60 to 240
#define ONEWIRE_WRITE1LOW 1 // 1 to 15
#define ONEWIRE_WRITE0LOW 59 // 1 to ONEWIRE_TIMESLOT-1
#define ONEWIRE_READLOW 1 // 1 to 14
#define ONEWIRE_READDURATION 15 // 15
#define ONEWIRE_READRELEASE 15 // 0 to 44

extern const unsigned short DEFAULT_TIMES[];

typedef void(OneWire::*OneWire_MicroInstruction)(void);

// type for detection handler
typedef void (* OneWire_Detection_Handler)(OneWire * which);

class OneWire {
public:
    OneWire(PinName pin);
    int send(OneWire_Instruction *, unsigned char);
    void endInstruction();
    void repeatInstruction();
    
    // stops all communication, resuming blocked thread with error.
    void abort(OneWire_Error err);
    
    // one wire wakeup function
    
    // public fields
    OneWire_Error error;
    char registers[16];
    
    // configuration
    OneWire_Detection_Handler detecthandle;
    const unsigned short * timeing;
    OneWire_ReadHandler readhandle;
    OneWire_InstCall execute;
    
    // high level IO ops
    void op_send1();
    void op_send0();
    void op_read();
    void op_reset();
    
private:
    // system resources
    DigitalInOut _pin;
    InterruptIn _detect;
    Timeout _timer;
    osThreadId _caller;
    
    // instruction system
    OneWire_Instruction * _instruction;
    unsigned char _instn;
    unsigned char _inststate;
    static const OneWire_MicroInstruction _OneWire_MicroProgram[];
    unsigned char _microinstc;
    unsigned char _microinstn;
    
    void _nextmicroinst();
    void _resumeinst();
    void _nextinst();
    
    // reset presence detect handler
    static void _presencedetect(OneWire * which, char bit);
    
    // external interrupt handler
    void _ei_detect();
    
    // low level IO ops
    void _io_high();
    void _io_low();
    void _io_read();
};

#endif