/*
  @file HIH6130.cpp
  
  @brief Humidity and Temperature Sensor HIH6130 Breakout I2C Library      

  @Author spiridion (http://mailme.spiridion.net)

  Tested on LPC1768 and FRDM-KL25Z
  
  Copyright (c) 2014 spiridion
  Released under the MIT License (see http://mbed.org/license/mit)

  Documentation regarding I2C communication with HIH6130 can be found here: 
  http://mbed.org/media/uploads/spiridion/i2c_comms_humidicon_tn_009061-2-en_final_07jun12.pdf
*/
 
#include "HIH6130.h"
#include "mbed.h"

HIH6130::HIH6130(PinName sda, PinName scl, int address) : m_i2c(sda,scl), m_addr(address)
{
    m_temperature = UNSET_HI6130_TEMPERATURE_VALUE;
    m_humidity = UNSET_HI6130_HUMIDITY_VALUE;
}

HIH6130::HIH6130(I2C& i2c, int address) : m_i2c(i2c), m_addr(address)
{
    m_temperature = UNSET_HI6130_TEMPERATURE_VALUE;
    m_humidity = UNSET_HI6130_HUMIDITY_VALUE;
}

int HIH6130::ReadData(float* pTemperature, float* pHumidity)
{
    int rsl = Measurement();
    
    if (rsl)
    { 
        m_temperature = TrueTemperature();
        m_humidity = TrueHumidity();
    }
    else
    {
        m_temperature = UNSET_HI6130_TEMPERATURE_VALUE;
        m_humidity = UNSET_HI6130_HUMIDITY_VALUE;
    }
    
    if (pTemperature)
        *pTemperature = m_temperature;
    if (pHumidity)
        *pHumidity = m_humidity;

    return rsl;
}

float HIH6130::TrueTemperature()
{   
    // T = T_output / (2^14-2)*165-40
    return ( ( ((unsigned int)m_data[2] << 8) | (unsigned int)m_data[3] ) >> 2 ) * 0.010072F - 40;  
}

float HIH6130::TrueHumidity()
{
    // H = H_output /(2^14-2)*100
    return ( (((unsigned int)m_data[0] & 0x3f) << 8) | ((unsigned int)m_data[1] & 0xff) ) * 0.006104F;
}

int HIH6130::Measurement()
{
    int errors;

    // Humidity and temperature measurement request
    errors = m_i2c.write(m_addr, m_data, 1);
    
    wait_ms(10);
    
    // Humidity and temperature data fetch
    errors += m_i2c.read(m_addr, m_data, 4);

    //  Check data validity
    if ( errors || !(m_data[0] & 0xC0)) 
        return 0;
        
    return 1;
}
