#include "WeatherStation.h"

// initialization of static data members
    const int WeatherStation::m_measurementInterval = 9000;
    const string WeatherStation::m_stationName = "GuiSpa";

// constructor
WeatherStation::WeatherStation(IDevKit& iDevKit, Logger& logger, BLE& ble) :
  GAPPeripheral(ble, iDevKit, m_stationName, GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED, m_eventQueue, logger),
  m_iDevKit(iDevKit),
  m_logger(logger),
  m_lps25hb(m_iDevKit.getSDA(), m_iDevKit.getSCL(), m_logger),
  m_hdc1000(m_iDevKit.getSDA(), m_iDevKit.getSCL(), m_iDevKit.getDataRdy(), m_logger),
  m_history(logger),
  m_batteryService(ble, m_logger),
  m_environmentalService(ble, m_logger) {
    //  Set the right timing at the Station's start
    time(NULL);
}

void WeatherStation::start() {
  m_logger.log("WeatherStation is starting\r\n");
    
  // make sure that the LPW25HB device is present
  // if not log an error and return
  if (m_lps25hb.probe()) {
      m_logger.log("LPS25HB device found\r\n");    
  } else {
      m_logger.log("Device LPS25HB not found!\r\n");
      return;
  }
    
  // make sure that the HDC1000 device is present
  // if not log an error and return
  if (m_hdc1000.probe()) {
      m_logger.log("HDC1000 device found\r\n");   
  } else {
      m_logger.log("Device HDC1000 not found!\r\n");
      return;
  }  
  
  //Cleaning the console by the addition of a blank line
  m_logger.log("\r\n");

  // schedule measurements every m_measurementInterval milliseconds
  // apply the same syntax as in the previous call
  // 1. the measurement interval is m_measurementInterval
  // 2. the object is the WeatherStation itself (recall how you refer to the current object in Java - the same applies in c++)
  // 3. the method to call is performMeasurements()
  // m_eventQueue.call_every(TO COMPLETED);
  m_eventQueue.call_every(m_measurementInterval, this, &WeatherStation::performMeasurements);
  
  // Starting the BLE peripheral with Advertising State
  GAPPeripheral::advertise();
}


void WeatherStation::onInitComplete(BLE::InitializationCompleteCallbackContext *event) {
  GAPPeripheral::onInitComplete(event);
  m_batteryService.addServiceToGattServer();
  m_environmentalService.addServiceToGattServer();
  performMeasurements();
}

 
void WeatherStation::performMeasurements(void) {
  m_logger.log("Performing measurements:\r\n");
  
  // get and log pressure
  double pressure = m_lps25hb.getPressure();
  m_logger.log("Pressure:      %.02f hPa\r\n", pressure);
  
  // get and log temperature
  double temp = m_hdc1000.getTemperature();
  m_logger.log("Temperature:    %.02f C\r\n", temp);
  
  // get and log humidity
  double humidity = m_hdc1000.getHumidity();
  m_logger.log("Humidiy:        %.02f %%\r\n", humidity);
  
  // get and log the battery level
  uint8_t battery = m_iDevKit.getBatteryLevel();
  m_logger.log("Battery:        %u %%\r\n", battery);
  
  //  Cleaning the console by the addition of a blank line
  m_logger.log("\r\n");
  
  //  get and encode the system's elapsed time
  time_t seconds = time(NULL);
  float time = (uint32_t)seconds;
  
  //  Saving the measured values
  m_history.addMeasurement(((uint32_t)(pressure*10)), ((int16_t)(temp*100)), ((uint16_t)(humidity*100)), ((uint32_t)time));
  
  //  Send the measured values to GATT's services
  m_environmentalService.updatePressureLevel    ((uint32_t)(pressure*10));
  m_environmentalService.updateTemperatureLevel ((int16_t)(temp*100));
  m_environmentalService.updateHumidityLevel    ((uint16_t)(humidity*100));
  //  get pressure trend for the environmental service
  uint8_t pressureTrend = m_history.getPressureTrend((uint32_t)(pressure*10));
  m_logger.log("Trend: %d\r\n", pressureTrend);
  m_environmentalService.updatePressureTrend    (pressureTrend);
  m_batteryService.updateBatteryLevel(battery);
}