#ifndef SERIALIZER_H
#define SERIALIZER_H


#define     BAUD                    19200
#define     MAX_SPEED               127

#define     WAIT                    .001
#define     PI                      3.1415926
#define     PIVET_ADJUSTMENT        7.4
#define     PIVET_SPEED             20

#define     PULSES_PER_REVOLUTION   624
#define     WHEEL_DIAMETER          2.25           //inches
#define     WHEEL_CIRCUMFERENCE     PI*WHEEL_DIAMETER
#define     PULSE_DISTANCE          WHEEL_CIRCUMFERENCE / 624
#define     PULSES_PER_INCH         int(PULSES_PER_REVOLUTION / WHEEL_CIRCUMFERENCE)


#include "mbed.h"

/**
 * Serializer Class to communicate with Robotics
 * Connection Serializer(tm) board
 */
class Serializer {

public:
    /**
     * Constructor.
     *
     */
    Serializer();

    /**
     * Destructor.
     *
     */
    ~Serializer();

    /**
     * Clears left motor encoder count
     *
     */
    void    ClearCountLeft();

    /**
     * Clears right motor encoder count
     *
     */
    void    ClearCountRight();

    /**
     * Clears motors encoder counts
     *
     */
    void    ClearCount();

    /**
     * Sets left motor speed
     * @param inPsec Motor Speed in inches per second
     */
    void    SetSpeedLeft(int inPsec);

    /**
     * Sets right motor speed
     * @param inPsec Motor Speed in inches per second
     */
    void    SetSpeedRight(int inPsec);

    /**
     * Sets speed for both motors
     * @param inPsec Motor Speed in inches per second
     */
    void    SetSpeed(int inPsec);

    /**
     * Sets VPID
     * @param p
     * @param i
     * @param d
     * @param l
     */
    void    SetVPID(int,int,int,int);

    /**
     * Sets DPID
     * @param p
     * @param i
     * @param d
     * @param a
     */
    void    SetDPID(int,int,int,int);

    /**
     * Sets left motor distance and speed
     * @param inches Distance in ticks
     * @param inPsec Motor Speed
     */
    void    DiGoLeft(int inches,int inPsec);

    /**
     * Sets right motor distance and speed
     * @param inches Distance in ticks
     * @param inPsec Motor Speed
     */
    void    DiGoRight(int inches,int inPsec);

    /**
     * Sets both motors distance and speed
     * @param inches Distance in ticks
     * @param inPsec Motor Speed
     */
    void    DiGo(int inches,int inPsec);

    void    SetLeftPWM(int pwm);
    void    SetRightPWM(int pwm);
    void    SetPWM(int pwm);
    void    SetPWM(int lPwm, int rPwm);
    int     IsBusy();

    /**
     * Stops both motors
     */
    void    Stop();

    void    TurnLeft(int deg);
    void    TurnRight(int deg);
    void    PivetLeft(int deg);
    void    PivetRight(int deg);

    int     GetCountLeft();
    int     GetCountRight();
    int     GetCount();
    float   GetDistanceLeft();
    float   GetDistanceRight();
    float   GetDistance();





//protected:
    int     GetReply();
    void    InterruptHandler();
    int     Initialize();
//private:
    Serial          *serial;
    volatile int    leftSpeed, rightSpeed;
    volatile int    _lPWM, _rPWM;
    volatile char   _isBusy;
    volatile char   _irqIsBusy;
};
#endif