//******************************************************************************
//   filter.c  
//
//   Description: filter module
//
//   Author: Jason shen
//   Version    1.00
//   DDC BJ
//   October 2011
//   Built with IAR Embedded Workbench Version: 5.30A
//******************************************************************************
//Change Log:
//******************************************************************************
//Version:  1.00
//Comments: Initial Release Version
//******************************************************************************
#include "filter.h"

/**************************************************************************************************
 * @fn          FirstOrderFilter
 *
 * @brief       prototype -> result = OldData * ( 256 - K ) / 256 + NewData * K / 256.
 *              optimized -> 
 *                result = OldData - ( OldData - NewData ) * K / 256 when ( NewData < OldData )
 *                result = OldData + ( NewData - OldData ) * K / 256 when ( NewData > OldData )
 *
 * @param       new_data = new sampled data
 *              old_data = last time filter result
 *              K        = filter coefficient (0~255), the weighted of the NewData in the result
 *
 * @return      The filter result for this time 
 **************************************************************************************************
 */
 /*
int32_t Filter::first_order_filter(int32_t old_data, int32_t new_data, int32_t K, int32_t threshold)
{
  int32_t result;
      
  if(((new_data - old_data) > threshold) || ((old_data - new_data) > threshold))
  {
    result = new_data;
  }
  else
  {
    if(new_data < old_data)
    {
      result = old_data - (( old_data - new_data ) << K + 128) >> 8;
    }
    else if((new_data > old_data))
    {
      result = old_data + (( new_data - old_data ) << K + 128) >> 8;
    }
    else
    {
      result = old_data;
    }
  }
  return result;
}
*/
float Filter::first_order_filter(float old_data, float new_data, float K, float threshold)
{
  float result;
      
  if(((new_data - old_data) > threshold) || ((old_data - new_data) > threshold))
  {
    result = new_data;
  }
  else
  {
    result = old_data + (new_data - old_data) * K; //it equals to result = old_data * (1.0 - K) + new_data* K;
  }
  return result;
}

/**************************************************************************************************
 * @fn          FirstOrderFilterAdaptiveK
 *
 * @brief       prototype -> result = OldData * ( 256 - K ) / 256 + NewData * K / 256.
 *              optimized -> 
 *                result = OldData - ( OldData - NewData ) * K / 256 when ( NewData < OldData )
 *                result = OldData + ( NewData - OldData ) * K / 256 when ( NewData > OldData )
 *
 * @param       new_data   = new sampled data
 *              old_data   = last time filter result
 *              K          = filter coefficient (0~255), the weighted of the NewData in the result
 *              threshold  = 
 *              K_increment= increment for K every time
 *              K_max      = max of K
 *
 * @return      The filter result for this time 
 **************************************************************************************************
 */
int32_t Filter::first_order_filter_adaptive_K(int32_t old_data, int32_t new_data, int32_t threshold, int32_t K_increment, int32_t K_max)
{
  static char data_change_direction_flag = 0;  // 0 is the positive direction(new<old), 1 is the negative direction(new>old)
  static char filter_cnt = 0;                  // filter times counter
  static int16_t  K = 0;
  
  static int32_t _v[3];
  
  _v[0] = _v[1];
  _v[1] = _v[2];
  _v[2] = new_data;
  

  if(((_v[1] >= _v[0]) && (_v[2] >= _v[1])) || ((_v[1] <= _v[0]) && (_v[2] <= _v[1])))
  {
    data_change_direction_flag = 1;
  }
  else data_change_direction_flag = 0;
  
  if(data_change_direction_flag == 0)
  {
    filter_cnt = 0;
    K = 1;
  }
  else
  {
    filter_cnt++;
    if(((_v[2] - _v[1]) > threshold) || ((_v[1] - _v[2]) > threshold))
    {
      filter_cnt += 2;
    }
    
    if(filter_cnt >= 4)
    {
      K += K_increment;
      if(K > K_max)
      {
        K = K_max;
      }
      //filter_cnt = 0;
    }
  }
  //FirstOrderFilter
  _v[2] = first_order_filter(_v[1], _v[2], K, threshold);
  
  return _v[2];
}

#define FILTER_CNT_MAX  12
float Filter::first_order_filter_adaptive_K(float new_data, filter_args_s * arg)
{
    arg->v[0] = arg->v[1];
    arg->v[1] = arg->v[2];
    arg->v[2] = new_data;
  

  
  if(((arg->v[2] - arg->v[1]) > arg->threshold) || ((arg->v[1] - arg->v[2]) > arg->threshold))
  {
      arg->v[0] = new_data;
      arg->v[1] = new_data;
      arg->v[2] = new_data;
      return arg->v[2];
  }
 

  if(((arg->v[2] >= arg->v[1]) && (arg->v[1] >= arg->v[0])) || ((arg->v[2] <= arg->v[1]) && (arg->v[1] <= arg->v[0])))
  {
    arg->dir_flag = 1;
  }
  else arg->dir_flag = 0;
  
  if(arg->dir_flag == 1)
  {
    arg->dir_cnt ++;
    if(((arg->v[2] - arg->v[1])*2.0 > arg->threshold) || ((arg->v[1] - arg->v[2])*2.0 > arg->threshold))
    {
      arg->dir_cnt += 5;
    }
    else if(((arg->v[2] - arg->v[1])*4.0 > arg->threshold) || ((arg->v[1] - arg->v[2])*4.0 > arg->threshold))
    {
      arg->dir_cnt += 4;
    }
    else if(((arg->v[2] - arg->v[1])*8.0 > arg->threshold) || ((arg->v[1] - arg->v[2])*8.0 > arg->threshold))
    {
      arg->dir_cnt += 2;
    }
    if(arg->dir_cnt >= FILTER_CNT_MAX)
    {
        arg->dir_cnt = 0;
        arg->K_curr += arg->K_inc;
    }
    if(arg->K_curr >= arg->K_max)
    {
        arg->K_curr = arg->K_max;
    }    
  }
  else
  {
    arg->dir_cnt = 0;
    arg->K_curr = arg->K_ini; 
  }
  
  arg->v[2] = (arg->v[1] + (arg->v[2] - arg->v[1]) * arg->K_curr);

  return arg->v[2];
}

/**************************************************************************************************
 * @fn          SignalFilter
 *              Library to Filter Sensor Data using digital filters
 *              Available filters: Chebyshev & Bessel low pass filter (1st & 2nd order)
 *
 * @brief       
 *              
 *               
 *                
 *
 * @param       data       = new sampled data
 *              _filter    = Select filter: 'c' -> Chebyshev, 'b' -> Bessel
 *              _order     = Select filter order (1 or 2)
 *
 * @return      The filtered result for this time 
 **************************************************************************************************
 */
int16_t Filter::SignalFilter(int16_t data, int16_t _filter, int16_t _order)
{
        static short _v[3];
    //  Uncomment for debugging
    //  Serial.println(_filter);
    //  Serial.println(_order); 
    if(_filter=='c')  // Chebyshev filters
    {
        if(_order==1)  //ripple -3dB
        {
            _v[0] = _v[1];
            int32_t tmp = ((((data * 3269048L) >>  2)  //= (3.897009118e-1 * data)
                + ((_v[0] * 3701023L) >> 3) //+(  0.2205981765*v[0])
                )+1048576) >> 21; // round and downshift fixed point /2097152
            _v[1]= (short)tmp;
            return (short)(_v[0] + _v[1]); // 2^
        }
        if(_order==2)  //ripple -1dB
        {
            _v[0] = _v[1];
            _v[1] = _v[2];
            int32_t tmp = ((((data * 662828L) >>  4)   //= (    7.901529699e-2 * x)
                + ((_v[0] * -540791L) >> 1) //+( -0.5157387562*v[0])
                + (_v[1] * 628977L) //+(  1.1996775682*v[1])
                )+262144) >> 19; // round and downshift fixed point /524288

            _v[2]= (short)tmp;
            return (short)((
                 (_v[0] + _v[2])
                +2 * _v[1])); // 2^
        }
    }
    if(_filter=='b')  // Bessel filters
        if(_order==1)  //Alpha Low 0.1
        {
            _v[0] = _v[1];
            int32_t tmp = ((((data * 2057199L) >>  3)  //= (    2.452372753e-1 * x)
                + ((_v[0] * 1068552L) >> 1) //+(  0.5095254495*v[0])
                )+524288) >> 20; // round and downshift fixed point /1048576
            _v[1]= (short)tmp;
            return (short)(((_v[0] + _v[1]))); // 2^
        }
        if(_order==2)  //Alpha Low 0.1
        {
            _v[0] = _v[1];
            _v[1] = _v[2];
            int32_t tmp = ((((data * 759505L) >>  4)   //= (    9.053999670e-2 * x)
                + ((_v[0] * -1011418L) >> 3)    //+( -0.2411407388*v[0])
                + ((_v[1] * 921678L) >> 1)  //+(  0.8789807520*v[1])
                )+262144) >> 19; // round and downshift fixed point /524288

            _v[2]= (short)tmp;
            return (short)(((_v[0] + _v[2])+2 * _v[1])); // 2^
        }
        return 0;
}

void Filter::max_min(uint16_t data[], uint16_t n, uint16_t *max, uint16_t *min)
{
  *max=0x0000;
  *min=0xFFFF;

  for(int16_t i=0;i<n;i++)
  {
    if(data[i] > *max) *max = data[i];
    if(data[i] < *min) *min = data[i];
  }
}

void Filter::sum_avg(uint16_t data[], uint16_t n, uint32_t *sum, float *avg)
{
  *sum = 0;
  for(int16_t i=0;i<n;i++)
  {
    *sum += (uint32_t)data[i];
  }
  *avg = (float)*sum/(float)n;
}

void Filter::stdev(uint16_t data[], uint16_t n, float *s)
{
    int16_t i;
  uint32_t sum1;
  float avg, sum2;
  sum1 = 0;
  for(i=0;i<n;i++)
  {
    sum1 += (uint32_t)data[i];
  }
  avg = (float)sum1/(float)n;
  sum2 = 0;
  for(i=0;i<n;i++)
  {
    sum2 += ((float)data[i] - avg)*((float)data[i] - avg);
  }
  sum2 = sum2/(float)(n-1);
  *s = sqrt(sum2);
  
}
