
#include "calculate.h"


const float lm57_mv_table[71] = {
    1152, 1147, 1142, 1137, 1132, 1127, 1122, 1117, 1112, 1106,
    1101, 1096, 1091, 1086, 1081, 1076, 1071, 1066, 1061, 1055,
    1050, 1045, 1040, 1035, 1030, 1025, 1020, 1015, 1009, 1004,
    999, 994, 989, 984, 979, 973, 968, 963, 958, 953,
    948, 942, 937, 932, 927, 922, 917, 911, 906, 901,
    896, 891, 885, 880, 875, 870, 865, 859, 854, 849,
    844, 839, 833, 828, 823, 818, 812, 807, 802, 797, 791
};

/* linterp -----------------------------------------------------------

Performs linear interpolation on a data array. The argument X array must be ascending.
If supplied X value is outside the X data array, the function performs extrapolation
by two last data points.

Input:  val             waiting to find
        size >= 2       size of X & Y data arrays
        _x[size]        >>ascending<< X data array
        _y[size]        Y data array

Return: Y value

*/
float Calculate::linterp(const float* _x, const float* _y, float val, unsigned char size)
{
    // take care the value is within range
    if (size < 2) return _y[0];    // on illegal call, return first element in Y array
    // val = constrain(val, _in[0], _in[size-1]);
    if (val <= _x[0]) return _y[0];
    if (val >= _x[size-1]) return _y[size-1];

    // search right interval
    unsigned char pos = 1;  // _x[0] allready tested
    while(val > _x[pos]) pos++;

    // this will handle all exact "points" in the _in array
    if (val == _x[pos]) return _y[pos];

    // interpolate in the right segment for the rest
    return ((val - _x[pos-1]) * (_y[pos] - _y[pos-1]) / (_x[pos] - _x[pos-1]) + _y[pos-1]);
}

//=====================================================================================================================
// Calculate the value of IR target temperature. Temperature unit is 0.1*S
//
void Calculate::calcu_Tir_oS(float * targTempS, float PSdOut, float ambTempC, const struct CalibrationConstants CalConst, float emissivity)
{
    const float         Ofs     = CalConst.Ofs;       // PSdOfs for                   gain 1
    const float * const En      = CalConst.En;       // Table of energy of target,   gain 1
    const float * const ToS     = CalConst.ToS;      //Table of temperature
    const float * const TempCo  = CalConst.TempCo; // Det tempco (alpha)
    const float         Td0     = CalConst.Td0;               // Cal det temp


    float Td;   // Temperature and energy of the detector itself
    float T;    // Target temperature           all gains

    float TempCo_C;
    
    Td = ambTempC;
    if(Td > Td0)
        TempCo_C = (1 + TempCo[0] * (Td - Td0)) * (1 + TempCo[1] * (Td - Td0));
    else
        TempCo_C = (1 + TempCo[0] * (Td0 - Td)) * (1 + TempCo[1] * (Td - Td0));

    PSdOut = (PSdOut * TempCo_C - Ofs) / emissivity;
    
    T = linterp(En,ToS,PSdOut,CALI_CONST_IR_TAB_NUM); // PSd1(T1) to T1, temperature of the target, gain 1, in 0.1*S
    
    *targTempS = T / 10.0;                             // Return temperature
}

float Calculate::get_Tamb_oC(float N_Tamb)
{
    float lm57_oC[71];
    float mV[71];
    float T_mV = -N_Tamb * 1200 / 16777215;
    for (int i = 0; i < 71; i++) {
        lm57_oC[i] = -10 + i;
        mV[i] = -lm57_mv_table[i];
    }
    return linterp(mV, lm57_oC, T_mV, 71); ;
}
