// ---------------------------------------------
// MultiTech Thermostat & Fan Demo (Sensor unit)
//
// This program runs on a MultiTech mDot with:
//   - 1 MultiTech UDK
//   - 1 Arduino base shield
//   - 2 Grove COM08211P push buttons
//   - 1 Grove SEN23292P v1.2 temperature sensor
// ---------------------------------------------

#include "mbed.h"
#include "mDot.h"
#include "PinDetect.h"

mDot* dot;
//Ticker ledTick;
//Ticker temperatureMonitorTick;
Ticker periodicSendTick;

//PinDetect up_button(PB_1);        //A0 on Arduino base shield
//PinDetect down_button(PB_0);      //A1 on Arduino base shield
AnalogIn temperatureSensor(PC_1); //A2 on Arduino base shield
//DigitalOut transmitLED(LED1);     
//DigitalOut buttonPressLED(PA_1);  //D6 on Arduino base shield

// Configuration variables
static std::string config_network_name = "watchlink";
static std::string config_network_pass = "watchlink";
static uint8_t config_frequency_sub_band = 1;

//Global Variables
//static uint16_t setPoint = 21;       //21 C is standard room temp
static volatile bool timeToReadTemperature = true;
static volatile bool dataChanged = true;
//static volatile bool thermostatActivated = false;
//static uint16_t sentSetPoint;

//Function prototypes
//void ledTock();
void periodicSendTock();
//void temperatureMonitorTock();
//void up_button_callback();
//void down_button_callback();
void printError(mDot* dot, int32_t returnCode);
void printVersion();
bool setFrequencySubBand(uint8_t subBand);
bool setNetworkName(const std::string name);
bool setNetworkPassphrase(const std::string passphrase);
bool setPower(uint8_t power);
bool setAck(uint8_t retries);
bool joinNetwork();
bool send(const std::string text);


int main()
{
    //Start LED startup sequence
    //ledTick.attach(&ledTock, 0.1);

    printf("\r\n\r\n");
    printf("=====================================\r\n");
    printf("MTS Thermostat/Fan Demo (Sensor unit)\r\n");
    printf("=====================================\r\n");
    printVersion();

    // get the mDot handle
    dot = mDot::getInstance();

    // reset to default config so we know what state we're in
    dot->resetNetworkSession();
    dot->resetConfig();

    // set up the mDot with our network information
    setNetworkName(config_network_name);
    setNetworkPassphrase(config_network_pass);
    setFrequencySubBand(config_frequency_sub_band);
    setPower(7);    // Reduce latency for 868 units
    setAck(0);      // Disable ack for less latency

    while (!joinNetwork()) { wait(2); dot->resetNetworkSession(); }

    //Stop LED startup sequence & configure them for operation
    //ledTick.detach();
    //transmitLED = 1;
    //buttonPressLED = 1;

    // Configure timers
    periodicSendTick.attach(&periodicSendTock, 1*60);
    //temperatureMonitorTick.attach(&temperatureMonitorTock, 0.15);

    // Setup Interrupt callback function
    //up_button.attach_deasserted(&up_button_callback);
    //down_button.attach_deasserted(&down_button_callback);

    // Start sampling buttons using interrupts
    //up_button.setSampleFrequency();
    //down_button.setSampleFrequency();

    uint16_t temperature = 0;
    //while (1) {
        //if (timeToReadTemperature) {
            uint16_t B = 4100;            //B value of the thermistor
            uint16_t analogReading;
            float resistance;

            analogReading = temperatureSensor.read_u16();

            // get the resistance of the sensor
            resistance=(65534-analogReading)*10000/analogReading;
            // convert to temperature via datasheet
            temperature=1/(log(resistance/10000)/B+1/298.15)-273.15;
            //printf("read: %d, res: %d, temp: %d,set: %d\r\n", analogReading, (uint16_t)resistance, temperature, setPoint);

            //if (!thermostatActivated && temperature > setPoint) {
                //thermostatActivated = true;
                //dataChanged = true;
            //}
            //else if (thermostatActivated && temperature < setPoint) {
                //thermostatActivated = false;
                //dataChanged = true;
            //}

            //timeToReadTemperature = false;
        //}
        
        dataChanged = true;

        
        if (dataChanged) {
            
            char latestData[100];
            //transmitLED = 1;
           // sentSetPoint = setPoint;
            sprintf(latestData, "temp: %d", temperature);
            printf("%s\r\n", latestData);

            if (send(latestData)) {
                dataChanged = false;
            }
            //transmitLED = 0;
        }
    //}
 //return true;   
}

void printVersion()
{
    printf("%s\r\n\r\n", dot->getId().c_str());
}

void periodicSendTock() {
    dataChanged = true;
}

bool setFrequencySubBand(uint8_t subBand)
{
    int32_t returnCode;
    printf("Setting frequency sub band to '%d'...\r\n", subBand);
    if ((returnCode = dot->setFrequencySubBand(subBand)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkName(const std::string name)
{
    int32_t returnCode;
    printf("Setting network name to '%s'...\r\n", name.c_str());
    if ((returnCode = dot->setNetworkName(name)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setNetworkPassphrase(const std::string passphrase)
{
    int32_t returnCode;
    printf("Setting passphrase to '%s'...\r\n", passphrase.c_str());
    if ((returnCode = dot->setNetworkPassphrase(passphrase)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool setPower(uint8_t power)
{
    int32_t returnCode;
    printf("Setting tx power to '%d'...\r\n", power);
    if ((returnCode = dot->setTxPower(power)) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    return true;
}


bool joinNetwork()
{
    int32_t returnCode;
    printf("\r\nJoining network...\r\n");
    if ((returnCode = dot->joinNetworkOnce()) != mDot::MDOT_OK) {
        printError(dot, returnCode);
        return false;
    }
    printf("Network Joined!\r\n");
    return true;
}

bool setAck(uint8_t retries)
{
    int32_t returnCode;
    printf("Setting ack to '%d'...\r\n", retries);
    if ((returnCode = dot->setAck(retries)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    return true;
}

bool send(const std::string text)
{
    int32_t returnCode;
    uint32_t timeTillSend = dot->getNextTxMs();
    if (timeTillSend != 0) {
        printf("waiting %lu ms to send\r\n", timeTillSend);
        return false;
    }

    printf("Sending data...  ");
    std::vector<uint8_t> data(text.begin(), text.end());
    if ((returnCode = dot->send(data, 1)) != mDot::MDOT_OK)
    {
        printError(dot, returnCode);
        return false;
    }
    printf("Data sent!\r\n");
    return true;
}

void printError(mDot* dot, int32_t returnCode)
{
    std::string error = mDot::getReturnCodeString(returnCode) + " - " + dot->getLastError();
    printf("%s\r\n", error.c_str());
 }

