/*******************************************************
 * This main program simulates a multi-game multiplayer
 * arcade gaming system. The program allows the player
 * to choose from four different games:
 *
 *  1) Simon
 *  2) Super Tic-Tac-Toe (first player, the red X)
 *  3) Pac-Man
 *  4) Rock, Paper, Scissors, Lizard, Spock
 * 
 * All games except Pac-Man communicate with another
 * gaming system via an XBee module to simulate
 * multiplayer. Pac-Man is multiplayer as well, but the
 * characters in the game are controlled using the three
 * tactile switches on one board.
 *******************************************************/


#include "mbed.h"
#include "PinDetect.h"
#include "SDFileSystem.h"
#include "uLCD_4DGL.h"
#include "wave_player.h"
#include <mpr121.h>
#include "RGBLED.h"
#include "RedX.h"
#include "Simon.h"
#include "Stage.h"
#include "Pacman.h"
#include "Ghost.h"
#include "RPSLK.h"


// Speaker
AnalogOut DACout(p18);
wave_player waver(&DACout);

// XBee module
DigitalOut reset(p12);
Serial XBee(p13, p14);

// SD card
SDFileSystem sd(p5, p6, p7, p8, "sd");

// uLCD display
uLCD_4DGL uLCD(p28, p27, p30);


// Select game to play
int selectGame();


int main() {
    int gameNumber = selectGame();

    // Play Simon Says
    if (gameNumber == 0) {
        // Analog noise
        AnalogIn noise(p15);

        // Pushbutton interrupts
        PinDetect button1(p17);
        PinDetect button2(p19);
        PinDetect button3(p20);
        PinDetect button4(p16);

        Simon simon = Simon(noise, reset, button1, button2, button3, button4, sd, XBee, uLCD, waver);
        simon.playSimon();
    // Play Super Tic-Tac-Toe
    } else if (gameNumber == 1) {
        // Touch keypad
        InterruptIn input(p21);
        I2C i2c(p9, p10);
        Mpr121 MPR121(&i2c, Mpr121::ADD_VSS);

        // RGB LED
        RGBLED RGB(p25, p23, p22);

        RedX player = RedX(reset, input, MPR121, RGB, sd, XBee, uLCD, waver);
        player.playSuperTicTacToe();
    // Play Pac-Man
    } else if (gameNumber == 2) {
        // Pac-Man controller
        PinDetect PacmanRight(p9);
        PinDetect PacmanDown(p10);
        PinDetect PacmanLeft(p11);
        PinDetect PacmanUp(p29);
        
        // Red ghost controller
        PinDetect redGhostRight(p26);
        PinDetect redGhostDown(p15);
        PinDetect redGhostLeft(p16);
        PinDetect redGhostUp(p19);
        
        // Yellow ghost controller
        PinDetect yellowGhostRight(p25);
        PinDetect yellowGhostDown(p24);
        PinDetect yellowGhostLeft(p23);
        PinDetect yellowGhostUp(p21);

        // Maximize the baud rate
        uLCD.baudrate(MAXBAUDRATE);
        
        // Set up the stage
        Stage stage(uLCD);
        stage.initialize();
    
        // Set up Pac-Man
        Pacman pacman(PacmanRight, PacmanDown, PacmanLeft, PacmanUp, stage, uLCD, waver);
        pacman.initialize();
    
        // Set up the red ghost
        Ghost redGhost(RED, pacman, redGhostRight, redGhostDown, redGhostLeft, redGhostUp, stage, uLCD);
        redGhost.initialize(60, 60, FACELEFT);

        // Set up the yellow ghost
        Ghost yellowGhost(YELLOW, pacman, yellowGhostRight, yellowGhostDown, yellowGhostLeft, yellowGhostUp, stage, uLCD);
        yellowGhost.initialize(68, 60, FACERIGHT);

        pacman.playSound(THEME);
    
        // Checks to see whether the game is over
        bool gameOver;
    
        // Loop through the game
        while (1) {
            pacman.displayStatus();
            gameOver = pacman.move();
    
            // If all pac dots are eaten or Pac-Man runs out of lives
            if (gameOver == true) {
                // Break out of the loop
                break;
            }

            redGhost.move();

            yellowGhost.move();
        }
    
        // Game over display
        pacman.gameOver();
    } else {
        // Tactile switch interrupts
        PinDetect rock(p21);
        PinDetect paper(p22);
        PinDetect scissors(p23);
        PinDetect lizard(p24);
        PinDetect spock(p25);

        RPSLK rpslk = RPSLK(reset, rock, paper, scissors, lizard, spock, sd, XBee, uLCD, waver);
        rpslk.playRPSLK();
    }
}


// Select game to play
int selectGame() {
    // Set up XBee
    reset = 0;
    wait(0.001);
    reset = 1;
    wait(0.001);

    selection:
    DigitalIn button1(p17);
    DigitalIn button2(p19);
    DigitalIn button3(p20);
    DigitalIn button4(p16);

    button1.mode(PullUp);
    button2.mode(PullUp);
    button3.mode(PullUp);
    button4.mode(PullUp);
    wait(0.01);

    uLCD.baudrate(MAXBAUDRATE);

    uLCD.text_width(1);
    uLCD.text_height(1);
    uLCD.textbackground_color(BLACK);
    uLCD.filled_rectangle(0, 0, 127, 127, BLACK);
    uLCD.background_color(BLACK);
    uLCD.color(GREEN);
    uLCD.locate(0, 0);

    uLCD.printf("Select game:\n\n");
    uLCD.printf("pb1: Simon Says\n");
    uLCD.printf("pb2: Super Tic-\n     Tac-Toe\n");
    uLCD.printf("pb3: Pac-Man\n");
    uLCD.printf("pb4: Rock, Paper,\n     Scissors,\n     Lizard, Spock");
    wait(0.1);

    int gameNumber = -1;

    while (1) {
        if (button1 != 1) {
            gameNumber = 0;
            break;
        } else if (button2 != 1) {
            gameNumber = 1;
            break;
        } else if (button3 != 1) {
            gameNumber = 2;
            break;
        } else if (button4 != 1) {
            gameNumber = 3;
            break;
        }
    }

    uLCD.cls();
    XBee.putc(gameNumber);
    int otherGame = -1;
    bool print = true;

    if ((gameNumber == 0) || (gameNumber == 1) || (gameNumber == 3)) {
        while (otherGame == -1) {
            if (XBee.readable() == true) {
                otherGame = XBee.getc();
                print = false;
            }

            if (print == true) {
                uLCD.printf("Waiting for other player...");
                print = false;
            }
        }

        if (otherGame != gameNumber) {
            uLCD.cls();
            uLCD.printf("Sorry, but the    ");
            uLCD.printf("other player chose");
            uLCD.printf("a different game. ");
            uLCD.printf("Returning to game ");
            uLCD.printf("selection menu.");
            wait(2);
            uLCD.cls();
            goto selection;
        }
    }

    uLCD.cls();
    return gameNumber;
}