/*
* FiniteStateMachine. Table driven Finite State Machine library 
* based on theHarel state machine, supporting actions on transitions, state
* entry and state exit.
*
* Copyright (C) <2009> Petras Saduikis <petras@petras.co.uk>
*
* This file is part of FiniteStateMachine.
*
* FiniteStateMachine is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
* 
* FiniteStateMachine is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with DebugTrace.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef SNATCH59_FSM_H
#define SNATCH59_FSM_H

#include <mbed.h>
#include "FSMDefs.h"

STATE_TEMPLATE_ class State;

TRANS_DEF_TEMPLATE_ 
struct TransitionDefinition
{
    const char* ownerState;
    int    eventId;
    StateBehaviour mode;
    ActionPtrType action;
    const char* newState;
};

TRANS_DEF_TEMPLATE_
class StateDefinition
{
public:
    const char* stateName;
    ActionPtrType stateEntryAction;
    ActionPtrType stateExitAction;
};

#include "State.h"

FSM_TEMPLATE_ 
class FiniteStateMachine
{
public:
    FiniteStateMachine();
    ~FiniteStateMachine();
    
    void initialize(StateObjectType* owner, 
                    const STATE_DEFINITION_* states, const int total_states, 
                    const TRANSITION_DEFINITION_* transitions, const int total_transitions,
                    const char* this_state);
    void traverse(const int new_event);
    void forceToState(const char* stat_name);
    bool isState(const char* this_state) const;
    const char* getCurrentStateName() const;

private:
    void addStates(const STATE_DEFINITION_* states, const int total_states, 
                          const TRANSITION_DEFINITION_* &transitions, const int total_transitions);
    void executeTraversal(const int event);

    // Data Members
    STATE_* currentState;          
    int totalStates;            
    StateObjectType* itsStateObject;      
    STATE_* itsStates[maxStates];   
};

FSM_TEMPLATE_
inline const char* FINITE_STATE_MACHINE_::getCurrentStateName() const
{
    return (currentState->getName());
}

FSM_TEMPLATE_
inline bool FINITE_STATE_MACHINE_::isState(const char* this_state) const
{
    if (0 == strcmp(currentState->getName(), this_state))    return true;
    
    return false;
}

FSM_TEMPLATE_
FINITE_STATE_MACHINE_::FiniteStateMachine() : totalStates(0), currentState(NULL)
{
}

FSM_TEMPLATE_
FINITE_STATE_MACHINE_::~FiniteStateMachine()
{
    if (itsStates != NULL)
    {
        // delete all its States
        for (int i = 0; i < totalStates; i++)
        {
            delete itsStates[i];
        }
    }
}

FSM_TEMPLATE_
void FINITE_STATE_MACHINE_::initialize(StateObjectType* owner, 
                                       const STATE_DEFINITION_* states, const int total_states,
                                       const TRANSITION_DEFINITION_* transitions, const int total_transitions,
                                       const char* this_state)
{
    itsStateObject = owner;

    for (int k = 0; k < total_states; k++)
    {
        addStates(states, total_states, transitions, total_transitions);
    }

    forceToState(this_state);
}

FSM_TEMPLATE_
void FINITE_STATE_MACHINE_::traverse(const int new_event)
{
    if (0 == totalStates)
    {
        return;
    }

    STATE_TRANSITION_* associated_transition = NULL;  
    STATE_* new_state = currentState->match(new_event, associated_transition);
        
    if (new_state != NULL && associated_transition != NULL)
    {
        StateBehaviour behaviour = associated_transition->getStateBehaviour();  

        if (actions == behaviour)
        {
            currentState->doExitAction(itsStateObject);
        }
    
        currentState = new_state;

        associated_transition->doAction(itsStateObject);

        if (actions == behaviour)
        {
            new_state->doEntryAction(itsStateObject);
        }
    }
}

FSM_TEMPLATE_
void FINITE_STATE_MACHINE_::forceToState(const char* state_name)
{
    for (int i = 0; i < totalStates; i++)
    {
        if (0 == strcmp(itsStates[i]->getName(), state_name))
        {
            currentState = itsStates[i];
            break;
        }
    } 
}

FSM_TEMPLATE_
void FINITE_STATE_MACHINE_::addStates(const STATE_DEFINITION_* states, const int total_states,
                                      const TRANSITION_DEFINITION_* &transitions, const int total_transitions)
{
    for (int i = 0; i < total_states; i++)
    {
            
            itsStates[i] = new STATE_(states[i].stateName, 
                                      states[i].stateEntryAction, 
                                      states[i].stateExitAction);
    }
    totalStates = total_states;

    for (int i = 0; i < total_states; i++)
    {
        itsStates[i]->addTransitions(transitions, total_transitions, itsStates, total_states);
    }
}

#endif
