#include "mbed.h"
#include <stdlib.h>     // srand, rand
#include <time.h>       // time

Serial pc(USBTX, USBRX);

typedef enum { RED, GREEN, BLUE } color;
typedef enum { SELECT, STROBE, FLOW } mode;

PwmOut red(p23);
PwmOut green(p24);
PwmOut blue(p22);
AnalogIn slider(p16);
InterruptIn c_button(p6);
InterruptIn m_button(p10);

mode m_select = SELECT; // start with SELECT mode selected
color c_select = RED;   // start with RED color selected
float latch_red = 0, latch_green = 0, latch_blue = 0;   // for storing previous rgb values temporarily
float red_goal = 0, green_goal = 0, blue_goal = 0;  // for FLOW mode, the goal rgb values
float r = 0, g = 0, b = 0;  // the current rgb values
int count = 0;

void switch_mode() {
    latch_red = r;
    latch_green = g;
    latch_blue = b;
    count = 0;
    
    // switch mode
    if (m_select == SELECT) {
        // switch to STROBE mode if currently in SELECT mode
        pc.printf("STROBE\n");
        m_select = STROBE;
    } else if (m_select == STROBE) {
        // switch to FLOW mode if currently in STROBE mode
        pc.printf("FLOW\n");
        m_select = FLOW;
    } else if (m_select == FLOW) {
        // switch to SELECt mode if currently in FLOW mode
        pc.printf("SELECT\n");
        m_select = SELECT;
    }
}

void switch_color() {
    float temp_r = red, temp_g = green, temp_b = blue;
    
    // switch color selected and flash purely the new color selected for half a second
    if (c_select == RED) {   
        c_select = GREEN;
        red = 0.0f;
        green = 1.0f;
        blue = 0.0f;
    } else if (c_select == GREEN) {
        c_select = BLUE;
        red = 0.0f;
        green = 0.0f;
        blue = 1.0f;
    } else if (c_select == BLUE) {
        c_select = RED;
        red = 1.0f;
        green = 0.0f;
        blue = 0.0f;
    }
    wait(0.5f);
    
    // return to original rgb color
    red = temp_r;
    green = temp_g;
    blue = temp_b;
}

int main() {
    float s = 0, delay = 0; // initialize slider value variable and delay value variable
    int flow_steps = 25;    // the number of steps the flow mode takes to reach each goal color
    srand(time(NULL));      // initialize random seed
    
    while(1) {
        // if color button is pressed, cause interrupt that switches the color currently selected
        c_button.fall(&switch_color);
        // if mode button is pressed, cause interrupt that switches the mode currently selected
        m_button.fall(&switch_mode);
        
        // read slider value
        s = slider;
        if (m_select == SELECT) {
            // slider changes value of of current color selected
            if (c_select == RED)          r = s;
            else if (c_select == GREEN)   g = s;
            else if (c_select == BLUE)    b = s;
            wait(0.1f);
        } else if (m_select == STROBE) {
            // determine the frequency of the strobe (2 * delay is period with 50% duty cycle);
            delay = 0.02f+pow((1.0f-s),2);
            
            if (count == 0) {
                count = 1;
                // turn the LEDs off for time delay
                red = 0.0f;
                green = 0.0f;
                blue = 0.0f;
            } else {
                count = 0;
                // turn them back on again for time delay
                r = latch_red;
                g = latch_green;
                b = latch_blue;
            }
            
            wait(delay);
        } else if (m_select == FLOW) {
            // determine the amount of delay between steps toward goal color
            delay = 0.02f+pow((1.0f-s),2)/2;
        
            // if count is zero, generate new goal color
            if (count == 0) {
                latch_red = r, latch_green = g, latch_blue = b;
                // set new random goal color to flow to
                red_goal = 0.25f * (rand() % 5);
                green_goal = 0.25f * (rand() % 5 );
                blue_goal = 0.25f * (rand() % 5);
                pc.printf("\nr %1.2f g %1.2f b %1.2f\n", red_goal, green_goal, blue_goal);
            }
            
            // count the number of steps toward goal color taken
            count++;
            
            // if goal color has been reached, reset count (causes new goal color to be generated)
            if (count > flow_steps) count = 0;
            
            // increment/decrement rgb values toward goal color
            r += (red_goal - latch_red) / flow_steps;
            g += (green_goal - latch_green) / flow_steps;
            b += (blue_goal - latch_blue) / flow_steps;
            
            // rate of flow is determined by slider
            wait(delay);
        }
        
        // apply new rgb colors
        red = r;
        green = g;
        blue = b;
        pc.printf("[ r %1.2f g %1.2f b %1.2f ] ", r, g, b);
    }
}
