/// @file IHM_L476.h Bibliothéque de classe pour la carte L476 compatible IHM NBOARD
/// @mainpage
///
///
///
/// @author Bruno LARNAUDIE, IUT Cachan
///
/// @note
/// History
/// v3.00 31/10/2019

#include "mbed.h"
#include "LCD_DISCO_L476VG.h"
#include "stm32l476g_discovery_glass_lcd.h"
#include "lcd.h"

//
///
/// Example:
/// @code
///    #include "ihm_L476.h"
/// 
/// IHM_L476 ihm;           // Déclaration de l'objet lcd permettant d'utiliser l'écran LCD
/// 
/// //-------- 2 LED (verte / rouge) seulement sur la Disco L476 (+ bargraph de 4 barres sur le LCD)
/// DigitalOut led_green(PE_8);
/// // Définir ici l'objet led rouge voir sur le board pinout à l'adresse suivante: https://os.mbed.com/platforms/ST-Discovery-L476VG/
/// 
/// //------------- les 5 BP du joystick (attention : celui du milieu est câblé différemment !)
/// DigitalIn central(PA_0, PullNone);    // le BP central du joystick a une Res EXTERNE de PullDown (1 si enfoncé) -> PullNone (ou PullDown)
/// // PA_0 ou USER_BUTTON
/// DigitalIn left(PA_1,PullDown);       // Pas de res externe (et broche Com à Vcc) sur le schéma des 4 directions du joystick
/// // Définir ici les objets boutons droite, haut et bas de la même facon que pour l'objet gauche
/// // voir sur le board pinout à l'adresse suivante: https://os.mbed.com/platforms/ST-Discovery-L476VG/
/// 
/// int main()
/// {
///     unsigned char compteur=0;
/// 
///     ihm.LCD_printf("HELLO ");   // affichage sur l'afficheur lcd
///     wait(3);                    // attente de 3 seconde
///     ihm.LCD_printf("Programme d'exemple de la carte L476");
///     wait(10);                    // attente de 10 seconde
///     ihm.LCD_clear();            // efface l'écran
/// 
///     while (true) {              // boule infinie. Equivalent à: while(1)
///         led_green = !led_green; // change l'état de la led
///         ihm.LCD_printf("%03d:%02xh",compteur,compteur);  // Affichage sur le ihm du compteur en décimal et hexa
///         ihm.BAR_set(compteur);  // Affichage sur le bargraphe de l'écran
///         wait(1);               // attente 1 seconde
///         if (central)
///             compteur=0;    // LED s'allume à 1 / BP enfoncé à 1
///         else
///            compteur++;             // incrémente le compteur. Equivalent à: compteur=compteur+1;
///     }
/// }
/// @endcode
///

class IHM_L476
{
public:
    /// Constructeur pour l'IHM de la carte L476, placé à l'extérieur du main
    ///
    IHM_L476();

    ~IHM_L476();
    /// commande des 4 leds du bargraph sur le lcd
    ///
    /// Affiche la valeur passée en argument sur le bargraph
    /// @param valeur à afficher sur le bargraph
    /// @returns void
    ///   
    void BAR_set(unsigned char valeur);
    
    /// effacement de l'afficheur LCD
    ///
    /// @param  void
    /// @returns void
    ///
    void LCD_clear(void);
    
    /// écriture formatée sur l'afficheur LCD
    /// si affichage de plus de 6 caractéres 
    /// alors l'affichage se met en mode défilement
    /// @param  format...
    /// @returns void
    ///
    void LCD_printf(const char* format, ...);

private:

};


void BSP_LCD_GLASS_CLEAR_WITHOUT_BAR(void);

void BSP_LCD_GLASS_SetBar(uint8_t Value);

void BSP_LCD_GLASS_DisplayString_iut_cachan(uint8_t *ptr);

void BSP_LCD_GLASS_ScrollSentence_iut_cachan(uint8_t* ptr, uint8_t size, uint16_t nScroll, uint16_t ScrollSpeed);

void Decalage_sentence(void);

extern uint8_t Global_Bar_Value;

extern LCD_HandleTypeDef LCDHandle;

extern const uint16_t CapLetterMap[26];

extern const uint16_t NumberMap[10];

extern uint32_t Digit[4];     /* Digit frame buffer */

extern uint8_t LCDBar;

