#include "mbed.h"
#include <math.h>
#include "Function.h"
#include "millis.h"
#include "SPATIAL.h"
#include "ROBOFRIEN_GUI.h"
#include "ROBOFRIEN_LED.h"
#include "ROBOFRIEN_MOTOR.h"
#include "ROBOFRIEN_PID.h"
#include "ROBOFRIEN_SBUS.h"
#include "ROBOFRIEN_BAT.h"



/* ************  USER INTERFACE ************** */
#define SENSOR_WAIT                 1
#define THROTTLE_MAX                7000
#define THROTTLE_BASE               4000
#define THROTTLE_MIN                3000
#define THROTTLE_CONTROL_LIMIT      1500
#define LIMIT_PROGRESS_VELOCITY     10
#define LIMIT_UP_VELOCITY           4
#define LIMIT_DOWN_VELOCITY         2
#define LIMIT_PLANE_VEL_INTEG       5           // Degree
#define LIMIT_VERTICAL_VEL_INTEG    500
/* ******************************************* */

/* *************************** */
// AutoFlight Sequence
// 0. Wait to Start
// 1. Take Off  (10m From Ground)
// 2. Heading Align to Marker
// 3. Go to Point ( Pitching & Altitude )
// 4. Hovering
// 5. Landing
#define WAIT_TO_START   1
#define TAKE_OFF        2
#define HEADING_ALIGN   3
#define GO_TO_POINT     4
#define CHECK_MARKER    5
#define LANDING         6
#define WAIT            7
int ex_AUTOFLIGHT_SEQUENCE;
int AUTOFLIGHT_SEQUENCE = WAIT_TO_START;
unsigned long AUTOFLIGHT_TIME_CNT;
/* *************************** */

ROBOFRIEN_GUI   GUI;
ROBOFRIEN_LED   LED;
ROBOFRIEN_MOTOR MOTOR;
ROBOFRIEN_PID   PID;
SPATIAL         SPATIAL;
ROBOFRIEN_SBUS  SBUS;
ROBOFRIEN_BAT   BAT;


void GUI_UPDATE();
void ALGORITHM();
void ANGLE_CONTROL(double [3]);
void PLANE_VEL_CONTROL(double, double, double *, double *);
void VERTICAL_VEL_CONTROL(double, double *);
void VEL_INTEG_RESET();
float DEG_TO_METER(float , float , float , float );

unsigned long millis_100hz, millis_10hz, millis_200hz;
double want_Angle[3];
double want_Rate[3];
double err_Angle[3];
double THROTTLE_VALUE;
double want_LNG, want_LAT, want_ALT;
int main()
{
  LED.Init();
  MOTOR.Init();
  wait(1);
  SPATIAL.Init(SENSOR_WAIT);
  SBUS.Init();
  PID.Init();
  GUI.Init();
  BAT.Init(3);        // 3cell
  millisStart();
  while (1) {
    GUI.pc_rx_update();
    GUI.TRANS_BUFFER();
    LED.update(GUI.headlight_period, GUI.headlight_dutyrate, GUI.sidelight_period, GUI.sidelight_dutyrate );
    SPATIAL.SPATIAL_RECEIVE();
    SBUS.update();
    if ( ( millis() - millis_200hz ) > 5 ) {
      millis_200hz = millis();
      if ( SPATIAL.SPATIAL_STABLE() == false ) {
        LED.ALARM(1);
        if (GUI.button[0] == true) {
          ALGORITHM();
          MOTOR.update();
        } else {
          MOTOR.OFF();
        }
      } else {
        LED.ALARM(0);
        ALGORITHM();
        MOTOR.update();
      }
    }

    if ( (millis() - millis_10hz ) > 100) {
      millis_10hz = millis();
      GUI_UPDATE();
      GUI.Trans();
      GUI.SET_DATA();
    }
    if (GUI.rx_bool() == true) {
      GUI.Refresh();
    }
  }
}


uint8_t EX_CONTROLLER_STATE;
bool MOTOR_FORCED_OFF_BOOL = true;
void ALGORITHM() {       // 200 Hz //
  /* *************************** */
  /* ******* Want Angle ******** */
  /* *************************** */
  switch (SBUS.CONTROLLER_STATE) {
    case SIGNAL_OFF: {
        want_Angle[0] = 0;  want_Angle[1] = 0;  want_Angle[2] = SPATIAL.YAW;
        THROTTLE_VALUE = 0;
      } break;
    case MOTOR_OFF: {
        MOTOR_FORCED_OFF_BOOL = true;
        want_Angle[0] = 0;  want_Angle[1] = 0;  want_Angle[2] = SPATIAL.YAW;
        THROTTLE_VALUE = 0;
      } break;
    case MANUAL_ATTITUDE: {
        MOTOR_FORCED_OFF_BOOL = false;
        want_Angle[0] = ((SBUS.channel[0]) / 1000.0) * GUI.limit_rollx100 / 100.0;       // Want Angle : Roll
        want_Angle[1] = ((SBUS.channel[1]) / 1000.0) * GUI.limit_pitchx100 / 100.0;      // Want Angle : Pitch
        if ( (SBUS.channel[3] > 3) | (SBUS.channel[3] < -3) )  want_Angle[2] += ((SBUS.channel[3]) / 2000.0) / 200.0 * (GUI.limit_yaw_ratex100 / 100.0);
        if ( want_Angle[2] > 360) want_Angle[2] -= 360;
        else if ( want_Angle[2] < 0) want_Angle[2] += 360;
        THROTTLE_VALUE = ((SBUS.channel[2] + 1000) / 2000.0 * THROTTLE_MAX);
      } break;
    case MANUAL_VELOCITY ... AUTO_FLIGHT: {         // MANUAL_VELOCITY(3), MANUAL_POSITION(4), AUTO_FLIGHT(5) //
        //// [LNG]Longitude Up : East Face, Longitude Down : West Face
        //// [LAT]Latitude Up : North Face, Latitude Down : South Face
        if ( EX_CONTROLLER_STATE != SBUS.CONTROLLER_STATE) {
          want_LAT = SPATIAL.LATITUDE;
          want_LNG = SPATIAL.LONGITUDE;
          want_ALT = SPATIAL.HEIGHT;
          AUTOFLIGHT_TIME_CNT = 0;
        }

        double want_Vel[3];  // VEL[0] = X-axis, VEL[1] = Y-axis, VEL[2] = Z-axis
        if ( SBUS.CONTROLLER_STATE == MANUAL_VELOCITY ) {
          /* ******************************************************* */
          /* ****************** MANUAL VELOCITY ******************** */
          /* ******************************************************* */
          MOTOR_FORCED_OFF_BOOL = false;
          want_Vel[0] = -(SBUS.channel[1] / 1000.0) * LIMIT_PROGRESS_VELOCITY;
          want_Vel[1] = (SBUS.channel[0] / 1000.0) * LIMIT_PROGRESS_VELOCITY;
          if ( SBUS.channel[2] >= 0 )      want_Vel[2] = (SBUS.channel[2] / 1000.0) * LIMIT_UP_VELOCITY;
          else if ( SBUS.channel[2] < 0 )  want_Vel[2] = (SBUS.channel[2] / 1000.0) * LIMIT_DOWN_VELOCITY;
          if ( (SBUS.channel[3] > 3) | (SBUS.channel[3] < -3) )  want_Angle[2] += ((SBUS.channel[3]) / 2000.0) / 200.0 * (GUI.limit_yaw_ratex100 / 100.0);

        } else {
          if ( SBUS.CONTROLLER_STATE == MANUAL_POSITION ) {
            /* ******************************************************* */
            /* ****************** MANUAL POSITION ******************** */
            /* ******************************************************* */
            MOTOR_FORCED_OFF_BOOL = false;
            double del_pos[3];
            del_pos[0] = (-(SBUS.channel[1] / 1000.0) * LIMIT_PROGRESS_VELOCITY);  //divided by time
            del_pos[1] = ((SBUS.channel[0] / 1000.0) * LIMIT_PROGRESS_VELOCITY);   //divided by time
            if (SBUS.channel[2] >= 0)        del_pos[2] = ((SBUS.channel[2] / 1000.0) * LIMIT_UP_VELOCITY);  //divided by time
            else if (SBUS.channel[2] < 0)    del_pos[2] = ((SBUS.channel[2] / 1000.0) * LIMIT_DOWN_VELOCITY);  //divided by time

            double NED_del_pos[3];
            NED_del_pos[0] = del_pos[0] * cos(SPATIAL.YAW * M_PI / 180.0) - del_pos[1] * sin(SPATIAL.YAW * M_PI / 180.0);
            NED_del_pos[1] = del_pos[0] * sin(SPATIAL.YAW * M_PI / 180.0) + del_pos[1] * cos(SPATIAL.YAW * M_PI / 180.0);

            if( (SBUS.channel[1] < 5) & (SBUS.channel[1] > -5) ) {}
            else want_LAT += (NED_del_pos[0] / 200.0) / 110574.0;
            if( (SBUS.channel[0] < 5) & (SBUS.channel[0] > -5) ) {}
            else want_LNG += (NED_del_pos[1] / 200.0) / 111320.0;
            if( (SBUS.channel[2] < 10) & (SBUS.channel[2] > -10) ) {}
            else want_ALT += (del_pos[2] / 200.0);

            if ( (SBUS.channel[3] > 3) | (SBUS.channel[3] < -3) )  want_Angle[2] += ((SBUS.channel[3]) / 2000.0) / 200.0 * (GUI.limit_yaw_ratex100 / 100.0);

          } else if ( SBUS.CONTROLLER_STATE == AUTO_FLIGHT) {
            /* ******************************************************* */
            /* ******************    AUTO FLIGHT  ******************** */
            /* ******************************************************* */
            if (AUTOFLIGHT_SEQUENCE != ex_AUTOFLIGHT_SEQUENCE) {
              AUTOFLIGHT_TIME_CNT = 0;
            }
            ex_AUTOFLIGHT_SEQUENCE = AUTOFLIGHT_SEQUENCE;
            AUTOFLIGHT_TIME_CNT ++;
            if ( AUTOFLIGHT_SEQUENCE == WAIT_TO_START) MOTOR_FORCED_OFF_BOOL = true;
            else MOTOR_FORCED_OFF_BOOL = false;
            switch (AUTOFLIGHT_SEQUENCE) {
              case WAIT_TO_START: {
                  if ( AUTOFLIGHT_TIME_CNT > (5 * 200) ) AUTOFLIGHT_SEQUENCE = TAKE_OFF;
                  want_LAT = SPATIAL.LATITUDE;
                  want_LNG = SPATIAL.LONGITUDE;
                  want_ALT = SPATIAL.HEIGHT;
                  want_Angle[2] = SPATIAL.YAW;
                  break;
                }
              case TAKE_OFF: {            /// Take off [ Ground --> 5m ]  in 1m/s
                  want_ALT += (1 / 200.0);
                  if ( AUTOFLIGHT_TIME_CNT > (5 * 200)) {
                    AUTOFLIGHT_SEQUENCE = CHECK_MARKER;
                    GUI.CurrentMarker = 1;
                  }
                  break;
                }
              case HEADING_ALIGN: {
                  float BEARING_ANGLE = bearing( SPATIAL.LATITUDE, SPATIAL.LONGITUDE, (float)(GUI.Marker_Lat[GUI.CurrentMarker - 1]) / 1000000.0, (float)(GUI.Marker_Lng[GUI.CurrentMarker - 1]) / 1000000.0);
                  want_Angle[2] = BEARING_ANGLE;
                  if ( ABSOL((BEARING_ANGLE - SPATIAL.YAW)) < 5 ) {
                    if (AUTOFLIGHT_TIME_CNT > (3 * 200) ) AUTOFLIGHT_SEQUENCE = GO_TO_POINT;
                  } else AUTOFLIGHT_TIME_CNT = 0;
//                  GUI.DEBUGx100[0] = (BEARING_ANGLE + 100) * 100;
//                  GUI.DEBUGx100[1] = (ABSOL((BEARING_ANGLE - SPATIAL.YAW)) + 100) * 100;
                  break;
                }
              case GO_TO_POINT: {
                  float Target_LAT = GUI.Marker_Lat[GUI.CurrentMarker - 1] / 1000000.0;
                  float Target_LNG = GUI.Marker_Lng[GUI.CurrentMarker - 1] / 1000000.0;
                  float Target_ALT = GUI.Marker_Alt[GUI.CurrentMarker - 1] / 100.0;
//                  float BEARING_ANGLE = bearing( SPATIAL.LATITUDE, SPATIAL.LONGITUDE, (float)(GUI.Marker_Lat[GUI.CurrentMarker - 1]) / 1000000.0, (float)(GUI.Marker_Lng[GUI.CurrentMarker - 1]) / 1000000.0);
//                  want_Angle[2] = BEARING_ANGLE;
                  // 1ms ++
                  /*                            if(Target_LAT > want_LAT) want_LAT += (1.5/200.0)/110574.0;
                                              else want_LAT -= (1/200.0)/110574.0;
                                              if(Target_LNG > want_LNG) want_LNG += (1.5/200.0)/111320.0;
                                              else want_LNG -= (1/200.0)/111320.0;
                                              if(Target_ALT > want_ALT) want_ALT += (1/200.0);
                                              else want_ALT -= (1/200.0);
                  */
//                  want_LAT = Target_LAT;
//                  want_LNG = Target_LNG;
//                  want_ALT = Target_ALT;
                  if(Target_LAT > want_LAT) want_LAT += (LIMIT_PROGRESS_VELOCITY/200.0)/110574.0;
                  else want_LAT -= (LIMIT_PROGRESS_VELOCITY/200.0)/110574.0;
                  if(Target_LNG > want_LNG) want_LNG += (LIMIT_PROGRESS_VELOCITY/200.0)/111320.0;
                  else want_LNG -= (LIMIT_PROGRESS_VELOCITY/200.0)/111320.0;
                  if(Target_ALT > want_ALT) want_ALT += (LIMIT_UP_VELOCITY/200.0);
                  else want_ALT -= (LIMIT_DOWN_VELOCITY/200.0);

                  bool escape_bool = false;
                  float POSITION_DISTANCE;
                  float HEIGHT_DISTANCE;
                  POSITION_DISTANCE = DEG_TO_METER( want_LAT, want_LNG, SPATIAL.LATITUDE, SPATIAL.LONGITUDE );
                  HEIGHT_DISTANCE = sqrt( (want_ALT - SPATIAL.HEIGHT) * (want_ALT - SPATIAL.HEIGHT) );
                  if ( ( POSITION_DISTANCE < 5 ) & (HEIGHT_DISTANCE < 5)) {     // 5m
                    if ( (POSITION_DISTANCE < 3) ) {                            // 3m
                      if ( (POSITION_DISTANCE < 1) ) {                          // 1m
                        ///////////// Target in the 1m OK !!!! /////////////////
                        escape_bool = true;
                        /////////////////////////////////////////////////
                      }
                      else {
                        if ( AUTOFLIGHT_TIME_CNT > (5 * 200) ) escape_bool = true;       // Time Out //
                      }
                    }
                    else {
                      if ( AUTOFLIGHT_TIME_CNT > (5 * 200) ) escape_bool = true;           // Time Out //
                    }
                  }
                  else {
                    AUTOFLIGHT_TIME_CNT = 0;
                  }
                  if ( escape_bool == true) {
                    AUTOFLIGHT_SEQUENCE = CHECK_MARKER;
                    GUI.CurrentMarker ++;
                  }
                  // 1m내외
                  break;
                }
              case CHECK_MARKER: {
                  if (AUTOFLIGHT_TIME_CNT > (3 * 200) ) {
                    if ( (GUI.Marker_Lat[GUI.CurrentMarker - 1] != 0) & (GUI.Marker_Lng[GUI.CurrentMarker - 1] != 0) ) {
                      AUTOFLIGHT_SEQUENCE = HEADING_ALIGN;
                    } else {
                      AUTOFLIGHT_SEQUENCE = LANDING;
                    }
                  }
                  break;
                }
              case LANDING: {
                  want_ALT -= (0.5 / 200.0);    // 0.5m/s
                  break;
                }
              case WAIT: {
                  break;
                }
            }
          }
          double NED_want_vel[3];
          NED_want_vel[0] = (want_LAT - SPATIAL.LATITUDE) * 110574.0 * (GUI.gain_px100[4] / 100.0);
          NED_want_vel[1] = (want_LNG - SPATIAL.LONGITUDE) * 111320.0 * (GUI.gain_px100[5] / 100.0);
          NED_want_vel[2] = (want_ALT - SPATIAL.HEIGHT) * (GUI.gain_px100[6] / 100.0);

          want_Vel[0] = NED_want_vel[0] * cos(SPATIAL.YAW * M_PI / 180.0) + NED_want_vel[1] * sin(SPATIAL.YAW * M_PI / 180.0);
          want_Vel[1] = NED_want_vel[1] * cos(SPATIAL.YAW * M_PI / 180.0) - NED_want_vel[0] * sin(SPATIAL.YAW * M_PI / 180.0);
          want_Vel[2] = NED_want_vel[2];

          float LIMIT_PROGRESS_VELOCITY_CALCULATED;
          float POSITION_DISTANCE = DEG_TO_METER( want_LAT, want_LNG, SPATIAL.LATITUDE, SPATIAL.LONGITUDE );
          if ( POSITION_DISTANCE < 20) LIMIT_PROGRESS_VELOCITY_CALCULATED = (0.04 * POSITION_DISTANCE + 0.2) * LIMIT_PROGRESS_VELOCITY;
          else LIMIT_PROGRESS_VELOCITY_CALCULATED = LIMIT_PROGRESS_VELOCITY;

          want_Vel[0] = MIN_MAX(want_Vel[0], -LIMIT_PROGRESS_VELOCITY_CALCULATED, LIMIT_PROGRESS_VELOCITY_CALCULATED);
          want_Vel[1] = MIN_MAX(want_Vel[1], -LIMIT_PROGRESS_VELOCITY_CALCULATED, LIMIT_PROGRESS_VELOCITY_CALCULATED);
          if (want_Vel[2] >= 0)        want_Vel[2] = MIN_MAX(want_Vel[2], -LIMIT_UP_VELOCITY, LIMIT_UP_VELOCITY);
          else if (want_Vel[2] < 0)    want_Vel[2] = MIN_MAX(want_Vel[2], -LIMIT_DOWN_VELOCITY, LIMIT_DOWN_VELOCITY);

        }

        /* ********** WANT ANGLE GENERATE ************ */
        PLANE_VEL_CONTROL(want_Vel[0], want_Vel[1], &want_Angle[0], &want_Angle[1]);
        if ( want_Angle[2] > 360) want_Angle[2] -= 360;
        else if ( want_Angle[2] < 0) want_Angle[2] += 360;

        /* ********** THROTTLE VALUE GENRATE ************* */
        VERTICAL_VEL_CONTROL(want_Vel[2], &THROTTLE_VALUE);
        if (MOTOR_FORCED_OFF_BOOL == true) THROTTLE_VALUE = 0;

      } break;
  }
  EX_CONTROLLER_STATE = SBUS.CONTROLLER_STATE;
  /* ************************** */
  /* ******* Want Rate ******** */
  /* ************************** */
  ANGLE_CONTROL( want_Angle );

  if (GUI.DPN_Info == 0) {
    if ( SBUS.CONTROLLER_STATE <= MOTOR_OFF) {
      PID.RAT_INTEGER_RESET();
      VEL_INTEG_RESET();
      for (int i = 0; i < 6; i++) MOTOR.Value[i] = 0;
    } else {
      int MOTOR_MIN_DEADZONE  = 5;
      for (int i = 0; i < 5; i++) {
        if (GUI.motor_min[i] > 100)GUI.motor_min[i] = 100;
      }
      MOTOR.Value[0] = (GUI.motor_min[0] - MOTOR_MIN_DEADZONE) * 10 + THROTTLE_VALUE + PID.output_Data[0] / 2.0    - PID.output_Data[1] * 0.87    - PID.output_Data[2];
      MOTOR.Value[1] = (GUI.motor_min[1] - MOTOR_MIN_DEADZONE) * 10 + THROTTLE_VALUE + PID.output_Data[0]        + 0                          + PID.output_Data[2];
      MOTOR.Value[2] = (GUI.motor_min[2] - MOTOR_MIN_DEADZONE) * 10 + THROTTLE_VALUE + PID.output_Data[0] / 2.0    + PID.output_Data[1] * 0.87    - PID.output_Data[2];
      MOTOR.Value[3] = (GUI.motor_min[3] - MOTOR_MIN_DEADZONE) * 10 + THROTTLE_VALUE - PID.output_Data[0] / 2.0    + PID.output_Data[1] * 0.87    + PID.output_Data[2];
      MOTOR.Value[4] = (GUI.motor_min[4] - MOTOR_MIN_DEADZONE) * 10 + THROTTLE_VALUE - PID.output_Data[0]        + 0                          - PID.output_Data[2];
      MOTOR.Value[5] = (GUI.motor_min[5] - MOTOR_MIN_DEADZONE) * 10 + THROTTLE_VALUE - PID.output_Data[0] / 2.0    - PID.output_Data[1] * 0.87    + PID.output_Data[2];
      for (int i = 0; i < 6; i++) {
        if ( MOTOR.Value[i] < ((GUI.motor_min[0] - MOTOR_MIN_DEADZONE) * 10 ) ) {
          MOTOR.Value[i] = ((GUI.motor_min[0] - MOTOR_MIN_DEADZONE) * 10 );
        }
      }
    }
  } else if (GUI.DPN_Info == 4) {
    MOTOR.Value[0] = (float)GUI.pwm_info[0] * 10;
    MOTOR.Value[1] = (float)GUI.pwm_info[1] * 10;
    MOTOR.Value[2] = (float)GUI.pwm_info[2] * 10;
    MOTOR.Value[3] = (float)GUI.pwm_info[3] * 10;
    MOTOR.Value[4] = (float)GUI.pwm_info[4] * 10;
    MOTOR.Value[5] = (float)GUI.pwm_info[5] * 10;
  }
}


void GUI_UPDATE() {
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////// [M/S] Mode and State ///////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  /* ************ MODE 1 ************* */
  GUI.Mode1 = SBUS.CONTROLLER_STATE;
  GUI.Alarm = !SPATIAL.SPATIAL_STABLE();

  GUI.MissionState = AUTOFLIGHT_SEQUENCE;
  GUI.Bat1 = BAT.update();
  // [ Button State & Home Point Chksum & Marker Chksum ] is change automatically in "ROBOFRIEN_GUI.cpp" //
  // You Can not change this setting, just use it ///
  if (GUI.button[0] == true) {}
  else {}
  if (GUI.button[1] == true) {
    AUTOFLIGHT_SEQUENCE = WAIT_TO_START;
  }
  else {}
  if (GUI.button[2] == true) {}
  else {}
  if (GUI.button[3] == true) {}
  else {}
  if (GUI.button[4] == true) {}
  else {}

  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////// [GPS] GPS //////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  GUI.utc_time = SPATIAL.UNIX_TIME;
  GUI.latitude = (SPATIAL.LATITUDE * 1000000);
  GUI.longitude = (SPATIAL.LONGITUDE * 1000000);
  GUI.altitude =  SPATIAL.HEIGHT * 100;
  GUI.SatNum = SPATIAL.SATELLITE_NUM;
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  ///////////////////////////////////////////////// AHRS //////////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  GUI.rollx100 = (SPATIAL.ROLL * 100);
  GUI.pitchx100 = (SPATIAL.PITCH * 100);
  GUI.yawx100 = ((SPATIAL.YAW ) * 100);
  GUI.roll_ratex100 =  SPATIAL.ROLL_RATE * 100;
  GUI.pitch_ratex100 =  SPATIAL.PITCH_RATE * 100;
  GUI.yaw_ratex100 =  SPATIAL.YAW_RATE * 100;
  GUI.VXx100 = SPATIAL.Vx * 100;
  GUI.VYx100 = SPATIAL.Vy * 100;
  GUI.VZx100 = SPATIAL.Vz * 100;

  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  ////////////////////////////////////////////// [C/P] CAP/PWM ////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////////////

  GUI.cap[0] = SBUS.channel[0] / 10.0;
  GUI.cap[1] = SBUS.channel[1] / 10.0;
  GUI.cap[2] = SBUS.channel[2] / 10.0;
  GUI.cap[3] = SBUS.channel[3] / 10.0;
  GUI.cap[4] = SBUS.channel[4] / 10.0;
  GUI.cap[5] = SBUS.channel[5] / 10.0;
  GUI.cap[6] = SBUS.channel[6] / 10.0;
  GUI.cap[7] = SBUS.channel[7] / 10.0;

  GUI.pwm[0] = MOTOR.Value[0] / 100;         // (0 ~ 10000) --> ( 0 ~ 100 )
  GUI.pwm[1] = MOTOR.Value[1] / 100;         // (0 ~ 10000) --> ( 0 ~ 100 )
  GUI.pwm[2] = MOTOR.Value[2] / 100;         // (0 ~ 10000) --> ( 0 ~ 100 )
  GUI.pwm[3] = MOTOR.Value[3] / 100;         // (0 ~ 10000) --> ( 0 ~ 100 )
  GUI.pwm[4] = MOTOR.Value[4] / 100;         // (0 ~ 10000) --> ( 0 ~ 100 )
  GUI.pwm[5] = MOTOR.Value[5] / 100;         // (0 ~ 10000) --> ( 0 ~ 100 )

  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  /////////////////////////////////////////// [E/D] Extra & Debug /////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////////////

  GUI.DEBUGx100[2] = (want_Angle[2] + 100) * 100;
  GUI.DEBUGx100[3] = GUI.Controller_Joystick[3];
  GUI.DEBUGx100[4] = GUI.Gimbal_Joystick[0];
  GUI.DEBUGx100[5] = GUI.Gimbal_Joystick[1];
  float PNU_RPM[2];
  PNU_RPM[0] = sin( millis() / 10000.0 ) * 10000;
  PNU_RPM[1] = cos( millis() / 10000.0 ) * 10000;
  GUI.DEBUGx100[6] = PNU_RPM[0];
  GUI.DEBUGx100[7] = PNU_RPM[1];

  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  /////////////////////////////////////////// Configuration ///////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////////////

  ///////////////////////////// -----------------DPN 1-------------------- ////////////////////////////////
  // You can set the this value from "Config.h"  ( EEPROM_MODEL_TYPE1, EEPROM_MODEL_TYPE2_UP, EEPROM_MODEL_TYPE2_DOWN ) //

  ///////////////////////////// -----------------DPN 2-------------------- ////////////////////////////////
  // You can set the this value from "Config.h"  ( MODEL_INFO, FIRMWARE_INFO ) //

  ///////////////////////////// -----------------DPN 3-------------------- ////////////////////////////////
  /// You can set the original value of receiver to [raw_cap] with scale down ( -127 ~ 127 ) ///
  /// You can use the [cap_min[8], cap_neu[8], cap_max[8]] for calibration of RC receiver value //

  ///////////////////////////// -----------------DPN 4-------------------- ////////////////////////////////
  /// You can use the [pwm_info , motor_min[4]] for calibration of motor pwm value //

  ///////////////////////////// -----------------DPN 5-------------------- ////////////////////////////////
  /*
      LED.HeadlightPeriod = GUI.headlight_period;
      LED.HeadlightDutyrate = GUI.headlight_dutyrate;
      LED.SidelightPeriod = GUI.sidelight_period;
      LED.SidelightDutyrate = GUI.sidelight_dutyrate;
  */
  ///////////////////////////// -----------------DPN 6-------------------- ////////////////////////////////
  /// You can set the original value of receiver to [raw_cap] with scale down ( -127 ~ 127 ) ///
  /// You can use the [cap_min[8], cap_neu[8], cap_max[8]] for calibration of RC receiver value //
  GUI.raw_cap[0] = (0 - 150) * 2;
  GUI.raw_cap[1] = (0 - 150) * 2;
  GUI.raw_cap[2] = (0 - 150) * 2;
  GUI.raw_cap[3] = (0 - 150) * 2;
  GUI.raw_cap[4] = (0 - 150) * 2;
  GUI.raw_cap[5] = (0 - 150) * 2;
  GUI.raw_cap[6] = 0;
  GUI.raw_cap[7] = 0;

  if (GUI.attitude_configuration_bool == true) {
    GUI.attitude_configuration_bool = false;
    // You can calibration of attitude (Roll, Pitch) //
    //                GUI.attitude_calibrate(accelData[0]*aRes,accelData[1]*aRes,(accelData[2]*aRes-1));
  }
  if (GUI.Compass_Calibration_Mode == 1) {
    // You can calibration of compass (Yaw)//
    //                if(ex_Compass_Calibration_Mode == 0){
    //                    magnetic_offset_reset();
    //                }else{
    //                    //// calibrating ... /////
    //                    magnetic_calibration();
    //                }
  } else if (GUI.Compass_Calibration_Mode == 2) {
    //                // You can finish the calibration of compass
    //                //// write to eeprom ////
    //                GUI.Compass_Calibration_Mode = 0;
    //                GUI.write_compass_setting_to_eeprom();
  }
  else if (GUI.Compass_Calibration_Mode == 3) {   // declination angle
    SPATIAL.DECLINATION_ANGLE = GUI.declination_angle;
  }
  //            ex_Compass_Calibration_Mode = GUI.Compass_Calibration_Mode;

  ///////////////////////////// -----------------DPN 7-------------------- ////////////////////////////////
  /// You can use the [limit_rollx100, limit_pitchx100, limit_roll_ratex100, limit_pitch_ratex100, limit_yaw_ratex100] for limit the angle and angular velocity //

  ///////////////////////////// -----------------DPN 8-------------------- ////////////////////////////////
  /// You can use the [gain_px100[20], gain_dx100[20], gain_ix100[20]] for gain tuning //
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////// Refresh ////////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////////////
  //    GUI.trans_flight_data(255, 0);
  GUI.pc_rx_update();
}

void ANGLE_CONTROL(double IN_WANT_ANGLE[3]) {
  /* ********* MIN - MAX ************* */
  IN_WANT_ANGLE[0] = MIN_MAX(IN_WANT_ANGLE[0], -GUI.limit_rollx100 / 100.0, GUI.limit_rollx100 / 100.0 );
  IN_WANT_ANGLE[1] = MIN_MAX(IN_WANT_ANGLE[1], -GUI.limit_pitchx100 / 100.0, GUI.limit_pitchx100 / 100.0 );
  err_Angle[0] = IN_WANT_ANGLE[0] - SPATIAL.ROLL;
  err_Angle[1] = IN_WANT_ANGLE[1] - SPATIAL.PITCH;
  err_Angle[2] = IN_WANT_ANGLE[2] - SPATIAL.YAW;
  if (err_Angle[2] > 180) err_Angle[2] -= 360;
  else if (err_Angle[2] < -180) err_Angle[2] += 360;
  want_Rate[0] = err_Angle[0] * GUI.gain_px100[0] / 100.0;    // Want Rate : Roll Rate
  want_Rate[1] = err_Angle[1] * GUI.gain_dx100[0] / 100.0;    // Want Rate : Pitch Rate
  want_Rate[2] = err_Angle[2] * GUI.gain_ix100[0] / 100.0;    // Want Rate : Yaw Rate
  want_Rate[0] = MIN_MAX(want_Rate[0], -GUI.limit_roll_ratex100 / 100.0, GUI.limit_roll_ratex100 / 100.0);
  want_Rate[1] = MIN_MAX(want_Rate[1], -GUI.limit_pitch_ratex100 / 100.0, GUI.limit_pitch_ratex100 / 100.0);
  want_Rate[2] = MIN_MAX(want_Rate[2], -GUI.limit_yaw_ratex100 / 100.0, GUI.limit_yaw_ratex100 / 100.0);
  PID.err_Rate[0] = (want_Rate[0] - SPATIAL.ROLL_RATE);
  PID.err_Rate[1] = (want_Rate[1] - SPATIAL.PITCH_RATE);
  PID.err_Rate[2] = (want_Rate[2] - SPATIAL.YAW_RATE);
  PID.RAT_KP[0] = -GUI.gain_px100[1] / 100.0;
  PID.RAT_KP[1] = -GUI.gain_px100[2] / 100.0;
  PID.RAT_KP[2] = -GUI.gain_px100[3] / 100.0;
  PID.RAT_KI[0] = -GUI.gain_ix100[1] / 100.0;
  PID.RAT_KI[1] = -GUI.gain_ix100[2] / 100.0;
  PID.RAT_KI[2] = -GUI.gain_ix100[3] / 100.0;
  PID.update();
}

double PLANE_VEL_INTEG[2];
void PLANE_VEL_CONTROL(double IN_VEL_X, double IN_VEL_Y, double *OUTPUT_WANT_ROLL, double *OUTPUT_WANT_PITCH) {
  double err_Vel[2];
  err_Vel[0] = ( IN_VEL_X - SPATIAL.Vx );
  err_Vel[1] = ( IN_VEL_Y - SPATIAL.Vy );
  err_Vel[0] = MIN_MAX(err_Vel[0], -LIMIT_PROGRESS_VELOCITY, LIMIT_PROGRESS_VELOCITY);
  err_Vel[1] = MIN_MAX(err_Vel[1], -LIMIT_PROGRESS_VELOCITY, LIMIT_PROGRESS_VELOCITY);
  *OUTPUT_WANT_PITCH = -(err_Vel[0] * (GUI.gain_dx100[4] / 100.0));
  *OUTPUT_WANT_ROLL  = +(err_Vel[1] * (GUI.gain_dx100[5] / 100.0));

  /* ******** VEL PID - I **************** */
  PLANE_VEL_INTEG[0] -= (err_Vel[0]) * (GUI.gain_ix100[4] / 100.0);
  PLANE_VEL_INTEG[1] += (err_Vel[1]) * (GUI.gain_ix100[5] / 100.0);
  PLANE_VEL_INTEG[0] = MIN_MAX(PLANE_VEL_INTEG[0], -LIMIT_PLANE_VEL_INTEG, LIMIT_PLANE_VEL_INTEG);
  PLANE_VEL_INTEG[1] = MIN_MAX(PLANE_VEL_INTEG[1], -LIMIT_PLANE_VEL_INTEG, LIMIT_PLANE_VEL_INTEG);
  *OUTPUT_WANT_PITCH += PLANE_VEL_INTEG[0];
  *OUTPUT_WANT_ROLL  += PLANE_VEL_INTEG[1];

}

double VERTICAL_VEL_INTEG;
void VERTICAL_VEL_CONTROL(double IN_VEL_Z, double *OUTPUT_THROTTLE_VALUE) {
  double err_Vel_Z;
  err_Vel_Z = IN_VEL_Z - SPATIAL.Vz;
  if (err_Vel_Z >= 0) err_Vel_Z = MIN_MAX(err_Vel_Z, -LIMIT_UP_VELOCITY, LIMIT_UP_VELOCITY);
  else err_Vel_Z = MIN_MAX(err_Vel_Z, -LIMIT_DOWN_VELOCITY, LIMIT_DOWN_VELOCITY);
  *OUTPUT_THROTTLE_VALUE = (err_Vel_Z * 10) * (GUI.gain_dx100[6] / 100.0);

  VERTICAL_VEL_INTEG += (err_Vel_Z) * (GUI.gain_ix100[6] / 100.0);
  VERTICAL_VEL_INTEG = MIN_MAX(VERTICAL_VEL_INTEG, -LIMIT_VERTICAL_VEL_INTEG, LIMIT_VERTICAL_VEL_INTEG);
  *OUTPUT_THROTTLE_VALUE += VERTICAL_VEL_INTEG;

  *OUTPUT_THROTTLE_VALUE = MIN_MAX(*OUTPUT_THROTTLE_VALUE, -THROTTLE_CONTROL_LIMIT, THROTTLE_CONTROL_LIMIT);

  *OUTPUT_THROTTLE_VALUE += THROTTLE_BASE;

  *OUTPUT_THROTTLE_VALUE = MIN_MAX( *OUTPUT_THROTTLE_VALUE, THROTTLE_MIN, THROTTLE_MAX );

}

void VEL_INTEG_RESET() {
  PLANE_VEL_INTEG[0] = 0; PLANE_VEL_INTEG[1] = 0; VERTICAL_VEL_INTEG = 0;
}

float DEG_TO_METER(float lat1, float lon1, float lat2, float lon2) {
  float R = 6378.137;     // Radius of earth in KM
  float dLat = lat2 * M_PI / 180.0 - lat1 * M_PI / 180.0;
  float dLon = lon2 * M_PI / 180.0 - lon1 * M_PI / 180.0;
  float a = sin(dLat / 2.0) * sin(dLat / 2.0) + cos(lat1 * M_PI / 180.0) * cos(lat2 * M_PI / 180.0) * sin(dLon / 2.0) * sin(dLon / 2.0);
  float c = 2 * atan2(sqrt(a), sqrt(1 - a));
  float d = R * c;
  return d * 1000;    // meters
}
