
/*! \file I2CEeprom.h */
/*! \class I2CEeprom */
/*! \brief Simple access class for I2C EEPROM chips like Microchip 24LC
 */

/* Copyright (c) 2015 Robin Hourahane
*  Copyright (c) 2020 Andy Little
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#ifndef __I2CEEPROM_H__
#define __I2CEEPROM_H__

#include <mbed.h>

/// Based on the original Mbed I2CEeprom library by Robin Hourahane.
///
/// Class to provide simple access to I2C EEPROM chips like Microchip's 24LC
/// or AMTELS AT24C series. Chips up to 64Kb in size are directly supported.
///
/// The library was tested on a Microchip 24LC128.
///
/// The I2CEeprom class handles multiple page writes so any amount of data can
/// be written in a single call to write.
///
/// The code is modified from the original to better support RTOS.
/// At start of a write(addr,buffer,size), a buffer of up to pageSize + 2
/// is allocated on the heap, and the data and address are copied to it.
/// This allows the cuurent write chunk to proceed without unlocking, which
/// prevents another device on the bus access and so aborting the write.
///
/// This allocation per write fits my usecase, where eeeprom variables
/// are written in a special menu mode at start up, but it may be better to
/// preallocate a buffer once at startup, if writes are occurring at arbitrary
/// times during execution, to prevent surprises with out of memory issues
/// when trying to write during normal execution.
///
/// The constructor takes the I2C bus by non-const reference argument. This
/// enables a single I2C bus to be shared among many i2c devices.
/// Mbed OS enforces a lock on the bus during a bus read/write so reads
/// are atomic. Writes are split into separate atomic chunks that only write
/// one eeprom page. After a chunk is sent, the thread is sent to sleep for
/// writeCycleTime_ms specified in the constructor arg, and is then polled
/// at 1 ms intervals to discover whether the write has completed.

class I2CEeprom
{
public:
    /// Constructor to create a new instance of the class.
    /// @param i2c A reference to the i2c bus that the chip is connected to.
    /// @param address The 8bit device I2C address
    /// @param pageSize The device page size.
    /// @param chipSize The device size for range check.
    /// @param writeCycleTime_ms The device write cycle time in ms.
    I2CEeprom(
        I2C& i2c,
        int address,
        size_t pageSize,
        size_t chipSize,
        uint8_t writeCycleTime_ms);

    /// Read a single byte from the EEprom.
    /// @param address EEprom read address.
    /// @param value Memory reference of char to read into.
    /// @returns Number of bytes read .
    size_t read(size_t address, char &value);

    /// Read multiple bytes from the EEprom.
    /// @param address EEprom start read address.
    /// @param buffer buffer to read into.
    /// @param size Number of bytes to read.
    /// @returns Number of bytes read.
    size_t read(size_t address, char *buffer, size_t size);

    /// Deserialise object of type T from Eeprom memory.
    /// @param address EEprom start address
    /// @param value Memory Reference of object to deserialise into .
    /// @returns Number of bytes read.
    template<typename T> size_t read(size_t address, T &value)
    {
        return read(address, reinterpret_cast<char *>(&value), sizeof(T));
    }

    /// Write a char to EEprom.
    /// @param address Eeprom write address.
    /// @param value Char value to write.
    /// @returns Number of bytes written.
    size_t write(size_t address, char value);

    /// Write multiple bytes to EEprom.
    /// In this implementation, the write is split into chunks of up to pageSize
    /// and for each chunk a buffer of up to pageSize + 2 is temporarily
    /// allocated on the heap while the write is in progress.
    /// @param address EEprom start address.
    /// @param buffer Buffer to write.
    /// @param size Number of bytes to write.
    /// @returns Number of bytes written.
    size_t write(size_t address, const char *buffer, size_t size);

    /// Serialise an object of type T.
    /// @param address EEprom start address.
    /// @param value Object to be serialised.
    /// @returns Number of bytes written.
    template<typename T> size_t write(size_t address, const T &value)
    {
        return write(address, reinterpret_cast<const char *>(&value), sizeof(T));
    }

private:
    /// Sleep thread while write completes.
    void waitForWrite();
    ///  atomic chunk up to page size write
    size_t ll_write(size_t address, const char *buffer, size_t size);

    /// Validate that proposed operation is in bounds.
    bool checkSpace(size_t address, size_t size)
    {
        return (address + size) < m_chipSize ;
    }

private:
    I2C & m_i2c;
    int const m_i2cAddress;
    size_t const m_chipSize;
    size_t const m_pageSize;
    uint8_t const m_writeCycleTime_ms;
};

#endif