/***************************************************
  This is a library for the Adafruit 1.8" SPI display.
  This library works with the Adafruit 1.8" TFT Breakout w/SD card
  ----> http://www.adafruit.com/products/358
  as well as Adafruit raw 1.8" TFT display
  ----> http://www.adafruit.com/products/618

  Check out the links above for our tutorials and wiring diagrams
  These displays use SPI to communicate, 4 or 5 pins are required to
  interface (RST is optional)
  Adafruit invests time and resources providing this open source code,
  please support Adafruit and open-source hardware by purchasing
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.
  MIT license, all text above must be included in any redistribution
 ****************************************************/

#include "mbed.h"
#include "ST7735.h"

Serial output1(USBTX, USBRX);

//uint16_t table[1028][18] __attribute__((section("AHBSRAM0")));

inline uint16_t swapcolor(uint16_t x) { 
  return (x << 11) | (x & 0x07E0) | (x >> 11);
}

// Constructor 
ST7735::ST7735(PinName mosi, PinName miso, PinName sck, PinName cs, PinName rs, PinName rst) 
        : lcdPort(mosi, miso, sck), _cs(cs), _rs(rs), _rst(rst), GFX(ST7735_TFTWIDTH_128, ST7735_TFTHEIGHT_160) 
{ }


void ST7735::writecommand(uint8_t c)
{
    _rs = 0;
    _cs = 0;
    lcdPort.write( c );
    _cs = 1;
}


void ST7735::writedata(uint8_t c)
{
    _rs = 1;
    _cs = 0;
    lcdPort.write( c );

    _cs = 1;
}


// Rather than a bazillion writecommand() and writedata() calls, screen
// initialization commands and arguments are organized in these tables
// stored in PROGMEM.  The table may look bulky, but that's mostly the
// formatting -- storage-wise this is hundreds of bytes more compact
// than the equivalent code.  Companion function follows.
#define DELAY 0x80
static unsigned char
  Bcmd[] = {                  // Initialization commands for 7735B screens
    18,                       // 18 commands in list:
    ST7735_SWRESET,   DELAY,  //  1: Software reset, no args, w/delay
      50,                     //     50 ms delay
    ST7735_SLPOUT ,   DELAY,  //  2: Out of sleep mode, no args, w/delay
      255,                    //     255 = 500 ms delay
    ST7735_COLMOD , 1+DELAY,  //  3: Set color mode, 1 arg + delay:
      0x05,                   //     16-bit color
      10,                     //     10 ms delay
    ST7735_FRMCTR1, 3+DELAY,  //  4: Frame rate control, 3 args + delay:
      0x00,                   //     fastest refresh
      0x06,                   //     6 lines front porch
      0x03,                   //     3 lines back porch
      10,                     //     10 ms delay
    ST7735_MADCTL , 1      ,  //  5: Memory access ctrl (directions), 1 arg:
      0x08,                   //     Row addr/col addr, bottom to top refresh
    ST7735_DISSET5, 2      ,  //  6: Display settings #5, 2 args, no delay:
      0x15,                   //     1 clk cycle nonoverlap, 2 cycle gate
                              //     rise, 3 cycle osc equalize
      0x02,                   //     Fix on VTL
    ST7735_INVCTR , 1      ,  //  7: Display inversion control, 1 arg:
      0x0,                    //     Line inversion
    ST7735_PWCTR1 , 2+DELAY,  //  8: Power control, 2 args + delay:
      0x02,                   //     GVDD = 4.7V
      0x70,                   //     1.0uA
      10,                     //     10 ms delay
    ST7735_PWCTR2 , 1      ,  //  9: Power control, 1 arg, no delay:
      0x05,                   //     VGH = 14.7V, VGL = -7.35V
    ST7735_PWCTR3 , 2      ,  // 10: Power control, 2 args, no delay:
      0x01,                   //     Opamp current small
      0x02,                   //     Boost frequency
    ST7735_VMCTR1 , 2+DELAY,  // 11: Power control, 2 args + delay:
      0x3C,                   //     VCOMH = 4V
      0x38,                   //     VCOML = -1.1V
      10,                     //     10 ms delay
    ST7735_PWCTR6 , 2      ,  // 12: Power control, 2 args, no delay:
      0x11, 0x15,
    ST7735_GMCTRP1,16      ,  // 13: Magical unicorn dust, 16 args, no delay:
      0x09, 0x16, 0x09, 0x20, //     (seriously though, not sure what
      0x21, 0x1B, 0x13, 0x19, //      these config values represent)
      0x17, 0x15, 0x1E, 0x2B,
      0x04, 0x05, 0x02, 0x0E,
    ST7735_GMCTRN1,16+DELAY,  // 14: Sparkles and rainbows, 16 args + delay:
      0x0B, 0x14, 0x08, 0x1E, //     (ditto)
      0x22, 0x1D, 0x18, 0x1E,
      0x1B, 0x1A, 0x24, 0x2B,
      0x06, 0x06, 0x02, 0x0F,
      10,                     //     10 ms delay
    ST7735_CASET  , 4      ,  // 15: Column addr set, 4 args, no delay:
      0x00, 0x02,             //     XSTART = 2
      0x00, 0x81,             //     XEND = 129
    ST7735_RASET  , 4      ,  // 16: Row addr set, 4 args, no delay:
      0x00, 0x02,             //     XSTART = 1
      0x00, 0x81,             //     XEND = 160
    ST7735_NORON  ,   DELAY,  // 17: Normal display on, no args, w/delay
      10,                     //     10 ms delay
    ST7735_DISPON ,   DELAY,  // 18: Main screen turn on, no args, w/delay
      255 },                  //     255 = 500 ms delay

  Rcmd1[] = {                 // Init for 7735R, part 1 (red or green tab)
    15,                       // 15 commands in list:
    ST7735_SWRESET,   DELAY,  //  1: Software reset, 0 args, w/delay
      150,                    //     150 ms delay
    ST7735_SLPOUT ,   DELAY,  //  2: Out of sleep mode, 0 args, w/delay
      255,                    //     500 ms delay
    ST7735_FRMCTR1, 3      ,  //  3: Frame rate ctrl - normal mode, 3 args:
      0x01, 0x2C, 0x2D,       //     Rate = fosc/(1x2+40) * (LINE+2C+2D)
    ST7735_FRMCTR2, 3      ,  //  4: Frame rate control - idle mode, 3 args:
      0x01, 0x2C, 0x2D,       //     Rate = fosc/(1x2+40) * (LINE+2C+2D)
    ST7735_FRMCTR3, 6      ,  //  5: Frame rate ctrl - partial mode, 6 args:
      0x01, 0x2C, 0x2D,       //     Dot inversion mode
      0x01, 0x2C, 0x2D,       //     Line inversion mode
    ST7735_INVCTR , 1      ,  //  6: Display inversion ctrl, 1 arg, no delay:
      0x07,                   //     No inversion
    ST7735_PWCTR1 , 3      ,  //  7: Power control, 3 args, no delay:
      0xA2,
      0x02,                   //     -4.6V
      0x84,                   //     AUTO mode
    ST7735_PWCTR2 , 1      ,  //  8: Power control, 1 arg, no delay:
      0xC5,                   //     VGH25 = 2.4C VGSEL = -10 VGH = 3 * AVDD
    ST7735_PWCTR3 , 2      ,  //  9: Power control, 2 args, no delay:
      0x0A,                   //     Opamp current small
      0x00,                   //     Boost frequency
    ST7735_PWCTR4 , 2      ,  // 10: Power control, 2 args, no delay:
      0x8A,                   //     BCLK/2, Opamp current small & Medium low
      0x2A,  
    ST7735_PWCTR5 , 2      ,  // 11: Power control, 2 args, no delay:
      0x8A, 0xEE,
    ST7735_VMCTR1 , 1      ,  // 12: Power control, 1 arg, no delay:
      0x0E,
    ST7735_INVOFF , 0      ,  // 13: Don't invert display, no args, no delay
    ST7735_MADCTL , 1      ,  // 14: Memory access control (directions), 1 arg:
      0xC0,                   //     row addr/col addr, bottom to top refresh
    ST7735_COLMOD , 1      ,  // 15: set color mode, 1 arg, no delay:
      0x05 },                 //     16-bit color

  Rcmd2green[] = {            // Init for 7735R, part 2 (green tab only)
    2,                        //  2 commands in list:
    ST7735_CASET  , 4      ,  //  1: Column addr set, 4 args, no delay:
      0x00, 0x02,             //     XSTART = 0
      0x00, 0x7F+0x02,        //     XEND = 127
    ST7735_RASET  , 4      ,  //  2: Row addr set, 4 args, no delay:
      0x00, 0x01,             //     XSTART = 0
      0x00, 0x9F+0x01 },      //     XEND = 159
  Rcmd2red[] = {              // Init for 7735R, part 2 (red tab only)
    2,                        //  2 commands in list:
    ST7735_CASET  , 4      ,  //  1: Column addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x7F,             //     XEND = 127
    ST7735_RASET  , 4      ,  //  2: Row addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x9F },           //     XEND = 159

  Rcmd2green144[] = {              // Init for 7735R, part 2 (green 1.44 tab)
    2,                        //  2 commands in list:
    ST7735_CASET  , 4      ,  //  1: Column addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x7F,             //     XEND = 127
    ST7735_RASET  , 4      ,  //  2: Row addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x7F },           //     XEND = 127
      
  Rcmd2green160x80[] = {              // Init for 7735R, part 2 (mini 160x80)
    2,                        //  2 commands in list:
    ST7735_CASET  , 4      ,  //  1: Column addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x7F,             //     XEND = 79
    ST7735_RASET  , 4      ,  //  2: Row addr set, 4 args, no delay:
      0x00, 0x00,             //     XSTART = 0
      0x00, 0x9F+0x01 },           //     XEND = 159

  Rcmd3[] = {                 // Init for 7735R, part 3 (red or green tab)
    4,                        //  4 commands in list:
    ST7735_GMCTRP1, 16      , //  1: Magical unicorn dust, 16 args, no delay:
      0x02, 0x1c, 0x07, 0x12,
      0x37, 0x32, 0x29, 0x2d,
      0x29, 0x25, 0x2B, 0x39,
      0x00, 0x01, 0x03, 0x10,
    ST7735_GMCTRN1, 16      , //  2: Sparkles and rainbows, 16 args, no delay:
      0x03, 0x1d, 0x07, 0x06,
      0x2E, 0x2C, 0x29, 0x2D,
      0x2E, 0x2E, 0x37, 0x3F,
      0x00, 0x00, 0x02, 0x10,
    ST7735_NORON  ,    DELAY, //  3: Normal display on, no args, w/delay
      10,                     //     10 ms delay
    ST7735_DISPON ,    DELAY, //  4: Main screen turn on, no args w/delay
      100 };                  //     100 ms delay


// Companion code to the above tables.  Reads and issues
// a series of LCD commands stored in byte array.
void ST7735::commandList(uint8_t *addr)
{
    
    uint8_t  numCommands, numArgs;
    uint16_t ms;

    numCommands = *addr++;   // Number of commands to follow
    while(numCommands--) {                 // For each command...
        writecommand(*addr++); //   Read, issue command
        numArgs  = *addr++;    //   Number of args to follow
        ms       = numArgs & DELAY;          //   If hibit set, delay follows args
        numArgs &= ~DELAY;                   //   Mask out delay bit
        while(numArgs--) {                   //   For each argument...
            writedata(*addr++);  //     Read, issue argument
        }

        if(ms) {
            ms = *addr++; // Read post-command delay time (ms)
            if(ms == 255) ms = 500;     // If 255, delay for 500 ms
            wait_ms(ms);
        }
    }
}


// Initialization code common to both 'B' and 'R' type displays
void ST7735::commonInit(uint8_t *cmdList)
{

    colstart  = rowstart = 0; // May be overridden in init func

    _rs = 1;
    _cs = 1;

    // use default SPI format
    lcdPort.format(8,0);
    //lcdPort.frequency(16000000);     // Lets try 4MHz
    lcdPort.frequency(64000000);     // Lets try 4MHz

    // toggle RST low to reset; CS low so it'll listen to us
    _cs = 0;
    _rst = 1;
    wait_ms(500);
    _rst = 0;
    wait_ms(500);
    _rst = 1;
    wait_ms(500);

    if(cmdList) commandList(cmdList);
}


// Initialization for ST7735B screens
void ST7735::initB(void)
{
    commonInit(Bcmd);
}


// Initialization for ST7735R screens (green or red tabs)
void ST7735::initR(uint8_t options) {
  commonInit(Rcmd1);
  if(options == INITR_GREENTAB) {
    commandList(Rcmd2green);
    colstart = 2;
    rowstart = 1;
  } else if(options == INITR_144GREENTAB) {
    _height = ST7735_TFTHEIGHT_128;
    _width = ST7735_TFTWIDTH_128;
    commandList(Rcmd2green144);
    colstart = 2;
    rowstart = 3;
  } else if(options == INITR_MINI160x80) {
    _height = ST7735_TFTHEIGHT_160;
    _width = ST7735_TFTWIDTH_80;
    commandList(Rcmd2green160x80);
    colstart = 24;
    rowstart = 0;
  } else if(options == INITR_MINI160x80_RS) {
    _height = ST7735_TFTHEIGHT_160;
    _width = ST7735_TFTWIDTH_80;
    commandList(Rcmd2green160x80);
    writecommand(1 ? ST7735_INVON : ST7735_INVOFF);
    colstart = 26;
    rowstart = 1;
    _tft_type = 2;
  } else {
    // colstart, rowstart left at default '0' values
    commandList(Rcmd2red);
  }
  commandList(Rcmd3);

  // if black, change MADCTL color filter
  if (options == INITR_BLACKTAB) {
    writecommand(ST7735_MADCTL);
    writedata(0xC0);
  }

  tabcolor = options;
}

void ST7735::setAddrWindow(uint8_t x0, uint8_t y0, uint8_t x1,
                                    uint8_t y1)
{

    writecommand(ST7735_CASET); // Column addr set
    writedata(0x00);
    writedata(x0+xstart);     // XSTART
    writedata(0x00);
    writedata(x1+xstart);     // XEND

    writecommand(ST7735_RASET); // Row addr set
    writedata(0x00);
    writedata(y0+ystart);     // YSTART
    writedata(0x00);
    writedata(y1+ystart);     // YEND

    writecommand(ST7735_RAMWR); // write to RAM
}

void ST7735::pushColor(uint16_t color)
{
    _rs = 1;
    _cs = 0;

    lcdPort.write( color >> 8 );
    lcdPort.write( color );
    _cs = 1;
}


void ST7735::drawPixel(int16_t x, int16_t y, uint16_t color)
{

    if((x < 0) ||(x >= _width) || (y < 0) || (y >= _height)) return;

    setAddrWindow(x,y,x+1,y+1);

    _rs = 1;
    _cs = 0;

    lcdPort.write( color >> 8 );
    lcdPort.write( color );

    _cs = 1;
}


void ST7735::drawFastVLine(int16_t x, int16_t y, int16_t h,
                                    uint16_t color)
{

    // Rudimentary clipping
    if((x >= _width) || (y >= _height)) return;
    if((y+h-1) >= _height) h = _height-y;
    setAddrWindow(x, y, x, y+h-1);

    uint8_t hi = color >> 8, lo = color;
    _rs = 1;
    _cs = 0;
    while (h--) {
        lcdPort.write( hi );
        lcdPort.write( lo );
    }
    _cs = 1;
}


void ST7735::drawFastHLine(int16_t x, int16_t y, int16_t w,
                                    uint16_t color)
{

    // Rudimentary clipping
    if((x >= _width) || (y >= _height)) return;
    if((x+w-1) >= _width)  w = _width-x;
    setAddrWindow(x, y, x+w-1, y);

    uint8_t hi = color >> 8, lo = color;
    _rs = 1;
    _cs = 0;
    while (w--) {
        lcdPort.write( hi );
        lcdPort.write( lo );
    }
    _cs = 1;
}



void ST7735::fillScreen(uint16_t color)
{
    fillRect(-10, -10,  _width+20, _height+20, color);
}


// fill a rectangle
void ST7735::fillRect(int16_t x, int16_t y, int16_t w, int16_t h,
                               uint16_t color)
{
    
    // rudimentary clipping (drawChar w/big text requires this)
    if((x >= _width) || (y >= _height)) return;
    if((x + w - 1) >= _width)  w = _width  - x;
    if((y + h - 1) >= _height) h = _height - y;

    setAddrWindow(x, y, x+w-1, y+h-1);

    uint8_t hi = color >> 8, lo = color;
    _rs = 1;
    _cs = 0;
    for(y=h; y>0; y--) {
        for(x=w; x>0; x--) {
            lcdPort.write( hi );
            lcdPort.write( lo );
        }
    }

    _cs = 1;
}


// Pass 8-bit (each) R,G,B, get back 16-bit packed color
uint16_t ST7735::Color565(uint8_t r, uint8_t g, uint8_t b)
{
    return ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
}

// Pass 8-bit (each) R,G,B, get back 16-bit packed color
char* ST7735::Color565string(uint8_t r, uint8_t g, uint8_t b)
{
    uint16_t color;
    char colorString[4];
    
    color = ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
    
    sprintf(colorString, "%04x", color);
    
    //output1.printf("0x%04x : %s\n", color, colorString);
    
    return colorString;
    //return "test";
}


#define MADCTL_MY  0x80
#define MADCTL_MX  0x40
#define MADCTL_MV  0x20
#define MADCTL_ML  0x10
#define MADCTL_RGB 0x00
#define MADCTL_BGR 0x08
#define MADCTL_MH  0x04

void ST7735::setRotation(uint8_t m) {
  _orientation = m;
  writecommand(ST7735_MADCTL);
  rotation = m % 4; // can't be higher than 3
  switch (rotation) {
   case 0:
     if ((tabcolor == INITR_BLACKTAB) || (tabcolor == INITR_MINI160x80)) {
       writedata(MADCTL_MX | MADCTL_MY | MADCTL_RGB);
     } else if (tabcolor == INITR_MINI160x80_RS) {
       writedata(MADCTL_MX | MADCTL_MY | MADCTL_BGR);
     } else {
       writedata(MADCTL_MX | MADCTL_MY | MADCTL_BGR);
     }

     if (tabcolor == INITR_144GREENTAB) {
       _height = ST7735_TFTHEIGHT_128;
       _width  = ST7735_TFTWIDTH_128;
     } else if ((tabcolor == INITR_MINI160x80) || (tabcolor == INITR_MINI160x80_RS))  {
       _height = ST7735_TFTHEIGHT_160;
       _width = ST7735_TFTWIDTH_80;
     } else {
       _height = ST7735_TFTHEIGHT_160;
       _width  = ST7735_TFTWIDTH_128;
     }
     xstart = colstart;
     ystart = rowstart;
     break;
   case 1:
     if ((tabcolor == INITR_BLACKTAB) || (tabcolor == INITR_MINI160x80)) {
       writedata(MADCTL_MY | MADCTL_MV | MADCTL_RGB);
     } else if (tabcolor == INITR_MINI160x80_RS) {
       writedata(MADCTL_MY | MADCTL_MV | MADCTL_BGR);
     } else {
       writedata(MADCTL_MY | MADCTL_MV | MADCTL_BGR);
     }

     if (tabcolor == INITR_144GREENTAB)  {
       _width = ST7735_TFTHEIGHT_128;
       _height = ST7735_TFTWIDTH_128;
     } else if ((tabcolor == INITR_MINI160x80) || (tabcolor == INITR_MINI160x80_RS))  {
       _width = ST7735_TFTHEIGHT_160;
       _height = ST7735_TFTWIDTH_80;
     } else {
       _width = ST7735_TFTHEIGHT_160;
       _height = ST7735_TFTWIDTH_128;
     }
     ystart = colstart;
     xstart = rowstart;
     break;
  case 2:
     if ((tabcolor == INITR_BLACKTAB) || (tabcolor == INITR_MINI160x80)) {
       writedata(MADCTL_RGB);
     } else if (tabcolor == INITR_MINI160x80_RS) {
       writedata(MADCTL_BGR);
     } else {
       writedata(MADCTL_BGR);
     }

     if (tabcolor == INITR_144GREENTAB) {
       _height = ST7735_TFTHEIGHT_128;
       _width  = ST7735_TFTWIDTH_128;
     } else if ((tabcolor == INITR_MINI160x80) || (tabcolor == INITR_MINI160x80_RS))  {
       _height = ST7735_TFTHEIGHT_160;
       _width = ST7735_TFTWIDTH_80;
     } else {
       _height = ST7735_TFTHEIGHT_160;
       _width  = ST7735_TFTWIDTH_128;
     }
     xstart = colstart;
     ystart = rowstart;
     break;
   case 3:
     if ((tabcolor == INITR_BLACKTAB) || (tabcolor == INITR_MINI160x80)) {
       writedata(MADCTL_MX | MADCTL_MV | MADCTL_RGB);
     } else if (tabcolor == INITR_MINI160x80_RS) {
       writedata(MADCTL_MX | MADCTL_MV | MADCTL_BGR);
     } else {
       writedata(MADCTL_MX | MADCTL_MV | MADCTL_BGR);
     }

     if (tabcolor == INITR_144GREENTAB)  {
       _width = ST7735_TFTHEIGHT_128;
       _height = ST7735_TFTWIDTH_128;
     } else if ((tabcolor == INITR_MINI160x80) || (tabcolor == INITR_MINI160x80_RS))  {
       _width = ST7735_TFTHEIGHT_160;
       _height = ST7735_TFTWIDTH_80;
     } else {
       _width = ST7735_TFTHEIGHT_160;
       _height = ST7735_TFTWIDTH_128;
     }
     ystart = colstart;
     xstart = rowstart;
     break;
  }
}

void ST7735::invertDisplay(boolean i)
{
    writecommand(i ? ST7735_INVON : ST7735_INVOFF);
}

void ST7735::DrawBMP(int offsetx, int offsety, FILE *Image)
{  
    uint32_t index = 0, width = 0, height = 0;
    int i=0, x, y, byteCour;
    
    int fpOffset = ftell(Image);
    //output1.printf("current cursor : %02x\n", ftell(Image));
    //output1.printf("first bytes bmp : %02x\n", fgetc(Image));
    
    /* Get bitmap data address offset */
    fseek(Image, 10+fpOffset , SEEK_SET );
    index = fgetc(Image);
    //output1.printf("index : 0x%02x ; ", index);
    
    /* Read bitmap width */
    fseek(Image, 18+fpOffset , SEEK_SET );
    width = fgetc(Image);
    width += fgetc(Image)<<8;
    width += fgetc(Image)<<16;
    width += fgetc(Image)<<24;
    //output1.printf("width : %d ; ", width);
    
    /* Read bitmap height */
    fseek(Image, 22+fpOffset , SEEK_SET );
    height = fgetc(Image);
    height += fgetc(Image)<<8;
    height += fgetc(Image)<<16;
    height += fgetc(Image)<<24;
    //output1.printf("height : %d\n", height);
    
    fseek(Image, index+fpOffset , SEEK_SET );
    
    byteCour = fgetc(Image);
    byteCour += fgetc(Image)<<8;

    for (y = height+offsety; offsety < y; y--) {
        setAddrWindow(offsetx, y, offsetx+width-1, y);
        _rs = 1;
        _cs = 0;
                
        for (x = 0; x < width; x++) {         
            if(x == width-1) fseek(Image, width%2, SEEK_CUR);
                    
            if(x <= width && y >= 0) {
                lcdPort.write(byteCour>>8);    // one line
                lcdPort.write(byteCour);
            }
            _cs = 1;
            byteCour = fgetc(Image);
            byteCour += fgetc(Image)<<8;
            _cs = 0;
        }
    }    
    _cs = 1;
}

// use imageconverter565.exe to generate bitmap array and put it in bitmap.c
void ST7735::DrawRam(unsigned int x, unsigned int y, unsigned int w, unsigned int h, int *Image) {
    unsigned int    i,j;
    setAddrWindow(x, y, x+w-1, y+h);
    _rs = 1;
    _cs = 0;
    for (j = 0; j < h; j++) {        //Lines
        for (i = 0; i < w; i++) {     // copy pixel data to TFT
            lcdPort.write(*Image>>8);    // one line
            lcdPort.write(*Image);  
            *Image++;
        }
    }
    _cs = 1;
}

void ST7735::DrawGIF(int offsetx, int offsety, FILE *Image) {
    
    uint32_t width = 0, height = 0;
    uint16_t code, preCode, clearCode, endCode, sizePalette;
    uint8_t r, v, b, pxl, pxls[256];
    int i, j;
    int fpOffset = ftell(Image);
    
    // palette
    uint16_t table[750][17];
    for(int i = 0; i < 750; i++)
        for(int j = 0; j < 17; j++)
            table[i][j] = NULL;
    
    // Read bitmap width
    fseek(Image, 6+fpOffset , SEEK_SET );
    width = fgetc(Image);
    width += fgetc(Image);
    //output1.printf("width : %d ; ", width);
    
    // Read bitmap height
    fseek(Image, 8+fpOffset , SEEK_SET );
    height = fgetc(Image);
    height += fgetc(Image);
    //output1.printf("height : %d\n", height);
    
    // Get size of palette
    fseek(Image, 10+fpOffset , SEEK_SET );
    sizePalette = fgetc(Image)<<5;
    sizePalette = ((uint8_t) sizePalette)>>5;
    sizePalette = pow(2, sizePalette+1.0);
    
    // Get palette
    uint16_t palette[sizePalette];
    fseek(Image, 13+fpOffset , SEEK_SET );
    for(i=0; i < sizePalette; i++) {
        r = fgetc(Image);
        v = fgetc(Image);
        b = fgetc(Image);
        
        // palette simple
        palette[i] = Color565(r, v, b);
        table[i][0] = 1;table[i][1] = i;
    }
    
    preCode = i;
    table[i][0] = i;
    clearCode = i;
    table[i+1][0] = i+1;
    endCode = i+1;
    
    // Find Image separator
    while(1) {
        pxl = fgetc(Image);
        if(0x2C == pxl) break;
    }
    
    // Get minimal size of code
    fseek(Image, 9 , SEEK_CUR );
    uint8_t sizeCode = fgetc(Image);sizeCode++;
    
    // Get size of block
    uint8_t sizeBlock = fgetc(Image);
    output1.printf("taille du block : %d\n", sizeBlock);
    
    setAddrWindow(offsetx, offsety, offsetx+width-1, offsety+height);
    _rs = 1;
    _cs = 0;
    
    // image data
    uint16_t imgByte = fgetc(Image);
    int sizeImgByte = 8;
    int16_t index = sizePalette;
    for(i=0; i < sizeBlock; ) {
        
        // ajout d'un octet pour le calcul du prochain code
        while(sizeImgByte < sizeCode) {
            pxl = fgetc(Image);
            i++;
            if(sizeBlock == 0xff && i == sizeBlock) {
                i = 0;
                sizeBlock = pxl;
                pxl = fgetc(Image);
            }
            imgByte = (pxl<<(sizeImgByte)) + bitExtracted(imgByte, sizeImgByte, 1);
            sizeImgByte += 8;
        }
        
        // extraction du code
        code = bitExtracted(imgByte, sizeCode, 1);
        
        //output1.printf(">%03x; %03x; %03x; ", code, preCode, index);
        
        //////////////////////
        // traitement image //
        //////////////////////
        
        if(code == clearCode) index = sizePalette;
        else if(code == endCode) break;
        else {
            
            // sauf si premier code : chargement du "pixel string" du precode
            if(preCode != clearCode){
                for(j=0; j < table[preCode][0];j++)
                    pxls[j] = table[preCode][j+1];
            }
            
            // si code connu :
            if(code < index) {
                
                for(int k=1; k <= table[code][0];k++) {
                    if(k==1) pxls[j] = table[code][k];

                    lcdPort.write((uint8_t) (palette[ table[code][k] ]>>8));
                    lcdPort.write((uint8_t) palette[ table[code][k] ]);
                }

                for(int k=1; k <= j+1; k++)
                    table[index][k] = pxls[k-1];

                table[index][0] = j+1;
                
                
            } else {
                
                // add new stream pixel in palette and display
                pxl = pxls[0];
                pxls[j] = pxl;
                for(int k=1; k <= j+1; k++) {
                    table[index][k] = pxls[k-1];

                    lcdPort.write((uint8_t) (palette[ pxls[k-1] ]>>8));
                    lcdPort.write((uint8_t) palette[ pxls[k-1] ]);
                    
                }
                table[index][0] = j+1;
            }
            
            // clear buffer string
            for(int k=0; k < 256; k++) pxls[k] = 0;
        }
        
        preCode = code;
        
        // to compute next code
        sizeImgByte -= sizeCode;
        imgByte = bitExtracted(imgByte, sizeImgByte, sizeCode+1);
        
        if( index == (pow(2, sizeCode*1.0)-1)) sizeCode++;
        
        index++;
    }
    
    _cs = 1;
 
}

void ST7735::DrawANI(int offsetx, int offsety, FILE *Anim, int frame) {
    
    int i;
    char buf[32];
    char sizeStr[12];
    char *valExit = "error";
    char namefile[32];
    int size, offset = 0;
    
    sprintf(namefile, "%d.bmp", frame); 
    
    while(!feof(Anim)) {
        for(i=0; i <= strlen(namefile); i++) buf[i] = fgetc(Anim);
        
        if(!strcmp(buf, namefile)) {
            valExit = "";
            break;
        }
        
        fseek(Anim , 124+offset , SEEK_SET);
        for(i=0; i < 12; i++) sizeStr[i] = fgetc(Anim);
        //printf("size string: %s; octal: %d; dec.: %x\n", sizeStr, atoi(sizeStr), oct2dec(atoi(sizeStr)) );
        
        size = oct2dec(atoi(sizeStr));
        if(offset % 512 != 0) offset += (size/512 + 2)*512;
        else offset += (size/512 + 1)*512;
        
        fseek(Anim , offset , SEEK_SET);  
    }
    
    if(valExit == "error") {
        printf("file \"%s\" not find in tar\n", namefile); 
        //return NULL;
    } else {
        fseek(Anim, offset+512 , SEEK_SET);
        
        DrawBMP(offsetx, offsety, Anim);
        
        //return 1;
    }
    
    
}

void ST7735::scrollBmp(uint8_t top_fix_height, uint8_t bottom_fix_height, int _scroll_direction, int *bitmap) {
    unsigned int i, startOffset, endOffset;// line = 0; 
    float a, b;
    
    a = 7.7142*pow( 162.0-(top_fix_height+bottom_fix_height), -2.0 );
    b = 8.0717*pow( 162.0-(top_fix_height+bottom_fix_height), -1.0 );
     
    if(!_orientation) _scroll_direction = !_scroll_direction;
    
    if((_scroll_direction && _orientation == 2) || (!_scroll_direction && !_orientation)) {
        for (i = 0; i < 3248; i++) *bitmap++;
    }
    
    if((!_scroll_direction && _orientation == 2) || (_scroll_direction && !_orientation)) {
        startOffset = top_fix_height;
        endOffset = bottom_fix_height+1;
    } else {
        startOffset = bottom_fix_height;
        endOffset = top_fix_height+1;
    }
    
    if((_scroll_direction && _orientation == 2) || (!_scroll_direction && !_orientation)) setScrollDefinition(bottom_fix_height, top_fix_height+1, _scroll_direction);
    else setScrollDefinition(top_fix_height+1, bottom_fix_height, _scroll_direction);
    
    for( int scroll = 1+startOffset; scroll <= 162-endOffset ; scroll++){
      
        VerticalScroll(scroll);
        
        if(scroll >=(53) && scroll <=109) {
            if(_orientation == 0){
                if(_scroll_direction) {
                    drawFastHLine(0, 162-(scroll)-2, 80, ST7735_BLACK);
                    
                    for (i = 0; i < 57; i++) {
                        drawPixel(57+11-i, 162-(scroll)-2, *bitmap);
                        *bitmap++;
                    }
                } else {
                    drawFastHLine(0, scroll-2, 80, ST7735_BLACK);
                
                    for (i = 0; i < 57; i++) { 
                        drawPixel(12+i, scroll-2, *bitmap);
                        *bitmap--;
                    }
                }
            } else if(_orientation == 2) {
                if(_scroll_direction) {
                    drawFastHLine(0, 162-scroll-1, 80, ST7735_BLACK);
                    
                    for (i = 0; i < 57; i++) {
                        drawPixel(57+10-i,162-scroll-1, *bitmap);
                        *bitmap--;
                    }
                } else {
                    drawFastHLine(0, scroll-1, 80, ST7735_BLACK);
                
                    for (i = 0; i < 57; i++) {   
                        drawPixel(11+i,scroll-1, *bitmap);
                        *bitmap++;
                    }
                }
            }
        }
        
        float tempo = (a*scroll*scroll)-b*scroll+2.9408;
        wait_ms(tempo);
        //wait(0.1);
    }
    if(!_orientation) setRotation(0);
    VerticalScroll(0);
}

int ST7735::oct2dec(int n) /* Function to convert octal to decimal */
{
    int decimal=0, i=0, rem;
    while (n!=0)
    {
        rem = n%10;
        n/=10;
        decimal += rem*pow(8.0,i);
        ++i;
    }
    return decimal;
}

int ST7735::bitExtracted(int number, int k, int p) 
{ 
    return (((1 << k) - 1) & (number >> (p - 1))); 
} 

void ST7735::makeTable(void) {
    remove("/sd/table.csv");
    remove("/sd/pointers.csv");
    FILE *table = fopen("/sd/table.csv", "w");
    FILE *pointers = fopen("/sd/pointers.csv", "w");
    fclose(table);
    fclose(pointers);
} 

uint16_t ST7735::readTable(unsigned int y, unsigned int x) {
    
    FILE *table = fopen("/sd/table.csv", "r");
    FILE *pointers = fopen("/sd/pointers.csv", "r");
    char out[3];
    char ptr[17];
    
    fseek(pointers , y*17 , SEEK_SET);
    //output1.printf("%d: ", y);
    for(int i=0; i < 17; i++) {
        ptr[i] = fgetc(pointers);
        //output1.printf("%c", ptr[i]);
    }   
    //output1.printf("\n%s>>%d\n", ptr, atoi(ptr));
    
    // utiliser pointers
    //for(int i=0; i < y;)
        //if(fgetc(table) == 0x0A) i++;
    fseek(table , atoi(ptr) , SEEK_SET);
        
    for(int i=0; i < x;)
        if(fgetc(table) == 0x3B) i++;
        
    out[0] = fgetc(table);
    out[1] = fgetc(table);
    out[2] = fgetc(table);
    
    //output1.printf("%s>>%x ", out, atoi(out));
    
    fclose(table);
    fclose(pointers);
    return atoi(out);
}

void ST7735::addrowTable(uint16_t *row) {
    
    int size;
    
    FILE *table = fopen("/sd/table.csv", "r"); 
    fseek(table , 0 , SEEK_END);
    size = ftell(table);
    fclose(table);
    
    table = fopen("/sd/table.csv", "a");
    for(int i = 0; i< row[0]+1; i++) 
        fprintf(table, "%03d;", row[i]);
    fprintf(table, "\n");
    
    FILE *pointers = fopen("/sd/pointers.csv", "a");
    fprintf(pointers, "%16d\n", size);

    fclose(table);
    fclose(pointers);
} 

void ST7735::init_star(STAR* star, int i)
    {
      /* randomly init stars, generate them around the center of the screen */
      
      star->xpos =  -10.0 + (20.0 * (rand()/(RAND_MAX+1.0)));
      star->ypos =  -10.0 + (20.0 * (rand()/(RAND_MAX+1.0)));
      
      star->xpos *= 3072.0; /*change viewpoint */
      star->ypos *= 3072.0;
    
      star->zpos =  i;
      star->speed =  2 + (int)(2.0 * (rand()/(RAND_MAX+1.0)));
    
      star->color = i*ST7735_WHITE >> 2; /*the closer to the viewer the brighter*/
    }

void ST7735::DrawStarfield(int centerx, int centery, int acc, int color, int NbrsStars, int duration)
{
    int i, j=0, tempx, tempy;  
    STAR stars[NbrsStars];
    Timer timerStars;
    
    timerStars.start();
    while(timerStars.read() < duration) {
        
      j++;
      
      // move and draw stars 
       
      for (i = 0; i < NbrsStars; i++) {
          tempx = (stars[i].xpos / stars[i].zpos) + centerx;
          tempy = (stars[i].ypos / stars[i].zpos) + centery;
          drawPixel(tempx,tempy, ST7735_BLACK);
          
            
          stars[i].zpos -= stars[i].speed*acc;
          
          if (stars[i].zpos <= 0) {
              init_star(stars + i, i + 1);
          }
    
          //compute 3D position
          tempx = (stars[i].xpos / stars[i].zpos) + centerx;
          tempy = (stars[i].ypos / stars[i].zpos) + centery;
    
          if (tempx < 0 || tempx > _width - 1 || tempy < 0 || tempy > _height - 1) { //check if a star leaves the screen
        
              init_star(stars + i, i + 1);
              continue;
          }
      
            if(color) drawPixel(tempx,tempy,color);
            else drawPixel(tempx,tempy,stars[i].color);
        
      }
    }
    
    timerStars.stop();
}

void ST7735::setScrollDefinition(uint8_t top_fix_height, uint8_t bottom_fix_height, int _scroll_direction){
  uint8_t scroll_height;
  scroll_height = _height - top_fix_height - bottom_fix_height;
  writecommand(ST7735_VSCRDEF);
  writedata(0x00);
  writedata(top_fix_height);
  writedata(0x00);
  writedata(scroll_height);
  writedata(0x00);
  writedata(bottom_fix_height);
  writecommand(ST7735_MADCTL);
  if(_scroll_direction){
    if(_tft_type == 0){
      writedata(0xD8);
    }
    if(_tft_type == 1){
      writedata(0xD0);
    }
    if(_tft_type == 2){
      writedata(0x18);
    }
  }
  else{
    if(_tft_type == 0){
      writedata(0xC8);
    }
    if(_tft_type == 1){
      writedata(0xC0);
    }
    if(_tft_type == 2){
      writedata(0x08);
    }
  }
}
void ST7735::VerticalScroll(uint8_t _vsp) {
  writecommand(ST7735_VSCRSADD);
  writedata(0x00);
  writedata(_vsp);
}

//void ST7735::read_area(unsigned int x, unsigned int y, unsigned int w, unsigned int h, unsigned char *buffer) {
void ST7735::read_area(unsigned int x, unsigned int y, unsigned int w, unsigned int h) {
    // BEWARE !
    // DOES NOT WORK CORRECTLY YET !!!
    int val;
    setAddrWindow2(x,y,w,h);
    writecommand(ST7735_RAMRD);  // write to RAM
    _cs = 0;
    _rs = 1;
    lcdPort.write(0);                                /* Dummy read 1                 */
    
    val   = lcdPort.write(0);                        /* Read D8..D15                 */
    val <<= 8;
    val  |= lcdPort.write(0);                        /* Read D0..D7                  */
    _cs = 1;                  
            output1.printf("%d ", val);
        
    
    
}


void ST7735::setAddrWindow2(uint8_t x0, uint8_t y0, uint8_t x1,
                                    uint8_t y1)
{

    writecommand(ST7735_CASET); // Column addr set
    writedata(0x00);
    writedata(x0+xstart);     // XSTART
    writedata(0x00);
    writedata(x1+xstart);     // XEND

    writecommand(ST7735_RASET); // Row addr set
    writedata(0x00);
    writedata(y0+ystart);     // YSTART
    writedata(0x00);
    writedata(y1+ystart);     // YEND

    //writecommand(ST7735_RAMWR); // write to RAM
}


