/********************************************************************************************

   Filename: MCP9803.cpp

   Original Author: Steven Kay

   Development Group: Autonomous Systems Group, RAL Space

   Original Creation Date: April 2017

   Description: <Desc>

   Revision History: Version 1.0 - Initial Release

 *********************************************************************************************/

#include "MCP9803.h"

MCP9803::MCP9803(PinName sda, PinName scl, int Address, int frequency)
{
    _I2C = new I2C(sda,scl);
   
    _I2C -> frequency(frequency);

    _I2C -> start();
    _I2C -> stop();
    
    chipAddress = Address;
    
    inBuffer = (char *)malloc(1);
}



int MCP9803::ConfigSensor(  int shutdown, int comp_int, int alert_polarity,
                            int fault_guide, int adc_res, int one_shot)
{
    // Assign passed values to struct members
    CONFIG_REG_VALUE.CONFIG_BITS.SHUTDOWN_BIT = shutdown;
    CONFIG_REG_VALUE.CONFIG_BITS.COMP_INT_BIT = comp_int;
    CONFIG_REG_VALUE.CONFIG_BITS.ALERT_POLARITY_BIT = alert_polarity;
    CONFIG_REG_VALUE.CONFIG_BITS.FAULT_QUEUE = fault_guide;
    CONFIG_REG_VALUE.CONFIG_BITS.ADC_RES = adc_res;
    CONFIG_REG_VALUE.CONFIG_BITS.ONE_SHOT = one_shot;

    // Generate Command to send to MCP9803
    // <Pointer to Config Register, CONFIG Register Value>
    char dataOut[] = {CONFIG_REG_POINT, CONFIG_REG_VALUE.CONFIG_VALUE}; 
    
    // Write Bytes to MCP9803
    int writeReturn = I2C_Write(dataOut,CONFIG_CMD_LENGTH);
    
    // Check on return status of the Write command
    if(writeReturn != 0)
    {
        // Failure
        return FAILURE;     
    }
    else
    {
        // Success
        return SUCCESS;   
    }
}


int MCP9803::RawTempValue()
{
    // Set MCP9803 Register Pointer to point to Temperature Register
    I2C_Write((char *)TEMP_REG_POINT,1);
    
    // Read TEMP_DATA_LENGTH bytes into temp buffer
    char *tempInBuffer = I2C_Read(TEMP_DATA_LENGTH);
    
    // Convert tempInBuffer to rawData
    // Shift first Byte (MSB) up by 8, OR with second Byte (LSB)
    // Then shift Result down by 4
    int rawData = ((*(tempInBuffer)<<8|*(tempInBuffer+1))>>4);

    // 
    return rawData;
}



float MCP9803::FormattedTempValue(int format)
{
    // Get Raw Temp Value
    int tempData = RawTempValue();
    
    if(tempData == READ_TEMP_FAIL_VALUE)
    {
        return READ_TEMP_FAIL_ERROR;
    }
    
    // Check and convert if 2's Complement
    if(tempData > 2047)
    {
        tempData -= 4096;
    }
    
    // Return formatted data based on format
    switch(format)
    {
        // Return celcius value of temperature
        case CELCIUS:
            return (RAW_TO_C*tempData);

        // Return farenheit value of temperature
        case FARENHEIT:
            return ((RAW_TO_C*tempData)*C_F_1)+C_F_2;

        // Return Kelvin value of temperature
        case KELVIN:
            return ((RAW_TO_C*tempData)+C_TO_F);
 
        // If input is invalid, return value outwith range
        // of other format options
        default:
            return FORMAT_FAIL;
    }
}



int MCP9803::I2C_Write(char *dataOut,int dataLen)
{    
    // Define Char Buffer of length dataLen
    char outBuffer[dataLen];
    
    // Populate Buffer with dataOut
    for(int i=0 ; i<dataLen ; i++)
    {
        outBuffer[i] = *(dataOut+i);
    }
    
    // Write out to I2C Bus
    int sendStatus = _I2C -> write(chipAddress,outBuffer,dataLen,0);
    
    // Return success value of the Write
    // Returns 0 for Success, 1 for failure
    return sendStatus;
}



char *MCP9803::I2C_Read(int dataLen)
{
    // Reallocate memory to dataLen length expected for inBuffer
    setBufferSize(dataLen);
    
    // Read from the I2C Bus
    int receiveStatus = _I2C -> read(chipAddress,inBuffer,dataLen,0);
   
   // Check success value
   // If receiveStatus != 0, indicates failure
    if(receiveStatus != 0)
    {
        // Generate unique command to indicate failure, based on number of 
        // expected incoming bytes
        for(int i = 0; i < dataLen; i++)
        {
            // For even indexes of inBuffer
            if(i % 2 == 0)
            {
                inBuffer[i] = FAIL_EVEN_VALUE;
            }
            // For odd indexes of inBuffer
            else
            {
                inBuffer[i] = FAIL_ODD_VALUE;
            }
        }
        // Return unique error command
        return inBuffer;
    }
    else
    {
        // Return Buffer contents based on successful read
        return inBuffer;
    }
}
    
void MCP9803::setBufferSize(int dataLen)
{
    // Reallocate memory for the inBuffer to account for differing lengths of 
    // data which can be read in
    inBuffer = (char *)realloc(inBuffer, dataLen);
}

/********************************************************************************************

   Method: LinearMotor::LinearMotor

   Description: Class initialiser sets up the Linear Motor's pins

   Method Visibility: Public

   Input Arguments
      Name               Type              Description
      -----------------------------------------------------------------------------------------
      pwmPin             Int               The integer number of the connected hardware pin for
                                           the PWM output to the linear motor

      sensorPin          Int               The integer number of the connected hardware pin for
                                           the Analog Sensor feedback from the Linear Motor

   Output Arguments
      ----------------------------------------------------------------------------------------
      Name               Type              Description
      N/A

 *********************************************************************************************/