#include "mbed.h"
#include "TextLCD.h"

Serial pc(USBTX, USBRX);
TextLCD lcd(p18, p19, p20, p21, p22, p23, p24);
InterruptIn M2counter(p9);                      // only active when actuators running (hall effect input)
InterruptIn M1counter(p8);                      // only active when actuators running (hall effect input)
InterruptIn cpr(p14);                           // push button that interrupts any process
InterruptIn sense(p7);                          // opto sensor that interrupts any process after activation
BusIn switches (p10,p11,p12,p13);               // 12 push button in's through a diode matrix giving a 4 bit binary code for each button
AnalogIn h_pot (p16);                           // softpot input for actuator 2
AnalogIn l_pot (p15);                           // softpot input for actuator 1
BusOut act2 (p29,p28);                          // two relays that control actuator 2 direction
BusOut act1 (p25,p26);                          // two relays that control actuator 1 direction
DigitalOut fet2 (p30);                          // fet for actuator 2 switched on after relay direction set
DigitalOut fet1 (p27);                          // fet for actuator 1 switched on after relay direction set
DigitalOut alarm (p17);                         // alarm o/p actvated when cpr pushed or sensor in tripped

Timer timer1;
Timer timer2;
Timer timer3;

int M2direction;                            // actuator 2 direction flag
int M1direction;                            // actuator 2 direction flag
int bus_state;                              // switches BusIn is read then stored in this reg
int M2count;                                // actuator 2 counter reg
int M1count;                                // actuator 1 counter reg
int M2_old;                                 // used to store first sample of counter reg for act' 2
int M2_new;                                 // used to store second sample of counter reg for act' 2
int M1_old;                                 // used to store second sample of counter reg for act' 1
int M1_new;                                 // used to store second sample of counter reg for act' 1
int result;       // old and new values (above) are compared by subtraction and when result = 0 act' limit switches have been met
int lock;                                   // flag used to lock keypad
int senseflag;                              // flag to say when = 0 sensor off and when = 1 sensor on
int abs_M2count;                            // absolute values of counter reg's
int abs_M1count;                            // absolute values of counter reg's

// the following reg's are used to evaluate the position of the act's relative to the target position

int h_target;                               // temp reg used in most routines
int fh_target;                              //
int h_high;
int fh_high;
int h_low;
int fh_low;
int l_target;
int fl_target;
int l_high;
int fl_high;
int l_low;
int fl_low;
// reg's to store pot readings in as reading
float hpot;
float lpot;
int result_l;
int result_h;
int alarmflag;
int F2_flag;
int M1flag;
int M2flag;

//_______________Interrupts___________________________________


void herga_splash (void) {              // main menu screen
    lcd.cls();
    lcd.locate(0,0);
    lcd.printf("LEGS|  HERGA  |HEAD");
    lcd.locate(4,1);
    lcd.printf("| CONTROL |");
    lcd.locate(4,2);
    lcd.printf("| SYSTEMS |");
    lcd.locate(1,3);
    lcd.printf("PATIENT DETECT OFF");
}

//---------------------------------

void sense_int () {                     // patient detect sensor tripped
    alarmflag = 1;                      // set flag to exit RTZ corectly if sensor interrupts it
    fet1 = 0;                           // switch motors off
    fet2 = 0;
    wait(0.05);                         // wait for induced motor current to dissipate
    act2 = 0x00;                        // switch relays off (break motor)
    act1 = 0x00;

    lcd.cls();                          // set up alarm screen
    lcd.locate(2,1);
    lcd.printf("MOVEMENT DETECTED");
    lcd.locate(2,2);
    lcd.printf("PRESS ALARM RESET");

    while (1) {                         // sit in alarm loop
        alarm = 1;
        wait (0.05);
        alarm = 0;
        wait (0.1);
        alarm = 1;
        wait (0.1);
        alarm = 0;
        wait (0.5);
        if (switches == 0x08) {         // has the alarm reset button been pressed
            break;
        }
    }
    sense.rise(NULL);                   // turn interrupt off (turned on when button pressed)
    herga_splash();                     // refresh main menu screen
    lock = 0;                           // unlock keypad
}

//------------------------------------

void cpr_int () {                           // CPR button pressed
    wait(0.1);                              // wait incase spike
    if (cpr == 1) {                         // if button still pressed (ie not a spike) then continue
        alarmflag = 1;                      // set flag to exit RTZ corectly if CPR interrupts it
        fet1 = 0;                           // switch off motors
        fet2 = 0;
        wait(0.05);                         // wait for the induced motor current
        act2 = 0x00;                        // switch relays off (break motor)
        act1 = 0x00;
        lcd.cls();                          // set up cpr alarm screen
        lcd.locate(3,1);
        lcd.printf("CPR EMERGENCY");
        lcd.locate(1,2);
        lcd.printf("PRESS ALARM RESET");
        while (1) {                         // alarm loop
            alarm = 1;
            wait (0.5);
            alarm = 0;
            wait (0.5);
            if (switches == 0x08) {         // if alarm reset button pressed...
                break;                      //     .. exit while loop
            }
        }
        if (lock ==1) {                     // if keypad locked ...
            lock = 0;                           // ....unlock keypad
            lcd.locate(0,0);
            lcd.printf("                    ");
            lcd.locate(0,1);
            lcd.printf("  KEYPAD UNLOCKED   ");
            lcd.locate(0,2);
            lcd.printf("                    ");
            wait(1);
        }
        herga_splash();                     // refresh main screen
    }
}

//-----------------------------------

void M2countup () {                         // counter for actuator 2 hall effect sensor
    if (M2direction == 1) {                 // check direction flag, set before motor started
        M2count++;                          // count up
        abs_M2count = abs(M2count);         // counter register absoluted
    }
    if (M2direction == 0) {
        M2count--;
        abs_M2count = abs(M2count);
    }
}

void M1countup () {                         // counter for actuator 2
    if (M1direction == 1) {                 // same as above
        M1count++;
        abs_M1count = abs(M1count);
    }
    if (M1direction == 0) {
        M1count--;
        abs_M1count = abs(M1count);
    }
}

//----------------------------------------

void rtz (void) {                           // returns each actuator to limits and zero's counters
    M1direction = 1;                        // set actuator direction
    M2direction = 1;                        // set actuator direction
    lcd.locate (7,0);                       // rtz screen
    lcd.printf ("     ");
    lcd.locate (0,1);
    lcd.printf ("DOWN|   RTZ   | UP ");
    lcd.locate (6,2);
    lcd.printf ("       ");                 // "   "
    act2 = 0x02;                            // set M2 relays for up
    act1 = 0x01;                            // set M1 relays for down
    wait(0.015);                            // wait for relay contacts to close
    fet2 = 1;                               // switch motors on
    fet1 = 1;                               //   "          "
    wait (0.1);                             // wait for a couple of counts

// sit in the following while loop until no more pulses are received from the hall effect sensor

    while (1) {
        pc.printf("abs_M2count  = %d\n",abs_M2count);
        M2_old = abs_M2count;               // take first count samples
        M1_old = abs_M1count;               //  "       "
        wait (0.1);                         // wait for a few more
        M2_new = abs_M2count;               // take second samples
        M1_new = abs_M1count;
        if ((M2_old - M2_new == 0)&&(M1_old - M1_new == 0)) {       // when both actuators have reached the limits
            break;                                                  // exit while loop
        }
    }
    fet1 = 0;                               // turn off motors
    fet2 = 0;
    wait (0.05);                            // wait for induced current to dissipate
    act1 = 0;                               // turn relays off
    act2 = 0;
    wait(0.05);
    lcd.locate(7,0);                        // refresh main menu screen
    lcd.printf("HERGA");
    lcd.locate(0,1);
    lcd.printf("    | CONTROL |    ");
    lcd.locate(6,2);
    lcd.printf("SYSTEMS");
    if (alarmflag == 1) {                   // if RTZ has been interrupted by sensor or cpr alarm button
        alarmflag = 0;
        return;
    }
    M2count = 0;                            // if RTZ not interrupted exit RTZ....
    M1count = 0;                            //       ....having zero'd count registers
    abs_M2count = abs(M2count);
    abs_M1count = abs(M1count);
}

//----------------------------------------
void head_up () {
    M2direction = 1;                            // count direction flag
    lcd.locate(15,1);                           // pos cursor
    lcd.printf(" UP ");                         // disp UP
    act2 = 0x02;                                // set relay direction
    wait(0.015);                                // wait for relay contacts to close
    fet2 = 1;                                   // switch FET on
}
//-----------------------------------------
void head_down () {
    M2direction = 0;                            // count direction flag
    lcd.locate(15,1);                           // pos cursor
    lcd.printf("DOWN");                         // disp DOWN
    act2 = 0x01;                                // set relay direction
    wait(0.015);                                // wait for relay contacts to close
    fet2 = 1;                                   // switch FET on
}
//-----------------------------------------
void head_off () {
    fet2 = 0;                                   // turn FET off
    wait(0.05);                                 // wait for induced motor current to dissipate
    act2 = 0x00;                                // break motor
    lcd.locate(15,1);                           // pos cursor
    lcd.printf("    ");                         // clear UP from screen
}
//------------------------------------------
void legs_up () {
    M1direction = 0;                            // count direction flag
    lcd.locate(0,1);                            // pos cursor
    lcd.printf(" UP ");                         // disp UP
    act1 = 0x02;                                // set relay direction
    wait(0.015);                                // wait for relay contacts to close
    fet1 = 1;                                   // turn FET on
}
//------------------------------------------
void legs_down () {
    M1direction = 1;                            // count direction flag
    lcd.locate(0,1);                            // pos cursor
    lcd.printf("DOWN");                         // disp DOWN
    act1 = 0x01;                                // set relay direction
    wait(0.015);                                // wait for relay contacts to close
    fet1 = 1;                                   // turn FET on
}
//------------------------------------------
void legs_off () {
    fet1 = 0;                                   // turn FET off
    wait(0.05);                                 // wait for induced motor current to disipate
    act1 = 0x00;                                // break motor
    lcd.locate(0,1);                            // clear "UP" from screen
    lcd.printf("    ");
}
//------------------------------------------
void manual_head_run () {
    while (1) {
        pc.printf("abs_M2count  = %d\n",abs_M2count);
        M2_old = abs_M2count;                   // take first sample
        wait (0.1);                             // allow some counts
        M2_new = abs_M2count;                   // take second sample
        if (M2_old - M2_new == 0) {             // sub 2nd sample from 1st, if ans = 0 then limit switch is met...
            if (M2direction == 1) {
                M2count = 0;                    //  .... therefore zero the count reg....
            }
            abs_M2count = abs(M2count);         //      ... and the abs_M2count reg
            break;                              // exit
        }
        if (M2direction == 1 && switches != 0x02) {                 // if button released....
            break;                              // exit
        }
         if (M2direction == 0 && switches != 0x01) {                 // if button released....
            break;                              // exit
        }
    }
}
//-----------------------------------------
void manual_legs_run () {
    while (1) {
        M1_old = abs_M1count;                           // take first sample
        wait (0.1);                                     // allow some counts
        M1_new = abs_M1count;                           // take second sample
        if (M1_old - M1_new == 0) {                     // sub 2nd sample from 1st, if ans = 0 then limit switch is met...
            if (M1direction == 1) {
                M1count = 0;                            //  .... therefore zero the count reg....
            }
            abs_M1count = abs(M1count);                 //      ... and the abs_M1count reg
            break;                                      //    .... therefore exit
        }
        if (M1direction == 1 && switches != 0x0A) {     // if button released....
            break;                                      // exit
        }
         if (M1direction == 0 && switches != 0x03) {    // if button released....
            break;                                      // exit
        }       
    }
}

//-----------------------------------------

void head_autorun () {
    while (1) {
        M2_old = abs_M2count;                           // take first sample
        wait (0.1);                                     // allow some counts
        M2_new = abs_M2count;                           // take second sample
        result = M2_old - M2_new;
        pc.printf("abs_M2count  = %d\n",abs_M2count);

        if ((abs_M2count >= h_low )&&(abs_M2count <= h_high)) {
            break;
        } else if (alarmflag ==1) {
            alarmflag = 0;
            break;
        } else if (result == 0) {                       // sub 2nd sample from 1st, if ans = 0 then limit switch is met...
            if (M2direction == 1) {
                M2count = 0;                            //  .... therefore zero the count reg....
            }
            abs_M2count = abs(M2count);                 //      ... and the abs_M1count reg
            break;
        } else if (switches == 0x01 || switches ==0x02) {
            break;
        }
    }
}

//-----------------------------------------

void legs_autorun () {
    while (1) {
        M1_old = abs_M1count;                           // take first sample
        wait (0.1);                                     // allow some counts
        M1_new = abs_M1count;                           // take second sample
        result = M1_old - M1_new;
        if ((abs_M1count >= l_low )&&(abs_M1count <= l_high)) {
            break;
        } else if (alarmflag ==1) {
            alarmflag = 0;
            break;
        } else if (result == 0) {
            if (M1direction == 1) {
                M1count = 0;                            //  .... therefore zero the count reg....
            }
            abs_M1count = M1count;                      //      ... and the abs_M1count reg
            break;                                      //    .... therefore exit
        } else if (switches == 0x03 || switches == 0x0A) {
            break;
        }
    }
}

//-----------------------MAIN-------------------------------

int main() {

    herga_splash();                     // display main screen
    M2counter.rise(&M2countup);         // set up actuator 2 counter interrupt
    M1counter.rise(&M1countup);         // set up actuator 1 counter interrupt
    rtz();                              // zero both actuators
    timer1.start();                     // start timer for switch data bus read
    wait_ms(10);                        // delay between timer starts, probably not necessary !!
    timer2.start();                     // start timer for potentiometer 1 read
    wait_ms(10);                        // again prob not necessary !!
    timer3.start();                     // start timer for potentiometer 2 read
    cpr.rise(&cpr_int);                 // set up cpr button interrupt
    M2count = 0;                        // zero counter registers
    M1count = 0;
    lock = 0;                           // unlock the keypad functions
    sense.mode(PullNone);               // opto sensor has external pull up

    while (1) {

        if (timer1.read_ms()>=5) {      // every 5 ms the state of the switch input bus is read
            if (lock == 0) {                           // when locked do not enter to check switch bus

                if (switches == 0x09) {                // if F2 (switch bus code 0x09) is pressed....
                    fh_target = abs_M2count;           //      .... store current actuator positions
                    fh_low = fh_target - 2;            // create lower count limit
                    fh_high = fh_target + 2;           // create upper count limit

                    fl_target = abs_M1count;
                    fl_low = fl_target - 2;
                    fl_high = fl_target + 2;

                    //pc.printf("fh_target      = %d\n",fh_target);         //for debugging
                    //pc.printf("fh_low         = %d\n",fh_low);
                    //pc.printf("fh_high        = %d\n",fh_high);

                    lcd.locate(0,0);                        // refresh main menu screen
                    lcd.printf("       SAVING       ");
                    lcd.locate(0,1);
                    lcd.printf("        USER        ");
                    lcd.locate(0,2);
                    lcd.printf("       PRESET       ");

                    while (switches != 0x00) {}             // wait till F2 button released
                    wait(1);
                    lcd.locate(0,0);
                    lcd.printf("LEGS|  HERGA  |HEAD");      // refresh main screen
                    lcd.locate(0,1);
                    lcd.printf("    | CONTROL |    ");
                    lcd.locate(0,2);
                    lcd.printf("    | SYSTEMS |    ");

                    F2_flag = 1;                            // If this 'if' has been entered do not check the switch bus again
                }
            }
            if (F2_flag == 0) {
                bus_state = switches;
            }
            F2_flag = 0;                                    // clear flag for next time around
            timer1.reset();
        }

// the pots used in this code are softpots embedded into a label and therefore act in a momentary way,
// ie when you press the pot you get a voltage level out of the wiper relevant to the position
// along the pots length and when you release it the output returns to 0V!
// search for spectrasymbol in google for more info

        if (timer2.read_ms()>=50) {                         // every 50ms the head pot (act 2) is checked
            hpot = h_pot;                                   // grab the pot value
            if (hpot >0.001) {                              // has it been pressed?
                wait (0.05);                                // if yes wait incase it was noise
                hpot = h_pot;                               // grab the pot value again
                if (hpot<0) {                               // if below 0 do nothing (i know.. not needed!)
                    bus_state = 0;
                } else if (hpot>0.02 && hpot<0.1) {         // if between these values
                    bus_state = 0x0F;                       // set bus_state for case statement later
                    h_target = 1200;                        // set counter target
                    h_high = 1202;                          // set upper limit
                    h_low = 1198;                           // set lower limit
                    // limits help in case counter encounters some noise and
                    // picks up a spurious rising edge
                } else if (hpot>0.15 && hpot<0.2) {
                    bus_state = 0x0F;
                    h_target = 900;
                    h_high = 902;
                    h_low = 898;
                } else if (hpot>0.27 && hpot<0.37) {
                    bus_state = 0x0F;
                    h_target = 600;
                    h_high = 602;
                    h_low = 598;
                } else if (hpot>0.45 && hpot<0.55) {
                    bus_state = 0x0F;
                    h_target = 300;
                    h_high = 302;
                    h_low = 298;
                } else if (hpot>0.65) {
                    bus_state = 0x0F;
                    h_target = 0;
                    h_high = -2;
                    h_low = 2;
                }
            }
            timer2.reset();
        }

        if (timer3.read_ms()>=50) {                     // as above but for legs pot (act 1)
            lpot = l_pot;
            if (lpot >0.001) {
                wait (0.05);
                lpot = l_pot;
                if (lpot<0) {
                    bus_state = 0;
                } else if (lpot>0.02 && lpot<0.1) {
                    bus_state = 0x0E;
                    l_target = 0;
                    l_high = -2;
                    l_low = 2;
                } else if (lpot>0.15 && lpot<0.2) {
                    bus_state = 0x0E;
                    l_target = 300;
                    l_high = 302;
                    l_low = 298;
                } else if (lpot>0.27 && lpot<0.37) {
                    bus_state = 0x0E;
                    l_target = 600;
                    l_high = 602;
                    l_low = 598;
                } else if (lpot>0.45 && lpot<0.55) {
                    bus_state = 0x0E;
                    l_target = 900;
                    l_high = 902;
                    l_low = 898;
                } else if (lpot>0.65) {
                    bus_state = 0x0E;
                    l_target = 1200;
                    l_high = 1202;
                    l_low = 1198;
                }
            }
            timer3.reset();
        }

        if (bus_state == 0x0C) {                                // toggle lock to block button inputs
            lock=!lock;
            if (lock == 1) {                                    //  locked so inform user
                lcd.locate(0,0);
                lcd.printf("                    ");
                lcd.locate(0,1);
                lcd.printf("   KEYPAD LOCKED    ");
                lcd.locate(0,2);
                lcd.printf("                    ");
            } else if (lock == 0) {                             // unlocked so return to main screen
                herga_splash();
            }
            while (switches != 0x00) {}
        }


        if (lock == 0) {                                        // when unlocked check all other buttons
            switch (bus_state) {
                case 0x02:                                      // head up button pressed
                    head_up();                                  // set direction of actuator
                    wait(0.1);                                  // wait to accumulate some pulses (needed if at 0)
                    manual_head_run();                          // move actuator
                    head_off();                                 // stop actuator
                    while (switches != 0x00) {}                 // stop any other button having an effect
                    break;
                case 0x01:                                      // head down button pressed
                    head_down();                                // set direction of actuator
                    wait(0.1);                                  // wait to accumulate some pulses (needed if at 0)
                    manual_head_run();                          // move actuator
                    head_off();                                 // stop actuator
                    while (switches != 0x00) {}                 // stop any other button having an effect
                    break;
                case 0x03:                                      // legs up button pressed
                    legs_up();                                  // set direction of actuator
                    wait(0.1);                                  // wait to accumulate some pulses (needed if at 0)                                  //
                    manual_legs_run();                          // move actuator
                    legs_off();                                 // stop actuator
                    while (switches != 0x00) {}                 // stop any other button having an effect
                    break;
                case 0x0A:                                      // legs down button pressed
                    legs_down();                                // set direction of actuator
                    wait(0.1);                                  // wait to accumulate some pulses (needed if at 0)
                    manual_legs_run();                          // move actuator
                    legs_off();                                 // stop actuator
                    while (switches != 0x00) {}                 // stop any other button having an effect
                    break;
                case 0x06:                                      // return to zero position
                    rtz();
                    break;
//***** HEAD AUTO
                case 0x0F:                                      // had an input on head pot
                    result = abs_M2count - h_target;            // decide on motor direction
                    if (result > 0) {                           // result is positive so set up move up
                        head_up();
                    } else if (result < 0 ) {
                        head_down();                            // result is negative so set up move down
                    }
                    head_autorun();                             // move act 2 to target position
                    head_off();
                    while (switches != 0x00) {}                 // sit here until button released
                    break;
//****** LEGS AUTO
                case 0x0E:
                    result = abs_M1count - l_target;            // do calculation to determine direction
                    if (result < 0) {                           // if result is negative set up to move up
                        legs_up();
                    } else if (result > 0 ) {                   // if positive result set up to move down
                        legs_down();
                    }
                    legs_autorun();                             // move act 1 to target position
                    legs_off();
                    while (switches != 0x00) {}
                    break;
//****** switch sensor on/off
                case 0x04 :                                     // sensor button has been pressed so toggle sensor arm/disarm
                    senseflag = !senseflag;                     // invert senseflag
                    if (senseflag == 0) {                       // if senseflag = 0
                        lcd.locate(17,3);                       // display OFF (PATIENT DETECT "OFF")
                        lcd.printf("FF ");
                        sense.rise(NULL);                       // ignore patient detect sensor
                    } else if (senseflag == 1) {                // if senseflag = 1
                        lcd.locate(17,3);                       // display OFF (PATIENT DETECT "ON")
                        lcd.printf("N  ");
                        sense.rise(&sense_int);                 // arm patient detect sensor
                    }
                    while (switches != 0x00) {}
                    break;
//******* F1
                case 0x05:                                       // F1 pressed so move both actuators to stored targets
                    result_h = abs_M2count - fh_target;
                    if (result_h > 0) {
                        M2direction = 1;
                        lcd.locate(15,1);
                        lcd.printf(" UP ");
                        act2 = 0x02;
                    } else if (result_h < 0 ) {
                        M2direction = 0;
                        lcd.locate(15,1);
                        lcd.printf("DOWN");
                        act2 = 0x01;
                    }
                    result_l = abs_M1count - fl_target;
                    if (result_l < 0) {                         // if result is negative actuator must move up
                        M1direction = 0;                        // decrement counter flag
                        lcd.locate(0,1);
                        lcd.printf(" UP ");
                        act1 = 0x02;
                    } else if (result_l > 0 ) {                 // if the result of the first calc was positive
                        pc.printf("M1 down\n");
                        M1direction = 1;                        // the acutator must move down
                        lcd.locate(0,1);
                        act1 = 0x01;
                    }
                    fet1 = 1;
                    fet2 = 1;
                    while (1) {
                        if (abs_M1count >= fl_low && abs_M1count <= fl_high) {
                            fet1 = 0;
                            wait(0.05);
                            act1 = 0x00;
                            M1flag =1;
                        }
                        if (abs_M2count >= fh_low &&abs_M2count <= fh_high) {
                            fet2 = 0;
                            wait(0.05);
                            act2 = 0x00;
                            M2flag =1;
                        }
                        M1_old = abs_M1count;                   // take first sample
                        M2_old = abs_M2count;
                        wait (0.1);                             // allow some counts
                        M1_new = abs_M1count;                   // take second sample
                        M2_new = abs_M2count;                   // take second sample
                        result_h = M2_old - M2_new;
                        result_l = M1_old - M1_new;
                        
                        if (M1flag == 1 && M2flag == 1) {
                            break;
                        } else if (alarmflag ==1) {
                            alarmflag = 0;
                            break;
                        } else if ((result_h == 0) && (result_l == 0)) {
                            if (M1count < 10) {
                                M1count = 0;                        //  .... therefore zero the count reg....
                            }
                            abs_M1count = M1count;                  //      ... and the abs_M1count reg
                            if (M2count <10) {
                                M2count = 0;                        //  .... therefore zero the count reg....
                            }
                            abs_M2count = M2count;                  //      ... and the abs_M1count reg
                            break;                                  //    .... therefore exit*/
                        } else if (switches == 0x01 || switches == 0x02 || switches == 0x03 || switches == 0x0A ) {
                            break;
                        }
                    }
                    M1flag = 0;
                    M2flag = 0;
                    fet1 = 0;
                    fet2 = 0;
                    wait(0.05);
                    act1 = 0x00;
                    act2 = 0x00;
                    lcd.locate(0,1);
                    lcd.printf("    ");
                    lcd.locate(15,1);
                    lcd.printf("    ");
                    while (switches !=0) {}
                    break;
            }

        }
    }
}












