/*
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mbed.h"
#include "USBHost.h"
#include "Utils.h"
#include "UsbStorage.h"

#include "EthernetNetIf.h"
#include "Dropbox.h"
#include "HTTPServer.h"
#include "HTTPServer.h"
#include "HttpHandlerSetup.h"
#include "HttpHandlerUsbBrowser.h"

// USB
Serial pc(USBTX, USBRX);
USBFileSystem fs;
int usbDevice = -1;

// NET
EthernetNetIf* eth;

// LED
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);

// Occurs on usb flash disk connection
int OnDiskInsert(int device)
{
    printf("\r\nOnDiskInsert\r\n");
    
    usbDevice = device;
    fs.SetDevice(usbDevice);
    
    return 0;
}

// Setup ethernet connection
bool setup_eth()
{
  printf("\r\nSetting up...\r\n");

  EthernetErr ethErr = eth->setup();
  if(ethErr)
  {
    printf("Error %d in setup.\n", ethErr);
    return false;
  }
  printf("\r\nSetup OK\r\n");
  
  return true;
}

// Setup HttpHandlers
bool setup_httpServer(HTTPServer& svr)
{
   svr.addHandler<HttpHandlerUsbBrowser>("/UsbBrowser");  
   svr.addHandler<HttpHandlerSetup>("/");
   svr.bind(80);

   printf("Http Server initialized\r\n");  
   
   return true;   
}

// Uploads a folder on usb flash-drive to dropbox
bool upload_files(char* username, char* password)
{
    DIR *d = opendir("/usb/DROPBOX");
    int totalFiles = -1;
    
    if (d)
    {
        char filenameSrc[64] = "/usb/DROPBOX/";
        char filenameDst[32] = "USBSYNC_";
        int c = 0;
        
        printf("Synchronize Start\n\r");
        if (!dropbox_syncTime())
        {
            printf("Failed to sync time");
            led3 = 1;
            return totalFiles;
        }
        if (!dropbox_getToken(username, password))
        {
            printf("Failed getting token\r\n");
            led3 = 1;
            return totalFiles;
        }
    
        struct dirent *p = readdir(d);
        while (p)
        {
            strcpy(filenameSrc + 13, p->d_name);
            strcpy(filenameDst + 8, p->d_name);
            printf("Uploading %s to %s\r\n", filenameSrc, filenameDst);
            
            dropbox_upload(filenameSrc, filenameDst);
            c++;
            p = readdir(d);
        }
        
        closedir(d);
        
        totalFiles = c;
        printf("Synchronizing finished (%d)\n\r", totalFiles);
        led2 = 1;
        
        return totalFiles;
    }
    else
    {
        led3 = 1;
        printf("Failed opening /usb/DROPBOX/\r\n");
        return totalFiles;
    }
}

int main()
{
    bool ethSetup = false, httpSetup = false;
    USBInit();    
    HTTPServer svr;
    
    int i = 0; // loop counter    
    led3 = 0; // clear error LED
    
    for (;;)
    {
        if (i++ % 500000 == 0) printf("looping %d\r\n", time(NULL));
        
        USBLoop();
        Net::poll();       
        
        // Setup ethernet
        if (!ethSetup)
        {
            eth = new EthernetNetIf();
            ethSetup = setup_eth();
            
            if (!ethSetup)
            {
                delete eth;
                eth = 0;
            }
            
            httpSetup = false;
        }
        
        // Setup http-server
        if (ethSetup && !httpSetup)
        {
           httpSetup = setup_httpServer(svr);
        }                

        // Check if Ethernet setup and USB-Disk connected
        if (usbDevice >= 0 && ethSetup)
        {
            usbDevice = -1;

            led1 = 1;
            led2 = 0;
            
            // Read username/password
            char username[32] = {0};
            char password[32] = {0};
            if (!ReadSettings(username, password))
            {
                printf("Failed opening /usb/DBSet.txt\r\n");
                led3 = 1;
                continue;
            }
            
            // Start synchronizing
            int totalFiles = upload_files(username, password);
            if (totalFiles >= 0)
            {
                //send_email(username);
            }

            // Update the log file
            FILE* logFile = fopen("/usb/dbLog.txt", "a+");
            if (logFile)
            {
                if (totalFiles >= 0)
                {
                    fprintf(logFile, "%d Synchronizing successfull %d files\r\n", time(NULL), totalFiles);
                }
                else
                {
                    fputs("Synchronizing failed\r\n", logFile);
                }
                
                fclose(logFile);
            }

            led1 = 0;

            //ClearPortPower();
        }
    }
}