/*
 * Glowing LED Library
 * Copyright (c) 2010 Sitti Amarittapark
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef SA_GLOWING_H
#define SA_GLOWING_H

#include "mbed.h"

/**  Glowing LED class, based on Ticker and PwmOut
 *
 *  Author: Sitti Amarittapark
 *
 *  Glow an LED from dark to bright continously until stop
 *
 *  Example:
 *  @code
 *  #include "Glowing.h"
 *  // Run full cycle in 2 seconds in
 *  // 500 steps and begins with step 100
 *  {
 *      Glowing *glow = new Glowing(LED1, 2.0, 500, 100);
 *      glow->Start();
 *      . . .
 *      glow->Stop();
 *  }
 *
 *  @endcode
 *
 */

class Glowing: public Ticker {

public:
    /** Select a PIN, cycle length in seconds, steps of change, offset in steps
     *
     * @param pin PwmOut pin to connect to
     * @param period cycle length in seconds
     * @param steps number of steps in a cycle, default to 100
     * @param offset number of steps offset from beginning, default to 0
     *        LED off. Apply the different offset values to multiple LEDs can
     *        create a cascading pattern.
     */
    Glowing(PinName pin, float period, int steps=100, int offset=0);
    /**
     * Stop the glowing cycle and delete the object
     */
    virtual ~Glowing();
    /**
     * Start the cycle with offset
     */
    void  Start(void);
    /**
     * Stop the cycle
     */
    void  Stop(void);

protected:
    PwmOut      *s_port;
    int         s_steps;
    int         s_step_offset;
    int         s_next_n;
    int         s_next_v;
    bool        s_running;

    float       c_step_time;
    float       c_step_gradient;

    void  next(void);
    int   scale(float v);
};

#endif // SA_GLOWING_H