/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ATCmdParser.h"
#include "FileHandle.h"
#include "onboard_modem_api.h"
#include "stdlib.h"

/* Definitions */
#define OUTPUT_ENTER_KEY  "\r"
#define AT_PARSER_BUFFER_SIZE   256
#define AT_PARSER_TIMEOUT       8*1000 // Milliseconds

/* The example program for the u-blox C030 boards. It sets up the Cellular Module then read out the IMSI of the eUICC/SIM.
 * The Uer LEDs (RGB) and User Button are utilised to show the activity on the board.
 * When GNSS has a location fix, the Red Timepulse LED, next to the User LEDs, blinks every 1 second.
 */

/* File handler */
FileHandle *fh;

/* AT Command  Parser handle */
ATCmdParser *at;

// User LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

// Ethernet socket LED
DigitalOut ledYellow(LED4,1);

// User Button
#ifdef TARGET_UBLOX_C027
// No user button on C027
InterruptIn userButton(NC);
#else
InterruptIn userButton(SW0);
#endif

// Delay between LED changes in second
volatile float delay = 0.5;

// To check if the user pressed the User Button or not
void threadBodyUserButtonCheck(void const *args)
{
    float delayToggle = delay;
    while (1) {
        if (userButton.read() == 1 ) {
            // User Button is pressed
            delay = 0.1;
            // Indicate with the Yellow LED on Ethernet socket
            ledYellow = 0;
        } else {
            // User button is released
            delay = 0.5;
            // Turn off the Yellow LED on Ethernet socket
            ledYellow = 1;
        }
    }
}

void sendATCommandAndPrintResponse(char* command)
{
    char buffer[505];//sa crtom
    for (int i=0; i<505; i++)buffer[i]=0;
    printf ("Sending following command:");
    printf(command);
    printf ("\r\n");

    if (!at->send(command)) {
        printf("failed!!!!\r\n");
        return;
    }
    printf ("RESPONSE:\r\n");
    at->read(buffer,500);
    printf(buffer);
    printf ("\r\n");
}
// Setup the modem
bool setup_modem()
{

    bool success = false;

    /* Initialize GPIO lines */
    onboard_modem_init();

    /* Give modem a little time to settle down */
    wait_ms(250);

    printf("Powering up the modem\r\n");
    onboard_modem_power_up();
#ifdef TARGET_UBLOX_C030_N211
    wait_ms(5000);
    sendATCommandAndPrintResponse("AT+CFUN=1");
#else
    wait_ms(500);
#endif

    // Set AT parser timeout to 1sec for AT OK check
    at->set_timeout(1000);

    printf("Checking for AT response from the modem\r\n");
    for (int retry_count = 0; !success && (retry_count < 20); retry_count++) {
        printf("...wait\r\n");
        // The modem tends to sends out some garbage during power up.
        at->flush();

        // AT OK talk to the modem
        if (at->send("AT")) {
            wait_ms(100);
            if (at->recv("OK")) {
                success = true;
            }
        }
    }

    // Increase the parser time to 8 sec
    at->set_timeout(8000);

    if (success) {
        printf("Configuring the modem\r\n");

#ifdef TARGET_UBLOX_C030_N211
        printf("TARGET IS N211");
        // Turn off modem echoing and turn on verbose responses
        success = at->send("AT+CMEE=1");
#else
        printf("TARGET IS NOT N211");
        // Set the final baud rate
        if (at->send("AT+IPR=%d", 115200) && at->recv("OK")) {
            // Need to wait for things to be sorted out on the modem side
            wait_ms(100);
            ((UARTSerial *)fh)->set_baud(115200);
        }

        // Turn off modem echoing and turn on verbose responses
        success = at->send("ATE0;+CMEE=2") && at->recv("OK") &&
                  // The following commands are best sent separately
                  // Turn off RTC/CTS handshaking
                  at->send("AT&K0") && at->recv("OK") &&
                  // Set DCD circuit(109), changes in accordance with
                  // the carrier detect status
                  at->send("AT&C1") && at->recv("OK") &&
                  // Set DTR circuit, we ignore the state change of DTR
                  at->send("AT&D0") && at->recv("OK");
#endif
    }

    return success;
}
/*
** Reading the modem IMEI and eUICC/SIM IMSI
**
*/
void connectNB(){
    
       
    sendATCommandAndPrintResponse("at+NCONFIG=\"AUTOCONNECT\",\"FALSE\"");
    sendATCommandAndPrintResponse("at+NCONFIG=\"CR_0354_0338_SCRAMBLING\",\"TRUE\"");


    sendATCommandAndPrintResponse("at+NCONFIG=\"CR_0859_SI_AVOID\",\"TRUE\"");
    sendATCommandAndPrintResponse("at+NCONFIG?");
    sendATCommandAndPrintResponse("at+cfun=0");
    sendATCommandAndPrintResponse("AT+CGDCONT=1, \"IP\",\"nb.inetd.gdsp\"");
    sendATCommandAndPrintResponse("at+cfun=1");

    sendATCommandAndPrintResponse("at+cimi");


    sendATCommandAndPrintResponse("at+cgatt=1");



    sendATCommandAndPrintResponse("at+cops=1,2,\"26202\"");
    wait_ms(5000);
  sendATCommandAndPrintResponse("at+cereg?");
    wait_ms(5000);
    sendATCommandAndPrintResponse("AT+CSQ");
    wait_ms(5000);
    sendATCommandAndPrintResponse("AT+COPS?");
    sendATCommandAndPrintResponse("AT+NBAND?");
    sendATCommandAndPrintResponse("AT+NBAND=20");
    sendATCommandAndPrintResponse("AT+NUESTATS");
    
    sendATCommandAndPrintResponse("AT+CGATT?");
    wait_ms(5000);
    sendATCommandAndPrintResponse("AT+CGPADDR");

}
void reverse(char *str)
{
    /* skip null */
    if (str == 0)
    {
        return;
    }

    /* skip empty string */
    if (*str == 0)
    {
        return;
    }

    /* get range */
    char *start = str;
    char *end = start + strlen(str) - 1; /* -1 for \0 */
    char temp;

    /* reverse */
    while (end > start)
    {
        /* swap */
        temp = *start;
        *start = *end;
        *end = temp;

        /* move */
        ++start;
        --end;
    }
}
  char* itoa(int num, char* str, int base) 
{ 
    int i = 0; 
    bool isNegative = false; 
  
    /* Handle 0 explicitely, otherwise empty string is printed for 0 */
    if (num == 0) 
    { 
        str[i++] = '0'; 
        str[i] = '\0'; 
        return str; 
    } 
  
    // In standard itoa(), negative numbers are handled only with  
    // base 10. Otherwise numbers are considered unsigned. 
    if (num < 0 && base == 10) 
    { 
        isNegative = true; 
        num = -num; 
    } 
  
    // Process individual digits 
    while (num != 0) 
    { 
        int rem = num % base; 
        str[i++] = (rem > 9)? (rem-10) + 'a' : rem + '0'; 
        num = num/base; 
    } 
  
    // If number is negative, append '-' 
    if (isNegative) 
        str[i++] = '-'; 
  
    str[i] = '\0'; // Append string terminator 
  
    // Reverse the string 
    reverse(str); 
  
    return str; 
} 

void sendUDPMessage(char* ip, int port, char* message){

char commandbuffer[100];
sprintf(commandbuffer,"AT+NSOCR=\"DGRAM\",17,%d",port);
sendATCommandAndPrintResponse(commandbuffer);
int ret=strlen(message);
char data[ret*2];
for (int i=0,j=0; i<ret; i++,j+=2){ 
    char res[2]; 
    itoa((message[i] & 0xFF), res, 16);
        if (res[1] == 0) {
            data[j] = 0x30; data[j+1] = res[0];
        }else {
            data[j] = res[0]; data[j + 1] = res[1];
        }
}

data[(ret * 2)] = 0;

sprintf(commandbuffer,"AT+NSOST=0,\"%s\",%d,%d,\"%s\"",ip,port,ret,data);
sendATCommandAndPrintResponse(commandbuffer);

}

int main()
{
    bool success = false;

    printf("\n\r\n\ru-blox C030 reading the modem IMEI and eUICC/SIM IMSI\n\r");
    printf("Initialising UART for modem communication");
    fh = new UARTSerial(MDMTXD, MDMRXD, 9600);
    printf("...DONE\r\n");

    // NOTE: if you are experiencing problems with the AT command
    // exchange, change the "false" below to "true" to get debug output
    // from the AT command parser
    printf("Initialising the modem AT command parser");
    at = new ATCmdParser(fh, OUTPUT_ENTER_KEY, AT_PARSER_BUFFER_SIZE,
                         AT_PARSER_TIMEOUT, false);
    printf("...DONE\r\n");

    printf("Initializing the modem\r\n");
    if (setup_modem()) {
        printf("...DONE\r\nThe modem powered up\r\n");
        char imei[25+1];
        char imsi[15+1];
        char buffer[512];
        printf("Reading IMEI and IMSI\r\n");

#ifdef TARGET_UBLOX_C030_N211
        if (at->send("AT+CGSN=1") && at->recv("\nOK\n%25[^\n]\nOK\n", imei)) {
            printf("IMEI: %s\r\n",imei + 6); // Avoid the "+CGSN:" prefix
        }

#else
        if (at->send("AT+CGSN") && at->recv("%15[^\n]\nOK\n", imei)) {
            printf("IMEI: %s\r\n",imei);
        }
#endif
        // Sometimes it takes a little while for the SIM to be initialised,
        // so retry this until done
        at->set_timeout(1000);
        for (int retry_count = 0; !success && (retry_count < 10); retry_count++) {
            if (at->send("AT+CIMI") && at->recv("%15[^\n]\nOK\n", imsi)) {
                printf("IMSI: %s\r\n",imsi);
                success = true;
            }
            
        }
        if (!success) {
            printf("Unable to read IMSI: has a SIM been inserted?\r\n");
        }
    } else {
        printf("Unable to intialize modem\r\n");
    }
    char buffer[500+5];
    printf("FINISHED...\r\n");
    printf ("sending nconfig");
    printf ("\r\n");


    connectNB();
    sendUDPMessage("52.215.10.12",3333,"HAUS");
 
   
    // Create threadUserButtonCheck thread
    Thread threadUserButtonCheck(threadBodyUserButtonCheck);

    // Set the LED states
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;

    // Main loop
    while(1) {
        wait(delay);
        // Shift the LED states
        int carry = ledBlue;
        ledBlue = ledRed;
        ledRed = ledGreen;
        ledGreen = carry;
    }
}

// End Of File