/* mbed Embedded Artists QVGA LCD Display Library
 * Copyright (c) 2007-2009 sford
 * Released under the MIT License: http://mbed.org/license/mit
 */
  
#include "EA_QVGALCD.h"

EA_QVGALCD::EA_QVGALCD(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName rst) : _spi(mosi, miso, sclk), _cs(cs), _rst(rst) {

	_spi.frequency(15000000);
	_spi.format(9);
	_cs = 1;
	_rst = 0;
	wait(0.001);
	_rst = 1;
	wait(0.001);

// 3-wire SPI Format (see 7.1 c)
// =================
//
// The 3-wire spi format uses 9-bit messages of the form:
//
// [Data/nControl D7 D6 D5 D4 D3 D2 D1 D0]
//  (first)                         (last)
// 
// Most commands and data are 16-bit, so are achieved by sending two
// command messages, [ CommandMSB | CommandLSB ], or two data messages
// [ DataMSB | DataLSB ].
//
// Controlling the Display
// =======================
// 
// The Index Register is used to setup which configuration register is
// modified. This address is changed by sending a command message:
//
//     command [ 0 | index[7:0] ]
//
// The data that follows update the register.
//
//
// 0x07 - Display Control
// ======================
//
// [ 0 0 0 PT1 PT0 VLE2 VLE1 SPT | 0 0 GON DTE CM 0 D1 D0 ]
// where:
//   PT[1:0]  - ? (0)
//   VLE[2:1] - ? (0)
//   SPT      - ? (0)
//   CM       - 8 colour mode enable
//   GON      - 0 = Gate-off level VGH 
//   DTE      - 0 = All gate outputs become VGL
//              1 = Selected gate outputs become VGH
//   D[1:0]   - D1 = Display enable
//              D0 = Display operating

// 15.5 On Sequence
//
// -> Set R07h: GON =1, DTE = 0, D[1:0] = 01
    _cs = 0;
	config(0x7
	  			, 1 << 5 // GON
				| 0 << 4 // DTE 
				| 0 << 3 // CM
				| 1 << 0 // D[1:0] = 01 - operate, but disp off
				);
// -> Set R00h to 0001h
// R00h [ 00000000 | 0000000 OSCEN ]
// OSCEN - 1 = oscillator enabled

	config(0x0, 1);

// -> Set R07h: GON = 1, DTE = 0, D[1:0] = 11	

	config(0x7
	  			, 1 << 5 // GON
				| 1 << 4 // DTE 
				| 0 << 3 // CM
				| 3 << 0 // D[1:0] = 3 - operate, display on
				);
				
// -> Set R10h at 0000h : Exit sleep mode				
// R10h [ 00000000 | 0000000 SLP ]
// SLP = enter sleep mode (retain but no function)

	config(0x10, 0);

// -> wait 30ms
	wait(0.030);
	
// -> LCD driver AC setting (R02h)
    config(0x02, 0x0600);
    config(0x01, 0x2b3f);  // 1011
    config(0x25, 0xa000);	// 70Hz freq

// -> Entry Mode setting (R11h)
// R11h [ VSMode DFM1 DFM0 TRANS OEDef WMode DMode1 DMode0 | TY1 TY0 ID1 ID0 AM LG2 LG1 LG0 ]
//  VSMode = freq dependant on VSYNC
//  DFM[1:0] colour display mode 11 - 65k, 10 - 262k
//  TRANS - allow transparent display
//  OEDef:
//    When OEDef = 1, OE defines the display window.
//    When OEDef = 0, the display window is defined by R4Eh and R4Fh.
//  WMode: Select the source of data to write in the RAM.
//     0 Normal data bus (POR)
//     1 Generic interface
//  Dmode - where to show from (0 = ram)
// TY - 262k mode options
// ID[1:0] [ VERTICAL HORIZONTAL ] increment = 1 decrement = 0
// AM - 0 = horizontal display, 1 = vertical
// LG - do maths n written data

	config(0x11
				, 0 << 15 // VSMode
				| 3 << 13 // DFM[1:0]
				| 0 << 12 // TRANS
				| 1 << 11 // OEDef
				| 0 << 10 // WMode
				| 0 << 8  // DMode[1:0]
				| 0 << 6  // TY[1:0]
				| 3 << 4  // ID[1:0]
				| 0 << 3  // AM 
				| 0 << 0  // LG[2:0]
				);
    _cs = 1;
    
    cls();
}

void EA_QVGALCD::pixel(int x, int y, int colour) {
    window(x, y, 1, 1);
    putp(colour);
}
     
int EA_QVGALCD::width() { return 240; }
int EA_QVGALCD::height() { return 320; }
     

void EA_QVGALCD::command(int value) {
	_spi.write(value & 0xFF);
}

void EA_QVGALCD::data(int value) {
	_spi.write(value | 0x100);
}

void EA_QVGALCD::config(int index, int value) {
	command(0);
	command(index);
	data(value >> 8);
	data(value);
}


void EA_QVGALCD::window(int x, int y, int w, int h) {
    _cs = 0;

	int hstart = x;
	int hend = x + w - 1;
	int vstart = y;
	int vend = y + h - 1;
	config(0x44, (hend << 8) | hstart);
	config(0x45, vstart);
	config(0x46, vend);

	config(0x4E, hstart & 0xFF);
    config(0x4F, vstart & 0x1FF);
	command(0);
	command(0x22);

    _cs = 1;
}

void EA_QVGALCD::putp(int colour) {
    _cs = 0;
	int top    = ((colour >> (8+8)) & 0xF8) // r7-3
	           | ((colour >> (5+8)) & 0x07); // g7-5 
	int bottom = ((colour >> 5) & 0xE0) // g4-2
			   | ((colour >> 3) & 0x1F); // b7-3
				   
 	data(top);
	data(bottom);
    _cs = 1; 
}

