// Example of a simple website hits/signup traffic monitor to drive 
// a display (e.g. servo or panel meter) and indicator (e.g. bell, solenoid)
// based on responses from a webserver
// http://mbed.org

#include "mbed.h"
#include "EthernetNetIf.h"
#include "HTTPClient.h"
#include "Servo.h"

#define SIGNUPS_URL "http://url-returning-signups-int"  // number of signups since last checked
#define HITS_URL "http://url-returning-hits-int"        // number of hits/sec since last checked
#define HITS_MAX 1000                                   // scale max hits/sec

EthernetNetIf eth;
HTTPClient http;

DigitalOut bell(p12);
Servo meter(p21);

// ding a bell (fire a digital out driving a solenoid)
void ding() {
    bell = 1;
    wait(0.25);
    bell = 0;
}

// fetch a integer from a url
int get_result(char *url) {
    HTTPText txt;
    HTTPResult r = http.get(url, &txt);

    if(r != HTTP_OK) {
        printf("Error [%d] fetching url [%s]\n", r, url);
        return -1;
    }
    
    printf("Response [%s] from url [%s]\n", txt.gets(), url);
    int v = 0;
    if(sscanf(txt.gets(), "%d", &v) != 1) {
        printf("Could not convert value to int\n");
    }
    
    return v;
}

int main() {
    printf("Setup the network...\n");
    EthernetErr ethErr = eth.setup();
    if (ethErr) {
        error("Error %d in setup\n", ethErr);
    }
    printf("OK!\n");

    while (1) {
        int hits = get_result(HITS_URL);
        int signups = get_result(SIGNUPS_URL);

        printf("hits = %d, signups = %d\n", hits, signups);
        
        // ding for every signup since last request
        for(int i=0; i<signups; i++) {
            ding();
            wait(0.5);
        }
    
        // set the servo based on hits/sec
        meter = (float)hits / (float)HITS_MAX;            
   }
}
