/* mbed LIS302 Accelerometer
 * Copyright (c) 2008-2009 cstyles, wreynolds, sford
 * Released under the MIT License: http://mbed.org/license/mit
 */

#ifndef MBED_LIS302_H
#define MBED_LIS302_H
 
#include "mbed.h"

/* Class: LIS302
 *  An abstraction for the LIS302 triple axis SPI accelerometer
 *
 * Example:
 * > // Print out the Z axis acceleration
 * >
 * > #include "mbed.h"
 * > #include "LIS302.h"
 * >
 * > LIS302 acc(p5, p6, p7, p8); // mosi, miso, clk, ncs
 * >
 * > int main() {
 * >     while(1) {
 * >         printf("Z axis acceleration = %.2f\n", acc.z());
 * >         wait(0.1);              
 * >     }
 * > }
 */
class LIS302  {
public:

    /* Constructor: LIS302
     *  Create an object for the LIS302, connected to the specified pins
     *
     * Variables:
     *  mosi - SPI data out
     *  miso - SPI data in
     *  clk - SPI clock
     *  ncs - Active low chip select. Any DigitalOut will do
     */
    LIS302(PinName mosi, PinName miso, PinName clk, PinName ncs);

    /* Function: x
     *  Read the X axis acceleration
     *
     * Variables:
     *  returns - A floating-point value representing acceleration in g
     */    
    float x();

    /* Function: y
     *  Read the Y axis acceleration
     *
     * Variables:
     *  returns - A floating-point value representing acceleration in g
     */    
    float y();

    /* Function: z
     *  Read the Z axis acceleration
     *
     * Variables:
     *  returns - A floating-point value representing acceleration in g
     */    
    float z();

    /* Function: range
     *  Select the range of the accelerometer
     *
     * Variables:
     *  range - 0 = 2g, 1 = 8g
     */        
    void range(int g);

    /* Function: Calibrate
     *  Configure the minima and maxima for the axes to linearise the readings
     *
     * Variables:
     *  maxx - float defining the maximum X value
     *  minx - float defining the minimum X value
     *  maxy - float defining the maximum Y value
     *  miny - float defining the minimum Y value
     *  maxz - float defining the maximum Z value
     *  minz - float defining the minimum Z value
     */        
    void calibrate(float maxx = 1, float minx = -1, float maxy = 1, float miny = -1, float maxz = 1, float minz = -1);
    
private:
    SPI _spi;
    DigitalOut _ncs;    

    int whoami();
    int status();
    
    float _factor;
    float _maxx, _minx;
    float _maxy, _miny;
    float _maxz, _minz;        
};

#endif
