#include "mbed.h"
#include "LEDColors.h"
#include "shell.h"

extern LEDColors *ledColors;

// create contructor
Shell::Shell(uint32_t thisBaudRate)
{
    usbSerial = new Serial(USBTX, USBRX);
    usbSerial->baud(115200);
    characterCount = 0;
    characterPointer = 0;
}
//-----------------------------------------------------------------------------

void Shell::sendText(char *thisText)
{
    // this can send any text
    usbSerial->printf(thisText);
}
//-----------------------------------------------------------------------------

void Shell::sendStartMessage()
{
    // sends the first greeting
    sendText(">     ** K64 LED controller **\n");
    sendText(">\n");
    sendText("> use commands like: turn blue LED on\n");
    sendText(">                or: turn red LED off\n");
    sendText(">\n> ");
}
//-----------------------------------------------------------------------------

void Shell::scanUSBSerialRx()
{
    // check if there is something to read
    if(usbSerial->readable()) {

        // if so ...
        char character = usbSerial->getc();

        // see if this is a semi colon or a carriage return
        // if so, give a new line cursor
        if((character == ';') || (character == 13)) {
            finishCharacterBuffer();
            parseCommands();
            usbSerial->printf("\n> ");
            // ledColors->flashWhite(10);
            ledColors->flash(lcBlue, 10);
        }

        // if not, just print the character
        else if(addCharacterToBuffer(character)) {
            usbSerial->printf("%c", character);
            ledColors->flash(lcGreen, 10);
        }
    }
}
//-----------------------------------------------------------------------------

bool Shell::addCharacterToBuffer(char newCharacter)
{
    bool accept = (characterPointer < ItsInputBufferSize_);
    if(accept) {
        inputBuffer[characterPointer++] = newCharacter;
        characterCount = characterPointer;
    }
    return accept;
}
//-----------------------------------------------------------------------------

void Shell::finishCharacterBuffer()
{
    inputBuffer[characterCount] = 0;
    characterCount = characterPointer;
    characterPointer = 0;
}
//-----------------------------------------------------------------------------

char Shell::lowercase(char thisChar)
{
    // check if this character is uppercase
    if((thisChar >= 'A') && (thisChar <= 'Z')) thisChar += 32;
    
    return thisChar;
}
//-----------------------------------------------------------------------------

bool Shell::findString(char *thisString)
{
    bool found = false;
    uint32_t startPointer = 0, characterPointer, newStartLocation, stringLength = 0;
    char firstChar, secondChar;

    // determine the length of thisString
    while(thisString[stringLength]) stringLength++;

    // start at the start position (0) on the input buffer
    while(!found
            && (characterCount >= stringLength)
            && (startPointer <= characterCount - stringLength)) {

        // start at the beginning of the compare string (thisString)
        found = true;
        characterPointer = 0;
        while(found && (characterPointer < stringLength)) {

            // get and compare characters
            firstChar = inputBuffer[startPointer + characterPointer];
            secondChar = thisString[characterPointer];
            found = (lowercase(firstChar) == lowercase(secondChar));

            // for debugging
            // usbSerial->printf("(compare %c with %c)\n", firstChar, secondChar);

            // move up to next characters
            characterPointer++;
        }

        // move start position
        if(found) {
            newStartLocation = startPointer + stringLength;

            // clean up from buffer
            characterPointer = newStartLocation;
            while(characterPointer < characterCount) {
                firstChar = inputBuffer[characterPointer];
                inputBuffer[characterPointer - newStartLocation] = firstChar;
                characterPointer++;
            }
            characterCount -= newStartLocation;
            inputBuffer[characterCount] = 0;

            // for debugging send buffer
            /*
            if(characterCount) {
                sendText(" is now \"");
                sendText(inputBuffer);
                sendText("\"");
            }
            */
        }

        startPointer++;
    }

    // report
    return found;
}
//-----------------------------------------------------------------------------

void Shell::parseCommands()
{
    if(findString("set")) sendText(" -- \"set\" found!");
    if(findString("Clark")) sendText(" -- \"Clark\" found!");
    if(findString("Stefan")) sendText(" -- Hi \"Stefan\"!");
    if(findString("Ralph")) sendText(" -- Hey, that's me!");

    if(findString("turn")) parseLEDState();
    if(findString("generate")) parseGenerate();
    if(findString("get")) parseGet();
}
//-----------------------------------------------------------------------------

void Shell::parseLEDState()
{
    // example text: turn blue LED on

    eLEDColor ledColor = lcNone;
    if(findString("red")) ledColor = lcRed;
    if(findString("blue")) ledColor = lcBlue;
    if(findString("green")) ledColor = lcGreen;

    // check if color is specified
    if(ledColor == lcNone) {
        sendText(" -- missing LED color: red, blue or green");
        sendExample();
    }
    else {
        
        // check if LED is specified
        if(findString("LED")) {

            // check for state
            eState state = stNone;
            if(findString("on")) state = stOn;
            if(findString("off")) state = stOff;
            
            if(state == stNone) {
                
                // state not specified
                sendText(" -- missing state: on or off");
                sendExample();
            }
            else {
                
                // check if state has changed
                if(ledColors->getState(ledColor) == state) {
                    
                    // state not changed
                    sendText(" -- already ");
                    (state == stOn) ? sendText("on") : sendText("off");
                }
                else {
                    
                    // state has changed
                    sendText(" -- done!");
                    ledColors->turn(ledColor, state);
                }
            }
        }
        else {
            
            // LED was not specified
            sendText(" -- specify LED");
            sendExample();
        }
    }
}
//-----------------------------------------------------------------------------

void Shell::sendExample()
{
    sendText(" -> example: turn red LED on");
}
//-----------------------------------------------------------------------------

void Shell::parseGenerate()
{
    if(findString("pi")) generatePi();
    else sendText(" -- generate what?");
}
//-----------------------------------------------------------------------------

void Shell::parseGet()
{
    if(findString("pi")) generatePi();
    else sendText(" -- get what?");
}
//-----------------------------------------------------------------------------

void Shell::reportPushButtonPress(ePushButton thisPushButton)
{
    switch(thisPushButton) {
        case pb2: sendText("SW2"); break;
        case pb3: sendText("SW3"); break;
    }
    sendText(" pressed\n> ");
}
//-----------------------------------------------------------------------------

void Shell::generatePi()
{
    const int n = 1000, len = 10 * n / 3 + 1;
    int i, j, k, q = 0, x, nines = 0, predigit = 0, digits = 0, a[len];

    decimalCounter = 0;
    for(j = 0; j < len; j++) a[j] = 2;
    for(j = 0; j < n; j++) {
        q = 0;
        for(i = len; i > 0; i--) {
            x = 10 * a[i] + q * i;
            a[i] = x % (2 * i - 1);
            q = x / (2 * i - 1);
        }
        a[1] = q % 10;
        q /= 10;
        if(q == 9) nines++;
        else {
            if(q == 10) {
                sendDecimal(predigit + 1);
                for(k = 0; k < nines; k++) sendDecimal(0);
                predigit = 0;
            } else {
                sendDecimal(predigit);
                predigit = q;
                for(k = 0; k < nines; k++) sendDecimal(9);
            }
            nines = 0;
            digits += nines + 1;
        }
    }
}
//-----------------------------------------------------------------------------

void Shell::sendDecimal(uint8_t decimal)
{
    switch(decimalCounter) {
        case 0: break;
        case 1: usbSerial->printf(" -- pi = %d.", decimal); break;
        default: usbSerial->printf("%d", decimal);
    }
    decimalCounter++;    
}
//-----------------------------------------------------------------------------
