#ifndef H_ROBOT_LOGIC
#define H_ROBOT_LOGIC

#include "mbed.h"
#include "m3pi.h"

typedef enum {
   LED_1 = 0,
   LED_2 = 1,
   LED_3 = 2,
   LED_4 = 3,
   LED_5 = 4,
   LED_6 = 5,
   LED_7 = 6,
   LED_8 = 7
} LedIndex;

typedef enum {
    LED_ON = 0,
    LED_OFF = 1,
    LED_TOGGLE = 2
} LedState;


/**
 * Drive the robot forward or backward.
 * If the robot was turning it will stop turning and drive in a straight line.
 *
 * @param speed The speed percentage with which to drive forward or backward.
 * Can range from -100 (full throttle backward) to +100 (full throttle forward).
 */
void drive(int speed);

/**
 * Turn the robot left or right while driving.
 *
 * @param turnspeed The percentage with which to turn the robot.
 *          Can range from -100 (full throttle left) to +100 (full throttle right).
 */
void turn(int turnspeed);

/**
 * Stop the robot.
 */
void stop(void);

/**
 * Calibrate the line follow sensors.
 * Take note that the robot should be placed over the line
 * before this function is called and that it will rotate to
 * both sides.
 */
void sensor_calibrate(void);

/**
 * Read the value from the line sensor. The returned value indicates the
 * position of the line. The value ranges from -100 to +100 where -100 is
 * fully left, +100 is fully right and 0 means the line is detected in the middle.
 *
 * @return The position of the line with a range of -100 to +100.
 */
int line_sensor(void);

/**
 * Initialize the PID drive control with
 * the P, I and D factors.
 *
 * @param p The P factor
 * @param i The I factor
 * @param d The D factor
 */
void pid_init(int p, int i, int d);

/**
 * Determine PID turnspeed with which the robot should
 * turn to follow the line at the given position.
 *
 * @param line_position The position of the line in a range of [-100, +100]
 *
 * @return The turnspeed in a range of [-100, +100]
 */
int pid_turn(int line_position);

/**
  *Show speed, turn and sensor data on the LCD
  */
void show_stats();

/**
 * Shows the name of the robot on the display.
 *
 * @param name C character string (null-terminated) with the name of the robot (max 8 chars)
 */
void show_name(char * name);


/**
  * Turn on, off or toggle a specific LED
  * @param i the LED number LED_0 .. LED_7
  * @param state the LED state LED_ON, LED_OFF, LED_TOGGLE 
  * @example led(LED_0, LED_ON); turns LED 0 on.
  */
void led(LedIndex i, LedState state);

/**
 * Wait for an approximate number of milliseconds.
 *
 * @param milliseconds The number of milliseconds to wait.
 */
void await(int milliseconds);

#endif