#pragma once

namespace Log{

class LoggerInterface
{
    public:

    enum Level {EMERGENCY, ALERT, CRITICAL, ERROR, WARNING, NOTICE, INFO, DEBUG};

    /**
     * System is unusable.
     *
     * @param string $message
     */
    virtual void emergency(const char* message, ...) = 0;

    /**
     * Action must be taken immediately.
     *
     * Example: Entire website down, database unavailable, etc. This should
     * trigger the SMS alerts and wake you up.
     *
     * @param string $message
     */
    virtual void alert(const char* message, ...) = 0;

    /**
     * Critical conditions.
     *
     * Example: Application component unavailable, unexpected exception.
     *
     * @param string $message
     */
    virtual void critical(const char* message, ...) = 0;

    /**
     * Runtime errors that do not require immediate action but should typically
     * be logged and monitored.
     *
     * @param string $message
     */
    virtual void error(const char* message, ...) = 0;

    /**
     * Exceptional occurrences that are not errors.
     *
     * Example: Use of deprecated APIs, poor use of an API, undesirable things
     * that are not necessarily wrong.
     *
     * @param string $message
     */
    virtual void warning(const char* message, ...) = 0;

    /**
     * Normal but significant events.
     *
     * @param string $message
     */
    virtual void notice(const char* message, ...) = 0;

    /**
     * Interesting events.
     *
     * Example: User logs in, SQL logs.
     *
     * @param string $message
     */
    virtual void info(const char* message, ...) = 0;

    /**
     * Detailed debug information.
     *
     * @param string $message
     */
    virtual void debug(const char* message, ...) = 0;

    /**
     * Logs with an arbitrary level.
     *
     * @param mixed $level
     * @param string $message
     */
    virtual void log(Level level, const char* message, ...) = 0;
};

}
