/**
================================================================================
                              GLASS LCD MAPPING
================================================================================
LCD allows to display informations on four 7-segment digits and 2 arrows:

      1       2       3       4
    -----   -----   -----   -----
 /\ |   |   |   | o |   |   |   |
    -- --   -- --   -- --   -- --
 \/ |   |   |   | o |   |   |   |
    ----- * ----- * ----- * -----

LCD segment mapping:
--------------------
  -----A-----        _
  |         |     L |_|
  F         B
  |         |        _
  -----G-----     L |_|
  |         |
  E         C
  |         |        _
  -----D-----   DP  |_|

 An LCD digir coding is based on the following matrix:
COM           0   1
SEG(n)      { E , G }
SEG(n+1)    { D , F }
SEG(23-n-1) { C , A }
SEG(23-n)   { DP, B }
with n positive odd number.

The DIGIT '0' for example is:
 -------------------------------
LSB   { 1 , 0 }
      { 1 , 1 }
      { 1 , 1 }
MSB   { 0 , 1 }
     -------------------
 'A' =  7    E  hexa
  */
#include "stm32l053c8_discovery_glass_lcd.h"
#include "mbed_wait_api.h"
const int NumberMap[10] ={
  /*    0     1     2     3     4     5     6     7     8     9  */
      0x7E, 0x28, 0x67, 0x3D, 0x4B, 0x67, 0x77, 0x4C, 0x7F, 0x6F
};

LCD_HandleTypeDef LCDHandle;

static void LCD_MspInit(LCD_HandleTypeDef *hlcd);
static void LCD_MspDeInit(LCD_HandleTypeDef *hlcd);

void BSP_LCD_GLASS_Init(void){
  LCDHandle.Instance              = LCD;
  LCDHandle.Init.Prescaler        = LCD_PRESCALER_1;
  LCDHandle.Init.Divider          = LCD_DIVIDER_16;
  LCDHandle.Init.Duty             = LCD_DUTY_STATIC;
  LCDHandle.Init.Bias             = LCD_BIAS_1_2;
  LCDHandle.Init.VoltageSource    = LCD_VOLTAGESOURCE_INTERNAL;
  LCDHandle.Init.Contrast         = LCD_CONTRASTLEVEL_5;
  LCDHandle.Init.DeadTime         = LCD_DEADTIME_0;
  LCDHandle.Init.PulseOnDuration  = LCD_PULSEONDURATION_6;
  LCDHandle.Init.HighDrive        = LCD_HIGHDRIVE_0;
  LCDHandle.Init.BlinkMode        = LCD_BLINKMODE_OFF;
  LCDHandle.Init.BlinkFrequency   = LCD_BLINKFREQUENCY_DIV32;
  LCDHandle.Init.MuxSegment       = LCD_MUXSEGMENT_DISABLE;

  /* Initialize the LCD */
  LCD_MspInit(&LCDHandle);
  HAL_LCD_Init(&LCDHandle);

  BSP_LCD_GLASS_Clear();
}

/**
  * @brief  DeInitialize the LCD GLASS relative GPIO port IOs and LCD peripheral.
  * @retval None
  */
void BSP_LCD_GLASS_DeInit(void){
  /* De-Initialize the LCD */
  LCD_MspDeInit(&LCDHandle);
  HAL_LCD_DeInit(&LCDHandle);
}

/**
  * @brief  Initialize the LCD MSP.
  * @param  hlcd: LCD handle
  * @retval None
  */
static void LCD_MspInit(LCD_HandleTypeDef *hlcd){
  GPIO_InitTypeDef  gpioinitstruct = {0};
  RCC_OscInitTypeDef oscinitstruct = {0};
  RCC_PeriphCLKInitTypeDef periphclkstruct = {0};

  /*##-1- Enable PWR  peripheral Clock #######################################*/
  __HAL_RCC_PWR_CLK_ENABLE();

  /*##-2- Configure LSE as RTC clock soucre ###################################*/
  oscinitstruct.OscillatorType  = RCC_OSCILLATORTYPE_LSE;
  oscinitstruct.PLL.PLLState    = RCC_PLL_NONE;
  oscinitstruct.LSEState        = RCC_LSE_ON;
  if (HAL_RCC_OscConfig(&oscinitstruct) != HAL_OK)
  {
    while (1);
  }

  /*##-3- Select LSE as RTC clock source.##########################*/
  /* Backup domain management is done in RCC function */
  periphclkstruct.PeriphClockSelection = RCC_PERIPHCLK_RTC;
  periphclkstruct.RTCClockSelection = RCC_RTCCLKSOURCE_LSE;
  HAL_RCCEx_PeriphCLKConfig(&periphclkstruct);

  /*##-4- Enable LCD GPIO Clocks #############################################*/
  __HAL_RCC_GPIOA_CLK_ENABLE();
  // __HAL_RCC_GPIOB_CLK_ENABLE();
  // __HAL_RCC_GPIOC_CLK_ENABLE();
  // __HAL_RCC_GPIOD_CLK_ENABLE();


  /*##-5- Configure peripheral GPIO ##########################################*/
  /* Configure Output for LCD */
  /* Port A */
  gpioinitstruct.Pin        = LCD_GPIO_BANKA_PINS;
  gpioinitstruct.Mode       = GPIO_MODE_AF_PP;
  gpioinitstruct.Pull       = GPIO_NOPULL;
  gpioinitstruct.Speed      = GPIO_SPEED_FREQ_HIGH;
  gpioinitstruct.Alternate  = GPIO_AF1_LCD;
  HAL_GPIO_Init(GPIOA, &gpioinitstruct);

  /* Port B */
  // gpioinitstruct.Pin        = LCD_GPIO_BANKB_PINS;
  // HAL_GPIO_Init(GPIOB, &gpioinitstruct);

  /* Port C*/
  // gpioinitstruct.Pin        = LCD_GPIO_BANKC_PINS;
  // HAL_GPIO_Init(GPIOC, &gpioinitstruct);

  /* Port D */
  // gpioinitstruct.Pin        = LCD_GPIO_BANKD_PINS;
  // HAL_GPIO_Init(GPIOD, &gpioinitstruct);

  /* Wait for the external capacitor Cext which is connected to the VLCD pin is charged
  (approximately 2ms for Cext=1uF) */
  wait_ms(2);

  /*##-6- Enable LCD peripheral Clock ########################################*/
  __HAL_RCC_LCD_CLK_ENABLE();
}

/**
  * @brief  DeInitialize the LCD MSP.
  * @param  hlcd: LCD handle
  * @retval None
  */
static void LCD_MspDeInit(LCD_HandleTypeDef *hlcd){
  uint32_t gpiopin = 0;

  /*##-1- Enable LCD GPIO Clocks #############################################*/
  __HAL_RCC_GPIOA_CLK_ENABLE();
  // __HAL_RCC_GPIOB_CLK_ENABLE();
  // __HAL_RCC_GPIOC_CLK_ENABLE();
  // __HAL_RCC_GPIOD_CLK_ENABLE();

  /*##-1- Configure peripheral GPIO ##########################################*/
  /* Configure Output for LCD */
  /* Port A */
  gpiopin = LCD_GPIO_BANKA_PINS;
  HAL_GPIO_DeInit(GPIOA, gpiopin);

  /* Port B */
  // gpiopin = LCD_GPIO_BANKB_PINS;
  // HAL_GPIO_DeInit(GPIOB, gpiopin);

  /* Port C*/
  // gpiopin = LCD_GPIO_BANKC_PINS;
  // HAL_GPIO_DeInit(GPIOC, gpiopin);

  /* Port D */
  // gpiopin = LCD_GPIO_BANKD_PINS;
  // HAL_GPIO_DeInit(GPIOD, gpiopin);

  /*##-5- Enable LCD peripheral Clock ########################################*/
  __HAL_RCC_LCD_CLK_DISABLE();
}

void BSP_LCD_GLASS_Clear(void){
  HAL_LCD_Clear(&LCDHandle);
}

void BSP_LCD_GLASS_WriteDigit(int digit){
    int Digit = NumberMap[digit];

    uint32_t data = 0x00;
    data = ((Digit & 0x10) << LCD_SEG0_SHIFT) | (((Digit & 0x20) >> 1) << LCD_SEG1_SHIFT)
             | (((Digit & 0x40) >> 2) << LCD_SEG2_SHIFT) | (((Digit & 0x80) >> 3) << LCD_SEG3_SHIFT);
    HAL_LCD_Write(&LCDHandle, LCD_RAM_REGISTER0, 0xFF, 0xFF);
    data = ((Digit & 0x01) << LCD_SEG0_SHIFT) | (((Digit & 0x02) >> 1) << LCD_SEG1_SHIFT)
             | (((Digit & 0x04) >> 2) << LCD_SEG2_SHIFT) | (((Digit & 0x08) >> 3) << LCD_SEG3_SHIFT);
//    HAL_LCD_Write(&LCDHandle, LCD_RAM_REGISTER1, 0xF0, 0xF0);
    HAL_LCD_UpdateDisplayRequest(&LCDHandle);
}
