/**
 * This is a class to drive 4D systems screens with Picaso graphics processors
 * 
 * This is a port of the 4D systems Picaso Serial Linux Library
 * Copyright (c) 2014 4D Systems PTY Ltd, Sydney, Australia
 * From https://github.com/4dsystems/Picaso-Serial-Linux-Library, retrieved March 5, 2015
 *
 * All software in this library is provided without warranty; use at your own risk.
 */
 
#include "mbed.h"
#include <string.h>
#include <math.h>
#include <ctype.h>

#include "Picaso_enums.h"
#include "uLCD_4D_Picaso.h"

//Error types
#define   Err4D_OK      0
#define   Err4D_Timeout 1
#define   Err4D_NAK		2


//Default wait time (in ms) for Serial acknowledgement
#define RX_MAX_WAIT_DEFAULT 2000

#if LCD_USING_MODSERIAL
//Serial port buffer sizes
#define TX_BUF_SIZE 256
#define RX_BUF_SIZE 2048
#endif



/*
 * Start of 4D Intrinsic Routines
*/
uLCD_4D_Picaso::uLCD_4D_Picaso(PinName tx, PinName rx, PinName reset) :
		#if LCD_USING_MODSERIAL
		serial(tx, rx, TX_BUF_SIZE, RX_BUF_SIZE),
		#else
		serial(tx, rx),
		#endif
		rst(reset) 
{
	TimeLimit4D = RX_MAX_WAIT_DEFAULT;
	serial.baud(9600);
	LCD_reset();
	gfx_Cls();
}

void uLCD_4D_Picaso::LCD_reset() {
	rst = 0;
	LCD_wait_ms(1);
	rst = 1;
	LCD_wait_ms(3000);
	rxFlush();
}

void uLCD_4D_Picaso::LCD_wait_ms(int ms) {
	timer.reset();
	timer.start();
	while (timer.read_ms() < ms);
	timer.stop();
	return;
}

void uLCD_4D_Picaso::rxFlush() {
	while (serial.readable()) {
		serial.getc();
	}
}

void uLCD_4D_Picaso::WriteBytes(char *psOutput, int nCount) {
    int i;
    rxFlush();
    for (i = 0; i < nCount; ++i) {
    	while (!serial.writeable());
		serial.putc(psOutput[i]);
	}
}

void uLCD_4D_Picaso::WriteChars(char *psOutput) {
    // Include NULL in output
    WriteBytes(psOutput, strlen((char *)psOutput) + 1);
    return;
}

void uLCD_4D_Picaso::WriteWords(uint16_t * Source, int Size) {
 	uint16_t wk ;
	int i ;
	for (i = 0; i < Size; i++)
	{
		wk = *Source++ ;
		wk = (wk >> 8) + (wk << 8) ;
		WriteBytes((char *)&wk, 2);
	}
}

// read string from the serial port
// return code:
//   >= 0 = number of characters read
//   -1 = read failed
int uLCD_4D_Picaso::ReadSerPort(char *psData, int iMax) {
    int i;
    Error4D = Err4D_OK;
    
    timer.reset();
    timer.start();
	
    for (i = 0; i < iMax; ++i) {
    	while (!serial.readable()) {  		
    		if (timer.read_ms() >= TimeLimit4D) {
    			Error4D = Err4D_Timeout;
    			timer.stop();
    			return -1;
			}
		}
		psData[i] = serial.getc();
	}
	
	timer.stop();
    return iMax;
}

void uLCD_4D_Picaso::getbytes(char *data, int size) {
 	int readc;
	readc = ReadSerPort(data, size);
	if (readc != size)	{
		printf("LCD serial: read timeout\r\n");
 	}
}

void uLCD_4D_Picaso::GetAck(void) {
	char readx;
   	getbytes(&readx, 1);
   	
	if (readx != 6) {
	   	Error4D = Err4D_NAK;
	   	printf("LCD serial: no ACK received\r\n");
	}
    return;
}


uint16_t uLCD_4D_Picaso::GetWord(void) {
 	char readx[2];
    getbytes(readx, 2);
	return readx[0] << 8 | readx[1];
}

void uLCD_4D_Picaso::getString(char *outStr, int strLen) {
	getbytes(outStr, strLen);

    // Append EOS
	outStr[strLen] = '\0';

	return;
}

uint16_t uLCD_4D_Picaso::GetAckResp(void) {
	GetAck();
	return GetWord();
}

uint16_t uLCD_4D_Picaso::WaitForAck(void) {
    int saveTimeout = TimeLimit4D;

    // check once per minute
    TimeLimit4D = 60 * 1000;
    do {
        GetAck();
    } while (Error4D != Err4D_OK);

    // Restore callback/timeout saves
    TimeLimit4D = saveTimeout;

    return GetWord();
}

uint16_t uLCD_4D_Picaso::GetAckRes2Words(uint16_t * word1, uint16_t * word2) {
	int Result;
	GetAck();
	Result = GetWord();
	*word1 = GetWord();
	*word2 = GetWord();
	return Result;
}

void uLCD_4D_Picaso::GetAck2Words(uint16_t * word1, uint16_t * word2) {
	GetAck();
	*word1 = GetWord();
	*word2 = GetWord();
}

uint16_t uLCD_4D_Picaso::GetAckResSector(char* Sector) {
	int Result;
	GetAck();
	Result = GetWord();
	getbytes(Sector, 512);
	return Result;
}

uint16_t uLCD_4D_Picaso::GetAckResStr(char * OutStr) {
	int Result;
	GetAck();
	Result = GetWord();
	getString(OutStr, Result);
	return Result;
}

uint16_t uLCD_4D_Picaso::GetAckResData(char* OutData, uint16_t size) {
	int Result;
	GetAck();
	Result = GetWord();
	getbytes(OutData, size);
	return Result;
}


void uLCD_4D_Picaso::SetThisBaudrate(Picaso::BaudRate Newrate) {
    int nBaud;

    switch (Newrate) {
	    case     Picaso::BAUD_110:       nBaud =    110; break;
	    case     Picaso::BAUD_300:       nBaud =    300; break;
	    case     Picaso::BAUD_600:       nBaud =    600; break;
	    case    Picaso::BAUD_1200:       nBaud =   1200; break;
	    case    Picaso::BAUD_2400:       nBaud =   2400; break;
	    case    Picaso::BAUD_4800:       nBaud =   4800; break;
	    case    Picaso::BAUD_9600:       nBaud =   9600; break;
	    case   Picaso::BAUD_14400:       nBaud =  14400; break;
	    case   Picaso::BAUD_19200:       nBaud =  19200; break;
	    case   Picaso::BAUD_31250:       nBaud =  31250; break;
	    case   Picaso::BAUD_38400:       nBaud =  38400; break;
	    case   Picaso::BAUD_56000:       nBaud =  56000; break;
	    case   Picaso::BAUD_57600:       nBaud =  57600; break;
	    case  Picaso::BAUD_115200:       nBaud = 115200; break;
	    case  Picaso::BAUD_128000:       nBaud = 133929; break;
	    case  Picaso::BAUD_256000:       nBaud = 281250; break;
	    case  Picaso::BAUD_300000:       nBaud = 312500; break;
	    case  Picaso::BAUD_375000:       nBaud = 401786; break;
	    case  Picaso::BAUD_500000:       nBaud = 562500; break;
	    case  Picaso::BAUD_600000:       nBaud = 703125; break;

	    default:
	    	nBaud = 9600 ;
	    }
	
	//Need to wait for the previous command to get out before setting new baud rate.
	//10 ms seems to work for 9600, 40 ms for 2400
	//Even using MODSERIAL's txIsBusy() doesn't seem to replace the need for this
	LCD_wait_ms(40);
    serial.baud(nBaud);
    
    //Wait 100ms, as specified in serial command manual
    LCD_wait_ms(100);
   
    return;
}

/*
 * End Of Intrinsic 4DRoutines here
*/

/*
 * Starts of 4D Compound Routines
*/

uint16_t uLCD_4D_Picaso::bus_In()
{
  char  towrite[2] ;

  towrite[0]= F_bus_In >> 8 ;
  towrite[1]= F_bus_In ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::bus_Out(uint16_t  Bits)
{
  char  towrite[4] ;

  towrite[0]= F_bus_Out >> 8 ;
  towrite[1]= F_bus_Out ;
  towrite[2]= Bits >> 8 ;
  towrite[3]= Bits ;
  WriteBytes(towrite, 4) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::bus_Read()
{
  char  towrite[2] ;

  towrite[0]= F_bus_Read >> 8 ;
  towrite[1]= F_bus_Read ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::bus_Set(uint16_t  IOMap)
{
  char  towrite[4] ;

  towrite[0]= F_bus_Set >> 8 ;
  towrite[1]= F_bus_Set ;
  towrite[2]= IOMap >> 8 ;
  towrite[3]= IOMap ;
  WriteBytes(towrite, 4) ;
  GetAck() ;
}

void uLCD_4D_Picaso::bus_Write(uint16_t  Bits)
{
  char  towrite[4] ;

  towrite[0]= F_bus_Write >> 8 ;
  towrite[1]= F_bus_Write ;
  towrite[2]= Bits >> 8 ;
  towrite[3]= Bits ;
  WriteBytes(towrite, 4) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::charheight(char  TestChar)
{
  char  towrite[3] ;

  towrite[0]= F_charheight >> 8 ;
  towrite[1]= F_charheight ;
  towrite[2]= TestChar;
  WriteBytes(towrite, 3) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::charwidth(char  TestChar)
{
  char  towrite[3] ;

  towrite[0]= F_charwidth >> 8 ;
  towrite[1]= F_charwidth ;
  towrite[2]= TestChar;
  WriteBytes(towrite, 3) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Close(uint16_t  Handle)
{
  char  towrite[4] ;

  towrite[0]= F_file_Close >> 8 ;
  towrite[1]= F_file_Close & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Count(char*  Filename)
{
  char  towrite[2] ;

  towrite[0]= F_file_Count >> 8 ;
  towrite[1]= F_file_Count ;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Dir(char*  Filename)
{
  char  towrite[2] ;

  towrite[0]= F_file_Dir >> 8 ;
  towrite[1]= F_file_Dir ;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Erase(char*  Filename)
{
  char  towrite[2] ;

  towrite[0]= F_file_Erase >> 8 ;
  towrite[1]= F_file_Erase & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResp() ;
}

Picaso::FileError uLCD_4D_Picaso::file_Error()
{
  char  towrite[2] ;

  towrite[0]= F_file_Error >> 8 ;
  towrite[1]= F_file_Error & 0xFF;
  WriteBytes(towrite, 2);
  return (Picaso::FileError) GetAckResp();
}

uint16_t uLCD_4D_Picaso::file_Exec(char*  Filename, uint16_t  ArgCount, uint16_t*  Args)
{
  char  towrite[2] ;

  towrite[0]= F_file_Exec >> 8 ;
  towrite[1]= F_file_Exec & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  towrite[0]= ArgCount >> 8 ;
  towrite[1]= ArgCount ;
  WriteBytes(towrite, 2) ;
  WriteWords(Args, ArgCount) ;
  return WaitForAck() ;
}

uint16_t uLCD_4D_Picaso::file_Exists(char*  Filename)
{
  char  towrite[2] ;

  towrite[0]= F_file_Exists >> 8 ;
  towrite[1]= F_file_Exists & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_FindFirst(char*  Filename)
{
  char  towrite[2] ;

  towrite[0]= F_file_FindFirst >> 8 ;
  towrite[1]= F_file_FindFirst & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_FindNext()
{
  char  towrite[2] ;
  towrite[0]= F_file_FindNext >> 8 ;
  towrite[1]= F_file_FindNext & 0xFF;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

char uLCD_4D_Picaso::file_GetC(uint16_t  Handle)
{
  char  towrite[4] ;
  towrite[0]= F_file_GetC >> 8 ;
  towrite[1]= F_file_GetC & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_GetS(char*  StringIn, uint16_t  Size, uint16_t  Handle)
{
  char  towrite[6] ;
  towrite[0]= F_file_GetS >> 8 ;
  towrite[1]= F_file_GetS & 0xFF;
  towrite[2]= Size >> 8 ;
  towrite[3]= Size ;
  towrite[4]= Handle >> 8 ;
  towrite[5]= Handle ;
  WriteBytes(towrite, 6) ;
  return GetAckResStr(StringIn) ;
}

uint16_t uLCD_4D_Picaso::file_GetW(uint16_t  Handle)
{
  char  towrite[4] ;
  towrite[0]= F_file_GetW >> 8 ;
  towrite[1]= F_file_GetW & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Image(uint16_t  X, uint16_t  Y, uint16_t  Handle)
{
  char  towrite[8] ;
  towrite[0]= F_file_Image >> 8 ;
  towrite[1]= F_file_Image & 0xFF;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Handle >> 8 ;
  towrite[7]= Handle ;
  WriteBytes(towrite, 8) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Index(uint16_t  Handle, uint16_t  HiSize, uint16_t  LoSize, uint16_t  Recordnum)
{
  char  towrite[10] ;
  towrite[0]= F_file_Index >> 8 ;
  towrite[1]= F_file_Index & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= HiSize >> 8 ;
  towrite[5]= HiSize ;
  towrite[6]= LoSize >> 8 ;
  towrite[7]= LoSize ;
  towrite[8]= Recordnum >> 8 ;
  towrite[9]= Recordnum ;
  WriteBytes(towrite, 10) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_LoadFunction(char*  Filename)
{
  char  towrite[2] ;
  towrite[0]= F_file_LoadFunction >> 8 ;
  towrite[1]= F_file_LoadFunction ;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_LoadImageControl(char*  Datname, char*  GCIName, uint16_t  Mode)
{
  char  towrite[2] ;
  towrite[0]= F_file_LoadImageControl >> 8 ;
  towrite[1]= F_file_LoadImageControl ;
  WriteBytes(towrite, 2) ;
  WriteChars(Datname) ;
  WriteChars(GCIName) ;
  towrite[0]= Mode >> 8 ;
  towrite[1]= Mode ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Mount()
{
  char  towrite[2] ;
  towrite[0]= F_file_Mount >> 8 ;
  towrite[1]= F_file_Mount & 0xFF;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Open(char*  Filename, char  Mode)
{
  char  towrite[2] ;
  towrite[0]= F_file_Open >> 8 ;
  towrite[1]= F_file_Open & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  towrite[0]= Mode;
  WriteBytes(towrite, 1) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_PlayWAV(char*  Filename)
{
  char  towrite[2] ;
  towrite[0]= F_file_PlayWAV >> 8 ;
  towrite[1]= F_file_PlayWAV & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_PutC(char Character, uint16_t  Handle)
{
  char  towrite[5] ;
  towrite[0]= F_file_PutC >> 8 ;
  towrite[1]= F_file_PutC ;
  towrite[2]= Character;
  towrite[3]= Handle >> 8 ;
  towrite[4]= Handle ;
  WriteBytes(towrite, 5) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_PutS(char *  StringOut, uint16_t  Handle)
{
  char  towrite[2] ;
  towrite[0]= F_file_PutS >> 8 ;
  towrite[1]= F_file_PutS & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(StringOut) ;
  towrite[0]= Handle >> 8 ;
  towrite[1]= Handle ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_PutW(uint16_t  Word, uint16_t  Handle)
{
  char  towrite[6] ;
  towrite[0]= F_file_PutW >> 8 ;
  towrite[1]= F_file_PutW & 0xFF;
  towrite[2]= Word >> 8 ;
  towrite[3]= Word ;
  towrite[4]= Handle >> 8 ;
  towrite[5]= Handle ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Read(char*  Data, uint16_t  Size, uint16_t  Handle)
{
  char  towrite[6] ;
  towrite[0]= F_file_Read >> 8 ;
  towrite[1]= F_file_Read & 0xFF;
  towrite[2]= Size >> 8 ;
  towrite[3]= Size ;
  towrite[4]= Handle >> 8 ;
  towrite[5]= Handle ;
  WriteBytes(towrite, 6) ;
  return GetAckResData(Data,Size) ;
}

uint16_t uLCD_4D_Picaso::file_Rewind(uint16_t  Handle)
{
  char  towrite[4] ;
  towrite[0]= F_file_Rewind >> 8 ;
  towrite[1]= F_file_Rewind & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Run(char *  Filename, uint16_t  ArgCount, uint16_t*  Args)
{
  char  towrite[2] ;
  towrite[0]= F_file_Run >> 8 ;
  towrite[1]= F_file_Run & 0xFF;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  towrite[0]= ArgCount >> 8 ;
  towrite[1]= ArgCount ;
  WriteBytes(towrite, 2) ;
  WriteWords(Args, ArgCount) ;
  return WaitForAck() ;
}

uint16_t uLCD_4D_Picaso::file_ScreenCapture(uint16_t  X, uint16_t  Y, uint16_t  Width, uint16_t  Height, uint16_t  Handle)
{
  char  towrite[12] ;
  towrite[0]= F_file_ScreenCapture >> 8 ;
  towrite[1]= F_file_ScreenCapture & 0xFF;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Width >> 8 ;
  towrite[7]= Width ;
  towrite[8]= Height >> 8 ;
  towrite[9]= Height ;
  towrite[10]= Handle >> 8 ;
  towrite[11]= Handle ;
  WriteBytes(towrite, 12) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Seek(uint16_t  Handle, uint16_t  HiWord, uint16_t  LoWord)
{
  char  towrite[8] ;
  towrite[0]= F_file_Seek >> 8 ;
  towrite[1]= F_file_Seek & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= HiWord >> 8 ;
  towrite[5]= HiWord ;
  towrite[6]= LoWord >> 8 ;
  towrite[7]= LoWord ;
  WriteBytes(towrite, 8) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_Size(uint16_t  Handle, uint16_t *  HiWord, uint16_t *  LoWord)
{
  char  towrite[4] ;
  towrite[0]= F_file_Size >> 8 ;
  towrite[1]= F_file_Size ;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckRes2Words(HiWord,LoWord) ;
}

uint16_t uLCD_4D_Picaso::file_Tell(uint16_t  Handle, uint16_t *  HiWord, uint16_t *  LoWord)
{
  char  towrite[4] ;
  towrite[0]= F_file_Tell >> 8 ;
  towrite[1]= F_file_Tell ;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckRes2Words(HiWord,LoWord) ;
}

void uLCD_4D_Picaso::file_Unmount()
{
  char  towrite[2] ;

  towrite[0]= F_file_Unmount >> 8 ;
  towrite[1]= F_file_Unmount & 0xFF;
  WriteBytes(towrite, 2) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::file_Write(uint16_t  Size, char*  Source, uint16_t  Handle)
{
  char  towrite[4] ;
  towrite[0]= F_file_Write >> 8 ;
  towrite[1]= F_file_Write ;
  towrite[2]= Size >> 8 ;
  towrite[3]= Size ;
  WriteBytes(towrite, 4) ;
  WriteBytes(Source, Size) ;
  towrite[0]= Handle >> 8 ;
  towrite[1]= Handle ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_ClearAttributes(uint16_t  Handle, uint16_t  Index, Picaso::ImageAttribute Value)
{
  char  towrite[8] ;
  towrite[0]= F_img_ClearAttributes >> 8 ;
  towrite[1]= F_img_ClearAttributes & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  towrite[6]= Value >> 8 ;
  towrite[7]= Value ;
  WriteBytes(towrite, 8) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_Darken(uint16_t  Handle, uint16_t  Index)
{
  char  towrite[6] ;
  towrite[0]= F_img_Darken >> 8 ;
  towrite[1]= F_img_Darken & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_Disable(uint16_t  Handle, uint16_t  Index)
{
  char  towrite[6] ;
  towrite[0]= F_img_Disable >> 8 ;
  towrite[1]= F_img_Disable & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_Enable(uint16_t  Handle, uint16_t  Index)
{
  char  towrite[6] ;
  towrite[0]= F_img_Enable >> 8 ;
  towrite[1]= F_img_Enable & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_GetWord(uint16_t  Handle, uint16_t  Index, Picaso::ImageControlOffset Offset )
{
  char  towrite[8] ;
  towrite[0]= F_img_GetWord >> 8 ;
  towrite[1]= F_img_GetWord & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  towrite[6]= Offset  >> 8 ;
  towrite[7]= Offset  ;
  WriteBytes(towrite, 8) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_Lighten(uint16_t  Handle, uint16_t  Index)
{
  char  towrite[6] ;
  towrite[0]= F_img_Lighten >> 8 ;
  towrite[1]= F_img_Lighten & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_SetAttributes(uint16_t  Handle, uint16_t  Index, Picaso::ImageAttribute Value)
{
  char  towrite[8] ;

  towrite[0]= F_img_SetAttributes >> 8 ;
  towrite[1]= F_img_SetAttributes & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  towrite[6]= Value >> 8 ;
  towrite[7]= Value ;
  WriteBytes(towrite, 8) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_SetPosition(uint16_t  Handle, uint16_t  Index, uint16_t  Xpos, uint16_t  Ypos)
{
  char  towrite[10] ;

  towrite[0]= F_img_SetPosition >> 8 ;
  towrite[1]= F_img_SetPosition & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  towrite[6]= Xpos >> 8 ;
  towrite[7]= Xpos ;
  towrite[8]= Ypos >> 8 ;
  towrite[9]= Ypos ;
  WriteBytes(towrite, 10) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_SetWord(uint16_t  Handle, uint16_t  Index, Picaso::ImageControlOffset Offset , uint16_t  Word)
{
  char  towrite[10] ;

  towrite[0]= F_img_SetWord >> 8 ;
  towrite[1]= F_img_SetWord & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  towrite[6]= Offset  >> 8 ;
  towrite[7]= Offset  ;
  towrite[8]= Word >> 8 ;
  towrite[9]= Word ;
  WriteBytes(towrite, 10) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_Show(uint16_t  Handle, uint16_t  Index)
{
  char  towrite[6] ;
  towrite[0]= F_img_Show >> 8 ;
  towrite[1]= F_img_Show & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::img_Touched(uint16_t  Handle, uint16_t  Index)
{
  char  towrite[6] ;
  towrite[0]= F_img_Touched >> 8 ;
  towrite[1]= F_img_Touched & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= Index >> 8 ;
  towrite[5]= Index ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::media_Flush()
{
  char  towrite[2] ;
  towrite[0]= F_media_Flush >> 8 ;
  towrite[1]= F_media_Flush ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::media_Image(uint16_t  X, uint16_t  Y)
{
  char  towrite[6] ;

  towrite[0]= F_media_Image >> 8 ;
  towrite[1]= F_media_Image ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  WriteBytes(towrite, 6) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::media_Init()
{
  char  towrite[2] ;

  towrite[0]= F_media_Init >> 8 ;
  towrite[1]= F_media_Init ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::media_RdSector(char*  SectorIn)
{
  char  towrite[2] ;

  towrite[0]= F_media_RdSector >> 8 ;
  towrite[1]= F_media_RdSector ;
  WriteBytes(towrite, 2) ;
  return GetAckResSector(SectorIn) ;
}

uint16_t uLCD_4D_Picaso::media_ReadByte()
{
  char  towrite[2] ;
  towrite[0]= F_media_ReadByte >> 8 ;
  towrite[1]= F_media_ReadByte ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::media_ReadWord()
{
  char  towrite[2] ;
  towrite[0]= F_media_ReadWord >> 8 ;
  towrite[1]= F_media_ReadWord ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::media_SetAdd(uint16_t  HiWord, uint16_t  LoWord)
{
  char  towrite[6] ;

  towrite[0]= F_media_SetAdd >> 8 ;
  towrite[1]= F_media_SetAdd ;
  towrite[2]= HiWord >> 8 ;
  towrite[3]= HiWord ;
  towrite[4]= LoWord >> 8 ;
  towrite[5]= LoWord ;
  WriteBytes(towrite, 6) ;
  GetAck() ;
}

void uLCD_4D_Picaso::media_SetSector(uint16_t  HiWord, uint16_t  LoWord)
{
  char  towrite[6] ;

  towrite[0]= F_media_SetSector >> 8 ;
  towrite[1]= F_media_SetSector ;
  towrite[2]= HiWord >> 8 ;
  towrite[3]= HiWord ;
  towrite[4]= LoWord >> 8 ;
  towrite[5]= LoWord ;
  WriteBytes(towrite, 6) ;
  GetAck() ;
}

void uLCD_4D_Picaso::media_Video(uint16_t  X, uint16_t  Y)
{
  char  towrite[6] ;

  towrite[0]= F_media_Video >> 8 ;
  towrite[1]= F_media_Video ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  WriteBytes(towrite, 6) ;
  GetAck() ;
}

void uLCD_4D_Picaso::media_VideoFrame(uint16_t  X, uint16_t  Y, uint16_t  Framenumber)
{
  char  towrite[8] ;

  towrite[0]= F_media_VideoFrame >> 8 ;
  towrite[1]= F_media_VideoFrame ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Framenumber >> 8 ;
  towrite[7]= Framenumber ;
  WriteBytes(towrite, 8) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::media_WriteByte(uint16_t  Byte)
{
  char  towrite[4] ;

  towrite[0]= F_media_WriteByte >> 8 ;
  towrite[1]= F_media_WriteByte ;
  towrite[2]= Byte >> 8 ;
  towrite[3]= Byte ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::media_WriteWord(uint16_t  Word)
{
  char  towrite[4] ;

  towrite[0]= F_media_WriteWord >> 8 ;
  towrite[1]= F_media_WriteWord ;
  towrite[2]= Word >> 8 ;
  towrite[3]= Word ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::media_WrSector(char*  SectorOut)
{
  char  towrite[2] ;

  towrite[0]= F_media_WrSector >> 8 ;
  towrite[1]= F_media_WrSector ;
  WriteBytes(towrite, 2) ;
  WriteBytes(SectorOut, 512) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::mem_Free(uint16_t  Handle)
{
  char  towrite[4] ;

  towrite[0]= F_mem_Free >> 8 ;
  towrite[1]= F_mem_Free & 0xFF;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::mem_Heap()
{
  char  towrite[2] ;

  towrite[0]= F_mem_Heap >> 8 ;
  towrite[1]= F_mem_Heap & 0xFF;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::peekM(uint16_t  Address)
{
  char  towrite[4] ;

  towrite[0]= F_peekM >> 8 ;
  towrite[1]= F_peekM ;
  towrite[2]= Address >> 8 ;
  towrite[3]= Address ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::pin_HI(Picaso::Pin Pin)
{
  char  towrite[4] ;

  towrite[0]= F_pin_HI >> 8 ;
  towrite[1]= F_pin_HI ;
  towrite[2]= Pin >> 8 ;
  towrite[3]= Pin ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::pin_LO(Picaso::Pin Pin)
{
  char  towrite[4] ;

  towrite[0]= F_pin_LO >> 8 ;
  towrite[1]= F_pin_LO ;
  towrite[2]= Pin >> 8 ;
  towrite[3]= Pin ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::pin_Read(Picaso::Pin Pin)
{
  char  towrite[4] ;

  towrite[0]= F_pin_Read >> 8 ;
  towrite[1]= F_pin_Read ;
  towrite[2]= Pin >> 8 ;
  towrite[3]= Pin ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::pin_Set(uint16_t  Mode, Picaso::Pin Pin)
{
  char  towrite[6] ;

  towrite[0]= F_pin_Set >> 8 ;
  towrite[1]= F_pin_Set ;
  towrite[2]= Mode >> 8 ;
  towrite[3]= Mode ;
  towrite[4]= Pin >> 8 ;
  towrite[5]= Pin ;
  WriteBytes(towrite, 6) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::pokeM(uint16_t  Address, uint16_t  WordValue)
{
  char  towrite[6] ;

  towrite[0]= F_pokeM >> 8 ;
  towrite[1]= F_pokeM ;
  towrite[2]= Address >> 8 ;
  towrite[3]= Address ;
  towrite[4]= WordValue >> 8 ;
  towrite[5]= WordValue ;
  WriteBytes(towrite, 6) ;
  GetAck() ;
}

void uLCD_4D_Picaso::putCH(uint16_t  WordChar)
{
  char  towrite[4] ;

  towrite[0]= F_putCH >> 8 ;
  towrite[1]= F_putCH ;
  towrite[2]= WordChar >> 8 ;
  towrite[3]= WordChar ;
  WriteBytes(towrite, 4) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::putStr(char *  InString)
{
  char  towrite[2] ;

  towrite[0]= F_putstr >> 8 ;
  towrite[1]= F_putstr ;
  WriteBytes(towrite, 2) ;
  WriteChars(InString) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::snd_BufSize(uint16_t  Bufsize)
{
  char  towrite[4] ;

  towrite[0]= F_snd_BufSize >> 8 ;
  towrite[1]= F_snd_BufSize & 0xFF;
  towrite[2]= Bufsize >> 8 ;
  towrite[3]= Bufsize ;
  WriteBytes(towrite, 4) ;
  GetAck() ;
}

void uLCD_4D_Picaso::snd_Continue()
{
  char  towrite[2] ;

  towrite[0]= F_snd_Continue >> 8 ;
  towrite[1]= F_snd_Continue & 0xFF;
  WriteBytes(towrite, 2) ;
  GetAck() ;
}

void uLCD_4D_Picaso::snd_Pause()
{
  char  towrite[2] ;

  towrite[0]= F_snd_Pause >> 8 ;
  towrite[1]= F_snd_Pause & 0xFF;
  WriteBytes(towrite, 2) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::snd_Pitch(uint16_t  Pitch)
{
  char  towrite[4] ;

  towrite[0]= F_snd_Pitch >> 8 ;
  towrite[1]= F_snd_Pitch & 0xFF;
  towrite[2]= Pitch >> 8 ;
  towrite[3]= Pitch ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::snd_Playing()
{
  char  towrite[2] ;

  towrite[0]= F_snd_Playing >> 8 ;
  towrite[1]= F_snd_Playing & 0xFF;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::snd_Stop()
{
  char  towrite[2] ;

  towrite[0]= F_snd_Stop >> 8 ;
  towrite[1]= F_snd_Stop & 0xFF;
  WriteBytes(towrite, 2) ;
  GetAck() ;
}

void uLCD_4D_Picaso::snd_Volume(uint16_t  Volume)
{
  char  towrite[4] ;

  towrite[0]= F_snd_Volume >> 8 ;
  towrite[1]= F_snd_Volume & 0xFF;
  towrite[2]= Volume >> 8 ;
  towrite[3]= Volume ;
  WriteBytes(towrite, 4) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::sys_Sleep(uint16_t  Units)
{
  char  towrite[4] ;

  towrite[0]= F_sys_Sleep >> 8 ;
  towrite[1]= F_sys_Sleep & 0xFF;
  towrite[2]= Units >> 8 ;
  towrite[3]= Units ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::touch_DetectRegion(uint16_t  X1, uint16_t  Y1, uint16_t  X2, uint16_t  Y2)
{
  char  towrite[10] ;

  towrite[0]= F_touch_DetectRegion >> 8 ;
  towrite[1]= F_touch_DetectRegion & 0xFF;
  towrite[2]= X1 >> 8 ;
  towrite[3]= X1 ;
  towrite[4]= Y1 >> 8 ;
  towrite[5]= Y1 ;
  towrite[6]= X2 >> 8 ;
  towrite[7]= X2 ;
  towrite[8]= Y2 >> 8 ;
  towrite[9]= Y2 ;
  WriteBytes(towrite, 10) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::touch_Get(uint16_t  Mode)
{
  char  towrite[4] ;

  towrite[0]= F_touch_Get >> 8 ;
  towrite[1]= F_touch_Get & 0xFF;
  towrite[2]= Mode >> 8 ;
  towrite[3]= Mode ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::touch_Set(uint16_t  Mode)
{
  char  towrite[4] ;

  towrite[0]= F_touch_Set >> 8 ;
  towrite[1]= F_touch_Set & 0xFF;
  towrite[2]= Mode >> 8 ;
  towrite[3]= Mode ;
  WriteBytes(towrite, 4) ;
  GetAck() ;
}

Picaso::TextAttribute uLCD_4D_Picaso::txt_Attributes(Picaso::TextAttribute Attribs)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Attributes >> 8 ;
  towrite[1]= F_txt_Attributes ;
  towrite[2]= Attribs >> 8 ;
  towrite[3]= Attribs ;
  WriteBytes(towrite, 4) ;
  return (Picaso::TextAttribute) GetAckResp() ;
}

Picaso::Color uLCD_4D_Picaso::txt_BGcolour(Picaso::Color Color)
{
  char  towrite[4] ;

  towrite[0]= F_txt_BGcolour >> 8 ;
  towrite[1]= F_txt_BGcolour ;
  towrite[2]= Color >> 8 ;
  towrite[3]= Color ;
  WriteBytes(towrite, 4) ;
  return (Picaso::Color) GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Bold(uint16_t  Bold)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Bold >> 8 ;
  towrite[1]= F_txt_Bold ;
  towrite[2]= Bold >> 8 ;
  towrite[3]= Bold ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

Picaso::Color uLCD_4D_Picaso::txt_FGcolour(Picaso::Color Color)
{
  char  towrite[4] ;

  towrite[0]= F_txt_FGcolour >> 8 ;
  towrite[1]= F_txt_FGcolour ;
  towrite[2]= Color >> 8 ;
  towrite[3]= Color ;
  WriteBytes(towrite, 4) ;
  return (Picaso::Color) GetAckResp() ;
}

Picaso::Font uLCD_4D_Picaso::txt_FontID(Picaso::Font FontNumber)
{
  char  towrite[4] ;

  towrite[0]= F_txt_FontID >> 8 ;
  towrite[1]= F_txt_FontID ;
  towrite[2]= FontNumber >> 8 ;
  towrite[3]= FontNumber ;
  WriteBytes(towrite, 4) ;
  return (Picaso::Font) GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Height(uint16_t  Multiplier)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Height >> 8 ;
  towrite[1]= F_txt_Height ;
  towrite[2]= Multiplier >> 8 ;
  towrite[3]= Multiplier ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Inverse(uint16_t  Inverse)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Inverse >> 8 ;
  towrite[1]= F_txt_Inverse ;
  towrite[2]= Inverse >> 8 ;
  towrite[3]= Inverse ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Italic(uint16_t  Italic)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Italic >> 8 ;
  towrite[1]= F_txt_Italic ;
  towrite[2]= Italic >> 8 ;
  towrite[3]= Italic ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

void uLCD_4D_Picaso::txt_MoveCursor(uint16_t  Line, uint16_t  Column)
{
  char  towrite[6] ;

  towrite[0]= F_txt_MoveCursor >> 8 ;
  towrite[1]= F_txt_MoveCursor ;
  towrite[2]= Line >> 8 ;
  towrite[3]= Line ;
  towrite[4]= Column >> 8 ;
  towrite[5]= Column ;
  WriteBytes(towrite, 6) ;
  GetAck() ;
}

Picaso::TextOpacity uLCD_4D_Picaso::txt_Opacity(Picaso::TextOpacity TransparentOpaque)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Opacity >> 8 ;
  towrite[1]= F_txt_Opacity ;
  towrite[2]= TransparentOpaque >> 8 ;
  towrite[3]= TransparentOpaque ;
  WriteBytes(towrite, 4) ;
  return (Picaso::TextOpacity) GetAckResp() ;
}

void uLCD_4D_Picaso::txt_Set(Picaso::TextSetFunc Func, uint16_t  Value)
{
  char  towrite[6] ;

  towrite[0]= F_txt_Set >> 8 ;
  towrite[1]= F_txt_Set ;
  towrite[2]= Func >> 8 ;
  towrite[3]= Func ;
  towrite[4]= Value >> 8 ;
  towrite[5]= Value ;
  WriteBytes(towrite, 6) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::txt_Underline(uint16_t  Underline)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Underline >> 8 ;
  towrite[1]= F_txt_Underline ;
  towrite[2]= Underline >> 8 ;
  towrite[3]= Underline ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Width(uint16_t  Multiplier)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Width >> 8 ;
  towrite[1]= F_txt_Width ;
  towrite[2]= Multiplier >> 8 ;
  towrite[3]= Multiplier ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Wrap(uint16_t  Position)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Wrap >> 8 ;
  towrite[1]= F_txt_Wrap ;
  towrite[2]= Position >> 8 ;
  towrite[3]= Position ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Xgap(uint16_t  Pixels)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Xgap >> 8 ;
  towrite[1]= F_txt_Xgap ;
  towrite[2]= Pixels >> 8 ;
  towrite[3]= Pixels ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::txt_Ygap(uint16_t  Pixels)
{
  char  towrite[4] ;

  towrite[0]= F_txt_Ygap >> 8 ;
  towrite[1]= F_txt_Ygap ;
  towrite[2]= Pixels >> 8 ;
  towrite[3]= Pixels ;
  WriteBytes(towrite, 4) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::file_CallFunction(uint16_t  Handle, uint16_t  ArgCount, uint16_t*  Args)
{
  char  towrite[6] ;

  towrite[0]= F_file_CallFunction >> 8 ;
  towrite[1]= F_file_CallFunction ;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  towrite[4]= ArgCount >> 8 ;
  towrite[5]= ArgCount ;
  WriteBytes(towrite, 6) ;
  WriteWords(Args, ArgCount) ;
  return WaitForAck() ;
}

uint16_t uLCD_4D_Picaso::sys_GetModel(char *  ModelStr)
{
  char  towrite[2] ;

  towrite[0]= F_sys_GetModel >> 8 ;
  towrite[1]= F_sys_GetModel ;
  WriteBytes(towrite, 2) ;
  return GetAckResStr(ModelStr) ;
}

uint16_t uLCD_4D_Picaso::sys_GetVersion()
{
  char  towrite[2] ;

  towrite[0]= F_sys_GetVersion >> 8 ;
  towrite[1]= F_sys_GetVersion ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::sys_GetPmmC()
{
  char  towrite[2] ;

  towrite[0]= F_sys_GetPmmC >> 8 ;
  towrite[1]= F_sys_GetPmmC ;
  WriteBytes(towrite, 2) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::writeString(uint16_t  Handle, char *  StringOut)
{
  char  towrite[4] ;

  towrite[0]= F_writeString >> 8 ;
  towrite[1]= F_writeString ;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  WriteChars(StringOut) ;
  return GetAckResp() ;
}

uint16_t uLCD_4D_Picaso::readString(uint16_t  Handle, char *  StringIn)
{
  char  towrite[4] ;

  towrite[0]= F_readString >> 8 ;
  towrite[1]= F_readString ;
  towrite[2]= Handle >> 8 ;
  towrite[3]= Handle ;
  WriteBytes(towrite, 4) ;
  return GetAckResStr(StringIn) ;
}

void uLCD_4D_Picaso::blitComtoDisplay(uint16_t  X, uint16_t  Y, uint16_t  Width, uint16_t  Height, char*  Pixels)
{
  char  towrite[10] ;

  towrite[0]= F_blitComtoDisplay >> 8 ;
  towrite[1]= F_blitComtoDisplay ;
  towrite[2]= X >> 8 ;
  towrite[3]= X ;
  towrite[4]= Y >> 8 ;
  towrite[5]= Y ;
  towrite[6]= Width >> 8 ;
  towrite[7]= Width ;
  towrite[8]= Height >> 8 ;
  towrite[9]= Height ;
  WriteBytes(towrite, 10) ;
  WriteBytes(Pixels, Width*Height*2) ;
  GetAck() ;
}

uint16_t uLCD_4D_Picaso::file_FindFirstRet(char *  Filename, char *  StringIn)
{
  char  towrite[2] ;

  towrite[0]= F_file_FindFirstRet >> 8 ;
  towrite[1]= F_file_FindFirstRet ;
  WriteBytes(towrite, 2) ;
  WriteChars(Filename) ;
  return GetAckResStr(StringIn) ;
}

uint16_t uLCD_4D_Picaso::file_FindNextRet(char *  StringIn)
{
  char  towrite[2] ;

  towrite[0]= F_file_FindNextRet >> 8 ;
  towrite[1]= F_file_FindNextRet ;
  WriteBytes(towrite, 2) ;
  return GetAckResStr(StringIn) ;
}

void uLCD_4D_Picaso::setbaudWait(Picaso::BaudRate Newrate)
{
  char  towrite[4] ;

  towrite[0]= F_setbaudWait >> 8 ;
  towrite[1]= F_setbaudWait ;
  towrite[2]= Newrate >> 8 ;
  towrite[3]= Newrate ;
  WriteBytes(towrite, 4) ;
  //printf("Baud command: %2x%2x%2x%2x\r\n", towrite[0], towrite[1], towrite[2], towrite[3]);
  SetThisBaudrate(Newrate) ; // change this systems baud rate to match new display rate, ACK is 100ms away
  GetAck() ;
}

int uLCD_4D_Picaso::_putc(int c)
{
	putCH(c); //your new LCD put to print an ASCII character on LCD
    return 0;
}
int uLCD_4D_Picaso::_getc() 
{
    return -1;
}

/*
 * Conpound 4D Routines Ends here
*/

