#include "unzen_hal.h"

namespace unzen 
{
        // Set up I2S peripheral to ready to start.
        // By this HAL, the I2S have to become : 
        // - slave mode
        // - clock must be ready
    void hal_i2s_setup(void)
    {
                // Setup Audio Clock. Feed the core clock (PLL1, 204Mhz) to audio.
        LPC_CGU->BASE_CLK[CLK_BASE_APLL] = 
                            0 << 1 |    // Power down. 1:Power Down, 0:Enabled
                            1 <<11 |    // Enable auto clocking. 1:Enable.
                            9 <<24 ;    // 9: PLL1
            
            // Assert DAI reset
        LPC_I2S0->DAO |= 1 << 4;    // I2S_DAI_RESET;
        LPC_I2S0->DAI |= 1 << 4;    // I2S_DAI_RESET;
            // Deassert DAI reset
        LPC_I2S0->DAO &= ~ ( 1<<4 ); // I2S_DAI_RESET;
        LPC_I2S0->DAI &= ~ ( 1<<4 ); // I2S_DAI_RESET;
            // Assert DAI stop
        LPC_I2S0->DAO |= 1 << 3;    // I2S_DAI_STOP;
        LPC_I2S0->DAI |= 1 << 3;    // I2S_DAI_STOP;
                
            // Kill all DMA
        LPC_I2S0->DMA2 = 0;
        LPC_I2S0->DMA1 = 0;
            // Kill all IRQ
        LPC_I2S0->IRQ = 0;
            // Kill all clocks
        LPC_I2S0->RXRATE = 0;
        LPC_I2S0->TXRATE = 0;
            // Bit rate must be 0 for slave mode.
        LPC_I2S0->RXBITRATE = 0;    
        LPC_I2S0->TXBITRATE = 0;
            // Clear mode setting
        LPC_I2S0->TXMODE = 0;
        LPC_I2S0->RXMODE = 0;


            // Configure DA0
        LPC_I2S0->DAO =
                3 <<  0 |   // word width, 3:32bit mode        
                0 <<  2 |   // Mono, 1:mono, 0:stereo
                1 <<  3 |   // STOP, 1:stop
                0 <<  4 |   // RESET, 1:reset
                1 <<  5 |   // WSEL, 1:Slave, 0:Master
                1 <<  6 ;   // MUTE, 1:mute, 0:normal 

            // Configure DAI
        LPC_I2S0->DAI =
                3 <<  0 |   // word width, 3:32bit mode        
                0 <<  2 |   // Mono, 1:mono, 0:stereo
                1 <<  3 |   // STOP, 1:stop
                0 <<  4 |   // RESET, 1:reset
                1 <<  5 |   // WSEL, 1:Slave, 0:Master
                31 <<  6;   // WS halfperiod. Not sure what I shoud do when the slave mode.
                
            // Configure IRQ At this moment, IRQ is disabled. 
        LPC_I2S0->IRQ =
                0 << 0  |   // RX IRQ Enable, 1:Enable, 0:Disable
                0 << 1  |   // TX IRQ Enable, 1:Enable, 0:Disable
                2 << 8  |   // RX DEPTH IRQ length for triggering interrupt  
                0 << 16 ;   // TX DEPTH IRQ length for triggering interrupt

            // Set RX module as slave operation to the external signal
        LPC_I2S0->RXMODE =
                0 << 0  |   // RXCLKSEL, 0:SCLK input. The reference manual says this is fractional devider, but in slave mode, it is SCLK.
                0 << 2  |   // RX4PIN, 1:4pin mode, 0:Clock by RX block itself
                0 << 3  ;   // RXMCENA, 1:Master clock output, 0:No output
                
            // SEt TX module as slave operation to the RX module (4PIN mode )
        LPC_I2S0->TXMODE =
                0 << 0  |   // TXCLKSEL, 0:SCLK input. Ignored by 4 pin mode.
                1 << 2  |   // TX4PIN, 1:4pin mode, 0:Clock by TX block itself
                0 << 3  ;   // TXMCENA, 1:Master clock output, 0:No output


            //  Fill up tx FIO by 3 stereo samples.
        hal_put_i2s_tx_data( 0 ); // left
        hal_put_i2s_tx_data( 0 ); // right
        hal_put_i2s_tx_data( 0 ); // left
        hal_put_i2s_tx_data( 0 ); // right
        hal_put_i2s_tx_data( 0 ); // left
        hal_put_i2s_tx_data( 0 ); // right

    }
    
        // Pin configuration and sync with WS signal
    void hal_i2s_pin_config_and_wait_ws(void)
    {
            // See UM10503 LPC4300 UM Rev 2.1 section 16.2.3, chapter 19
            // See https://developer.mbed.org/platforms/LPCXpresso4337/
            // See mbed src lpc43xx.h http://preview.tinyurl.com/lpc43xxh
            
            
            // setup WS pin as GPIO input
        LPC_SCU->SFSP[6][1] = SCU_MODE_INBUFF_EN;       // P6_1, function 0. GPIO3[0]
        
            // if WS is 1, wait for WS (GPIO3_0) becomes 0
        while ( LPC_GPIO_PORT->B[3][0] == 1 )
            ;
            
            // and then, if WS is 0, wait for WS (GPIO3_0) becomes 1
        while ( LPC_GPIO_PORT->B[3][0] == 0 )
            ;
            
        // Now, we are at the rising edge of WS. 
        // We can setup the I2S without worry of the TX/RX shift.

            // setup I2S pin configuration.
            // for input, bit 6 have to be 1 ( input buffer enable )
        LPC_SCU->SFSP[3][5] = 5;                        // P3_5, function 5. IS20_TX_SDA
        LPC_SCU->SFSP[6][0] = 4 | SCU_MODE_INBUFF_EN;   // P6_0, function 4. IS20_RX_SCK
        LPC_SCU->SFSP[6][1] = 3 | SCU_MODE_INBUFF_EN;   // P6_1, function 3. IS20_RX_WS
        LPC_SCU->SFSP[3][2] = 1 | SCU_MODE_INBUFF_EN;   // P3_2, function 1. IS20_RX_SDA
    }

    
        // Start I2S transfer. Interrupt starts  
    void hal_i2s_start(void)
    {
            //Clear STOP,RESET and MUTE bit
        LPC_I2S0->DAO &= ~(1 << 3);    // release I2S_DAO_STOP;
        LPC_I2S0->DAI &= ~(1 << 3);    // release I2S_DAI_STOP;
        
        LPC_I2S0->DAO &= ~(1 << 6);    // release I2S_DAO_MUTE;
        LPC_I2S0->IRQ |= 1 << 0;       // set I2S RX IRQ enable
    }
 
    IRQn_Type hal_get_i2s_irq_id(void)
    {
        return I2S0_IRQn;
    }
    
    
    IRQn_Type hal_get_process_irq_id(void)
    {
        return ( IRQn_Type )30;   // LPC4300's unsed interrupt
    }
    
    
         // The returned value must be compatible with CMSIS NVIC_SetPriority() API. That mean, it is integer like 0, 1, 2...
    unsigned int hal_get_i2s_irq_priority_level(void)
    {
           // LPC4300 has 3 bits priority field. So, heighest is 0, lowest is 7.
           // But CMSIS NVIC_SetPriority() inverse the priority of the interupt ( F**k! )
           // So, 7 is heighest, 0 is lowerest in CMSIS.
           // setting 6 as i2s irq priority allows, some other interrupts are higher 
           // and some others are lower than i2s irq priority.
        return 6;
    }


        // The returned value must be compatible with CMSIS NVIC_SetPriority() API. That mean, it is integer like 0, 1, 2...
    unsigned int hal_get_process_irq_priority_level(void)
    {
           // LPC4300 has 3 bits priority field. So, heighest is 0, lowest is 7.
           // But CMSIS NVIC_SetPriority() inverse the priority of the interupt ( S**t! )
           // So, 7 is heighest, 0 is lowerest in CMSIS.
           // setting 1 as process priority allows, some other interrupts are higher 
           // and some other interrupts are lower then process priority.
        return 1;   
    }
 
        // LPC4337 transferes 2 workd ( left and right ) for each interrupt.
    unsigned int hal_data_per_sample(void)
    {
        return 2;
    }

        // return true when the sample parameter is ready to read.
        // return false when the sample is not ready to read.
    void hal_get_i2s_rx_data( int & sample)
    {
        sample = LPC_I2S0->RXFIFO;
    }
    
        // put a sample to I2S TX data regisger
    void hal_put_i2s_tx_data( int sample )
    {
        LPC_I2S0->TXFIFO = sample;
    }
}


