/**
* \file ukifune.h
* \brief header file for the ukifune UI module.
* \arthur Seiichi Horie
* \date 22/Jan/2017
* \details
* First of all, call init() API before calling other API.
*/


#ifndef _UKIFUNE_H_
#define _UKIFUNE_H_

#include <cstdint>
#include "unzen.h"

/**
* \brief API to control UI board "Ukifune"
*/
namespace ukifune 
{
    /**
    * \brief LED accessing names.
    */
    enum LED{
            led1_1 = 0,     ///< LED1_1
            led1_2,         ///< LED1_2
            led1_3,         ///< LED1_3
            led1_4,         ///< LED1_4
            led2_1,         ///< LED2_1
            led2_2,         ///< LED2_2 
            led2_3,         ///< LED2_3
            led2_4,         ///< LED2_4
            led3_1,         ///< LED3_1
            led3_2,         ///< LED3_2
            led3_3,         ///< LED3_3
            led3_4         ///< LED3_4
        } ;
    
    /**
    * \brief Buttun Switch accesing names.
    */
    enum SWITCH{
            swm1 = 0,   ///< button switch SWM1
            swm2,       ///< button switch SWM2
            swm3,       ///< button switch SWM3
            swk1,       ///< button switch SWK1
            swk2,       ///< button switch SWK2
            swk3,       ///< button switch SWK3
            swk4,       ///< button switch SWK4
            swk5,       ///< button switch SWK5
            swk6,       ///< button switch SWK6
            swk7,       ///< button switch SWK7
            swk8,       ///< button switch SWK8
            swk9,       ///< button switch SWK9
            swk10,       ///< button switch SWK10
            swk11,       ///< button switch SWK11
            swk12,       ///< button switch SWK12
            swk13       ///< button switch SWK13
        } ;
    
        /**
        * \brief Initialize internal variable of module. Have to call once before all other processing of Ukifune.
        * \pram audio Pointer to the unzen audio framework. Test points are registered as interrupt / process status flag.
        */
    extern void init( unzen::Framework *audio);
    
        /**
        * \breif get the state of the button switches. 
        * \param pushing   [out] Bit map of the switch which is changing state from released to pushed. "1" is pushed. 
        * \param releasing [out] Bit map of the switch which is changing state from pushed to released. "1" is releasing.
        * \param holding   [out] Bit map of the switch which is kept pushed. "1" is holding.
        * \details
        * The bit position in the bit map are represented by enum SWITCH.
        */
    extern void get_button_state(
                        uint32_t & pushing, 
                        uint32_t & releasing, 
                        uint32_t & holding 
                        );
    
        /**
        * \brief Turn the specific LED ON.
        * \param LED The LED to be on.
        */
    extern void turn_led_on( enum LED );

        /**
        * \brief Turn the specific LED off.
        * \param LED The LED to be off.
        */
    extern void turn_led_off( enum LED );
    
        /**
        * \brief Toggle the specific LED.
        * \param LED The LED to be toggled.
        */
    extern void toggle_led( enum LED );

        /**
        * \brief Get the state of LED.
        * \param LED The LED to be checked.
        * \return "non-zero" On.
        * \return 0 Off.
        */
    extern int get_led_state( enum LED );
    
        /**
        * \brief get the level of volume.
        * \param number the numver of volueme [0,7]
        * \return the volume level. [0.0, 1.0]
        */
    extern float get_volume( int number ); 

        /**
        * \brief Update the internal. state of ukifune module. Have better to call around every 30mS
        * 
        */
    extern void tick(void);
    
}

#endif // _UKIFUNE_H_
