#ifndef _SIGNAL_PROCESSING_H_
#define _SIGNAL_PROCESSING_H_

#include "amakusa.h"

#define SAMPLING_FREQUENCY 48000

enum wave_form { triangle, square };
enum svf_mode { lpf, hpf, bpf };
enum eg_state { release, attack, decay };

    // Variable Frequency Oscillator. Only square and triangle
class VFO {
public:
    VFO( int32_t block_size );
    virtual ~VFO();
    void run(           
        float out_buffer[]         // place to write the right output samples
        );
           
        // parameter settings
    void set_frequency( float freq );     // unit is Hz.
    void set_Fs( int Fs );              // unit is Hz.
    void set_duty_cycle( float duty );  // 0 ... 0.5
    void set_wave_form( wave_form form );
private:
    int32_t block_size;

        // control variables.
    float frequency;          // VFO frequency [Hz]
    int Fs;                 // sampling Frequency [Hz]
    float duty_cycle;       // VFO duty cycle. 0 ... 0.5
    wave_form form;         // form of the wave form.
    
        // internal variable.
    float current_phase;      // internal variable of VFO.
    float half_way;           // change point by duty cycle. ( period * duty_cycle ).
    float rising_rate;
    float falling_rate;
    
    void update_parameters(void);    // call one of the parameter is changed.
};

    // Blocking DC 
class DCBlocker : public amakusa::AbstractFilter {
public:
    DCBlocker( uint32_t blockSize );
    virtual void run( float *pSrc, float *pDst );
private:    
    float x_last;
    float y_last;
};

    // State Variable Filter
    // The f * f_factor must be < Fs/6.
class SVFilter : public amakusa::AbstractFilter {
public:
    SVFilter( uint32_t a_block_size );
    virtual void run( float *pSrc, float *pDst);
    void set_Q( float Q );              // real Q factor [ 0.5 ... inf ]
    void set_Fs( int new_Fs );              // Hz
    void set_fc( int new_fc );                // Hz
    void set_f_factor( float new_f_factor );   
    void set_mode( svf_mode new_mode );
private:    
        // internal variable
    float d1, d2;                       // delay 1, delay 2;
    float q, f;                         // q = 1/Q, f = 2 * sin( fc*f_factor*pi/Fs );
        // parameter set by method
    int Fs, fc;                             // sampling frequency and control frequency
    float f_factor;
    svf_mode mode;                          // lpf, hpf, bpf
    void update_parameters( void );
};

class EG {
public:
    EG ( int32_t a_block_size );
    virtual void run( float *pEnvelope );
    void on(void);
    void off(void);
    void set_attack( float attack );                // [0,1.0]
    void set_decay( float attack );                 // [0,1.0]
    void set_sustain( float sustain );              // [0,1.0]
    void set_release ( float attack );              // [0,1.0]
private:
    float current_level, sustain_level;
    float attack_time_constant, decay_time_constant, release_time_constant;
    eg_state  state;
    int32_t block_size;
};

    // Monophonic synthsizer class
class Monophonic {
public:
    Monophonic( unsigned int  block_size );
    virtual ~Monophonic(void);
    void run(           
        float out_buffer[]         // place to write the right output samples
        );
    void set_Fs( int Fs );                  // unit is Hz.
    void set_vfo_frequency( float freq );     // unit is Hz.
    void set_vfo_duty_cycle( float duty );  // 0 ... 0.5
    void set_vfo_wave_form( wave_form form );
    void set_filter_mode( svf_mode mode );
    void set_filter_Q( float Q );           // 0.5 .. inf
    void set_filter_f_factor( float f_factor ); // 0.0..1.0
    void eg_on(void);
    void eg_off(void);
    void set_eg_attack( float attack );                // [0,1.0]
    void set_eg_decay( float decay  );                 // [0,1.0]
    void set_eg_sustain( float sustain );              // [0,1.0]
    void set_eg_release ( float release );             // [0,1.0]
private:
    VFO *vfo;
    DCBlocker *dc_blocker;
    SVFilter *sv_filter;
    EG *eg;
    amakusa::LimitterLinAtan *limitter;
    float *work_buf_a, *work_buf_b;
    int32_t block_size;
};


    // User Signal processing Class 
class SignalProcessing {
public:
        // essential members. Do not touch
    SignalProcessing( unsigned int  block_size );
    void run(           
        float rx_left_buffer[],     // array of the left input samples
        float rx_right_buffer[],    // array of the right input samples
        float tx_left_buffer[],     // place to write the left output samples
        float tx_right_buffer[],    // place to write the right output samples
        unsigned int block_size     // block size [sample]
        );
           
        // project depenedent members.
    void set_volume( float vol );
    void set_Fs( int Fs );                      // unit is Hz.
    void set_vfo_frequency( float freq );         // unit is Hz.
    void set_vfo_duty_cycle( float duty );      // 0 ... 1.0
    void set_vfo_wave_form( wave_form form );
    void set_filter_mode( svf_mode mode );
    void set_filter_Q( float Q );               // 0.0..1.0
    void set_filter_f_factor( float f_factor ); // 0.0..1.0
    void eg_on(void);
    void eg_off(void);
    void set_eg_attack( float attack );                // [0,1.0]
    void set_eg_decay( float decay );                  // [0,1.0]
    void set_eg_sustain( float sustain );              // [0,1.0]
    void set_eg_release ( float release );             // [0,1.0]
private:
        // essential members. Do not touch.
    void enter_critical_section(void);
    void leave_critical_section(void);

        // project dependent members.
    float volume_level;     // 0 ... 1.0
    Monophonic * note;
};

#endif
