/*
 * Mbed Library / Akizuki 7-segment LED driver [OSL10564-74HC595-x]
 *  http://akizukidenshi.com/catalog/g/gM-12643/
 *
 * Copyright (c) 2018 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:    January    7th, 2018
 *      Revised:    January   22nd, 2018
 */

#ifndef SEVENSEG_LED_DRVR_H
#define SEVENSEG_LED_DRVR_H

#include "mbed.h"

#define BUFFER_SIZE     32

//extern Serial pc;     // only for dubug

/**
 * @code
 * #include "mbed.h"
 * #include "7segLed_HC595.h"
 *
 * // case for Nucleo-F446RE, F411RE, F401RE
 * //              SPI MOSI, SCLK,port/out,PWM, num of digits
 * SevenSegLed led_7segs(D4, D3,  D5,      D2,  6);
 *
 *  int main() {
 *      int32_t count = 123456U;
 *      while(true) {
 *          led_7segs = count++;
 *          wait(0.1f);
 *      }
 *  }
 * @endcode
 */

class SevenSegLed
{
public:
    /** 7segments LED driver with 74HC595 (SER, SRCLK, RCLK, /OE)
     *  @param sdo (->SER), sclk (->SRCLK), latch (->RCLK), bright (->/OE)
     *  @param sdo must be SPI MOSI and sclk must be same SPI SCLK
     *  @param latch is any output port
     *  @param bright must be PwmOut port
     *  @param num_of_digit -> # of 7seg LED's
     */
    SevenSegLed(PinName sdo, PinName sclk, PinName latch, PinName bright,
                uint8_t num_of_digit);

    /** Set data into 7seg LED's
     *  @param  unsigned data (overflow may occur depend on number of LEDs)
     *  @return none
     */
    void put_num(int32_t dt);

    SevenSegLed& operator= (int value) {
        put_num(value);
        return *this;
    }

    /** Set ascii number into 7seg LED's
     *  @param  ASCII data
     *  @return none
     */
    void put_ascii(const char *data);

    /** Set ascii character strings into 7seg LED's
     *  @param  all ASCII character
     *  @return none
     */
    void put_strings(const char *data);

    /** Set raw data into 7seg LED's
     *  @param  Raw data (Segment element)
     *  @return none
     */
    void put_raw(uint8_t *data);

    /** Set dot into 7seg LED's
     *  @param  position (1= LSB, 2=2nd, 4=3rd, 8=4th, 16, 32, 64 ----)
     *  @return none
     */
    void put_dot(uint16_t position);
    void clear_all_dot(void);

    /** Set Brightness
     *  @param  britness max(1.0f) and min = all LED off(0.0f)
     *  @return none
     */
    void brightness(float bright);

    /** Set Zero Suppress (defalt OFF)
     *  @param  mode=true -> zero suppress ON, mode = false -> OFF
     *  @return none
     */
    void zero_suppress(bool mode);

    /** Set clock frequency
     *  @param  clock frequency (default 1000000Hz(1MHz))
     *  @return none
     */
    void frequency(int32_t freq);

protected:
    SPI         spi_;
    DigitalOut  latch_;
    PwmOut      pwm_;

    uint8_t     buf[BUFFER_SIZE];
    uint8_t     seg[BUFFER_SIZE];
    uint8_t     num;
    uint16_t    dot_inf;
    uint32_t    wrk;
    bool        zerosuppress;
    bool        num_mode;
    float       bright_data;

    void seven_seg_n_digt(void);
    void set_dot(void);
    void spi_out(void);

};

#endif // SEVENSEG_LED_DRVR_H
