#pragma once

#include "mbed.h"
#include "EthernetInterface.h"


enum MessageID {
    MESSAGE_A = 0,
    MESSAGE_B,
    MESSAGE_C,
    MESSAGE_D,
    MESSAGE_E,
    MESSAGE_F,
    MESSAGE_G,
    MESSAGE_H,
    MESSAGE_NONE = -1
};

enum ColorID {
    COLOR_RED = 0,
    COLOR_BLUE,
    COLOR_GREEN,
    COLOR_YELLOW,
    COLOR_NONE = -1
};

enum CourseID {
    COURSE_LEFT = 0,
    COURSE_CENTER,
    COURSE_RIGHT,
    COURSE_TOP,
    COURSE_MIDDLE,
    COURSE_BOTTOM,
    COURSE_NONE = -1
};

class UDPManager {
private:
    EthernetInterface eth;
    UDPSocket sock;
    SocketAddress adr;
    char sendBuff[256];
    char recBuff[256];
    bool isOpen;
    uint32_t index;

    void sendMessage(MessageID msgID, ColorID colorID, CourseID courseID = COURSE_NONE, int contentID = -1) {
        memset(sendBuff, '\0', sizeof(sendBuff));
        
        if (msgID == MESSAGE_NONE) return;
        switch (msgID) {
            case MESSAGE_A:
                strcat(sendBuff, "01");
                break;
            case MESSAGE_B:
                strcat(sendBuff, "02");
                break;
            case MESSAGE_C:
                strcat(sendBuff, "03");
                break;
            case MESSAGE_D:
                strcat(sendBuff, "04");
                break;
        }
        
        // TimeStamp
        sprintf(sendBuff, "%s,%d", sendBuff, index++);
        
        if (colorID == COLOR_NONE) return;
        switch (colorID) {
            case COLOR_RED:
                strcat(sendBuff, ",R");
                break;
            case COLOR_BLUE:
                strcat(sendBuff, ",B");
                break;
            case COLOR_GREEN:
                strcat(sendBuff, ",G");
                break;
            case COLOR_YELLOW:
                strcat(sendBuff, ",Y");
                break;
        }
        
        if (courseID != COURSE_NONE) {
            switch (courseID) {
                case COURSE_LEFT:
                    strcat(sendBuff, ",L");
                    break;
                case COURSE_CENTER:
                    strcat(sendBuff, ",C");
                    break;
                case COURSE_RIGHT:
                    strcat(sendBuff, ",R");
                    break;
                case COURSE_TOP:
                    strcat(sendBuff, ",T");
                    break;
                case COURSE_MIDDLE:
                    strcat(sendBuff, ",M");
                    break;
                case COURSE_BOTTOM:
                    strcat(sendBuff, ",B");
                    break;
            }
        }
        
        if (contentID >= 0) {
            sprintf(sendBuff, "%s,%d", sendBuff, contentID);
        }
        
        nsapi_size_or_error_t err = sock.sendto(adr, sendBuff, strlen(sendBuff));
        if (err < 0) {
            printf("sock.sendto error: %d\n", err);
        }
    }
    
public:
    UDPManager() : adr("255.255.255.255", 12345), isOpen(false), index(0) {}

    bool init() {
        isOpen = false;
        
        int err = eth.connect();
        if (err != 0) return false;

        sock.open(&eth);
        sock.set_blocking(false);
        sock.bind(12345);

        isOpen = true;
        return true;
    }
    
    bool init(const char* ip, const char* mask, const char* gateway) {
        isOpen = false;
        eth.set_network(ip, mask, gateway);
        int err = eth.connect();
        if (err != 0) return false;

        sock.open(&eth);
        sock.set_blocking(false);
        sock.bind(12345);

        isOpen = true;
        return true;
    }    
    
    const char* getIPAdr() {
        if (!isOpen) return "";
        return eth.get_ip_address();
    }
    
    const char* getLatestMessage() {
        return sendBuff;
    }
    
    bool sendA(ColorID colorID) {
        if (!isOpen) return false;
                
        sendMessage(MESSAGE_A, colorID);
        return true;
    }
    
    bool sendB(ColorID colorID, CourseID courseID, int contentID) {
        if (!isOpen) return false;
        
        sendMessage(MESSAGE_B, colorID, courseID, contentID);
        return true;
    }
    
    bool sendC(ColorID colorID) {
        if (!isOpen) return false;
                
        sendMessage(MESSAGE_C, colorID);
        return true;
    }
    
    bool sendD(ColorID colorID, CourseID courseID) {
        if (!isOpen) return false;

        sendMessage(MESSAGE_D, colorID, courseID);
        return true;
    }
    
    bool receive(MessageID &msgID, int &timeStamp, ColorID &colorID, CourseID &courseID) {
        msgID = MESSAGE_NONE;
        timeStamp = 0;
        colorID = COLOR_NONE;
        courseID = COURSE_NONE;

        SocketAddress recAdr;
        nsapi_size_or_error_t result = sock.recvfrom(&recAdr, recBuff, sizeof(recBuff));

        if (result < 0) {
            if (result != NSAPI_ERROR_WOULD_BLOCK) {
                printf("sock.recvfrom error: %d\n", result);
            }
            return false;
        }
        recBuff[result] = '\0';

        char* tmp = strtok(recBuff, ",");
        if (strcmp(tmp, "01") == 0) {
            msgID = MESSAGE_A;
        } else if (strcmp(tmp, "02") == 0) {
            msgID = MESSAGE_B;
        } else if (strcmp(tmp, "03") == 0) {
            msgID = MESSAGE_C;
        } else if (strcmp(tmp, "04") == 0) {
            msgID = MESSAGE_D;
        } else if (strcmp(tmp, "05") == 0) {
            msgID = MESSAGE_E;
        } else if (strcmp(tmp, "08") == 0) {
            msgID = MESSAGE_H;
            return true;
        } else {
            msgID = MESSAGE_NONE;
            return false;
        }
        
        // TBD: TimeStamp
        tmp = strtok(NULL, ",");
        
        tmp = strtok(NULL, ",");
        switch (msgID) {
            case MESSAGE_A:
            case MESSAGE_B:
            case MESSAGE_C:
            case MESSAGE_D:
                if (strcmp(tmp, "R") == 0) {
                    colorID = COLOR_RED;
                } else if (strcmp(tmp, "B") == 0) {
                    colorID = COLOR_BLUE;
                } else if (strcmp(tmp, "G") == 0) {
                    colorID = COLOR_GREEN;
                } else if (strcmp(tmp, "Y") == 0) {
                    colorID = COLOR_YELLOW;
                } else {
                    msgID = MESSAGE_NONE;
                    return false;
                }
                break;
                
            case MESSAGE_E:
                // TBD
                break;
            case MESSAGE_H:
                //  no param
                break;
        }
        
        tmp = strtok(NULL, ",");
        switch (msgID) {
            case MESSAGE_B:
            case MESSAGE_C:
            case MESSAGE_D:
                if (strcmp(tmp, "L") == 0) {
                    courseID = COURSE_LEFT;
                } else if (strcmp(tmp, "C") == 0) {
                    courseID = COURSE_CENTER;
                } else if (strcmp(tmp, "R") == 0) {
                    courseID = COURSE_RIGHT;
                } else if (strcmp(tmp, "T") == 0) {
                    courseID = COURSE_TOP;
                } else if (strcmp(tmp, "M") == 0) {
                    courseID = COURSE_MIDDLE;
                } else if (strcmp(tmp, "B") == 0) {
                    courseID = COURSE_BOTTOM;
                } else {
                    msgID = MESSAGE_NONE;
                    return false;
                }
                break;
        }
        
        return true;
    }
};
