/**
 * Pachube API interface driver. (Version 0.0.1)
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

#ifndef _PACHUBE_H_
#define _PACHUBE_H_

#include <string>

class Pachube {
public:

    explicit Pachube(std::string APIKEY);

    virtual ~Pachube();

    /*
     * =================================================================
     *  Environments (feeds)
     * =================================================================
     */

    /**
     * List all available feeds: GET /v2/feeds
     *
     * @param page Integer indicating which page of results you are requesting.
     * @param per_page Integer defining how many results to return per page (1 to 1000).
     * @param content String parameter ('full' or 'summary') describing whether we want full or summary results. Full results means all datastream values are returned, summary just returns the environment meta data for each feed.
     * @param q Full text search parameter. Should return any feeds matching this string.
     * @param tag Returns feeds containing datastreams tagged with the search query.
     * @param user Returns feeds created by the user specified.
     * @param units Returns feeds containing datastreams with units specified by the search query.
     * @param status Possible values ('live', 'frozen', or 'all'). Whether to search for only live feeds, only frozen feeds, or all feeds. Defaults to all.
     * @param order Order of returned feeds. Possible values ('created_at', 'retrieved_at', or 'relevance').
     */
    virtual int listAllAvailableFeeds(
        int page,
        int per_page,
        std::string content,
        std::string q,
        std::string tag,
        std::string user,
        std::string units,
        std::string status,
        std::string order,
        std::string &datatext) = 0;

    /**
     * Create new feed: POST /v2/feeds
     */
    virtual int createNewFeed(void) = 0;

    /**
     * Read feed: GET /v2/feeds/<feed_id>
     */
    virtual int readFeed(int feed_id, std::string &datatext) = 0;

    /**
     * Update feed: PUT /v2/feeds/<feed_id>
     */
    virtual int updateFeed(int feed_id) = 0;

    /**
     * Delete feed: DELETE /v2/feeds/<feed_id>
     */
    virtual int deleteFeed(int feed_id) = 0;

    /*
     * =================================================================
     *  Datastreams
     * =================================================================
     */

    /**
     * Create new datastream: POST /v2/feeds/<feed_id>/datastreams
     *
     * @param feed_id Feed ID.
     * @param stream_id Stream ID text.
     * @param value value.
     *
     * @return Return code from a web server.
     */
    virtual int createNewDataStream(int feed_id, std::string stream_id, std::string value) = 0;

    /**
     * Read datastream - GET /v2/feeds/<feed_id>/datastreams/<datastream_id>
     *
     * @param feed_id Feed ID.
     * @param stream_id Stream ID text.
     *
     * @return Return code from a web server.
     */
    virtual int readDataStream(int feed_id, std::string stream_id, std::string &datatext) = 0;

    /**
     * Update datastream: PUT /v2/feeds/<feed_id>/datastreams/<datastream_id>
     *
     * @param feed_id Feed ID.
     * @param stream_id Stream ID text.
     * @param value value.
     *
     * @return Return code from a web server.
     */
    virtual int updateDataStream(int feed_id, std::string stream_id, std::string value) = 0;

    /**
     * Delete datastream: DELETE /v2/feeds/<feed_id>/datastreams/<datastream_id>
     *
     * @param feed_id Feed ID.
     * @param stream_id Stream ID text.
     *
     * @return Return code from a web server.
     */
    virtual int deleteDataStream(int feed_id, std::string stream_id) = 0;

protected:
    const std::string APIKEY;
    static const std::string REQUESTHEADER;
    static const std::string URLBASE_V2;
};

#endif

/*
Pachube v2 API Documentation
http://api.pachube.com/v2/

# Environments (feeds)

o   * List all available feeds: GET /v2/feeds
o   * Create new feed: POST /v2/feeds
o   * Read feed: GET /v2/feeds/<feed_id>
o   * Update feed: PUT /v2/feeds/<feed_id>
o   * Delete feed: DELETE /v2/feeds/<feed_id>

# Datastreams

o   * Create new datastream: POST /v2/feeds/<feed_id>/datastreams
o   * Read datastream - GET /v2/feeds/<feed_id>/datastreams/<datastream_id>
o   * Update datastream: PUT /v2/feeds/<feed_id>/datastreams/<datastream_id>
o   * Delete datastream: DELETE /v2/feeds/<feed_id>/datastreams/<datastream_id>

# Datapoints

    * Create datapoint: POST /v2/feeds/<feed_id>/datastreams/<datastream_id>/datapoints
    * Read datapoint: GET /v2/feeds/<feed_id>/datastreams/<datastream_id>/datapoints/<timestamp>
    * Update datapoint: PUT /v2/feeds/<feed_id>/datastreams/<datastream_id>/datapoints/<timestamp>
    * Delete datapoint: DELETE /v2/feeds/<feed_id>/datastreams/<datastream_id>/datapoints/<timestamp>

# Triggers

    * List triggers: GET /v2/triggers
    * Create trigger: POST /v2/triggers
    * Read trigger: GET /v2/triggers/<trigger_id>
    * Update trigger: PUT /v2/triggers/<trigger_id>
    * Delete trigger: DELETE /v2/triggers/<trigger_id>

# Users

    * List all users: GET /v2/users
    * Create user: POST /v2/users
    * Read user: GET /v2/users/<user_id>
    * Update user: PUT /v2/users/<user_id>
    * Delete user: DELETE /v2/users/<user_id>
*/