/*******************************************************/
/* file: ports.c                                       */
/* abstract:  This file contains the routines to       */
/*            output values on the JTAG ports, to read */
/*            the TDO bit, and to read a byte of data  */
/*            from the prom                            */
/* Revisions:                                          */
/* 12/01/2008:  Same code as before (original v5.01).  */
/*              Updated comments to clarify instructions.*/
/*              Add print in setPort for xapp058_example.exe.*/
/*******************************************************/
#include "ports.h"
#include "mbed.h"

typedef struct work {
    FILE *fp;
    int done;
    int total;
    void (*cbfunc_progress)(int done, int total);
    void (*cbfunc_waittime)(int microsec);
} work_t;

work_t work;
DigitalOut g_iTCK(p15); /* For mbed */
DigitalOut g_iTMS(p16); /* For mbed */
DigitalOut g_iTDI(p17); /* For mbed */
DigitalIn  g_iTDO(p18); /* For mbed */

void initPort(FILE *fp, void (*cbfunc_progress)(int done, int total), void (*cbfunc_waittime)(int microsec)) {
    work.fp = fp;
    fseek(fp, 0, SEEK_END);
    work.total = ftell(fp);
    work.done = 0;
    fseek(fp, 0, SEEK_SET);
    work.cbfunc_progress = cbfunc_progress;
    work.cbfunc_waittime = cbfunc_waittime;
}

/* setPort:  Implement to set the named JTAG signal (p) to the new value (v).*/
/* if in debugging mode, then just set the variables */
void setPort(short p,short val) {
    /*
     * Modified for mbed.
     */
    /* Printing code for the xapp058_example.exe.  You must set the specified
       JTAG signal (p) to the new value (v).  See the above, old Win95 code
       as an implementation example. */
    if (p == TCK) {
        g_iTCK = val;
        // printf( "TCK = %d;  TMS = %d;  TDI = %d\n", g_iTCK.read(), g_iTMS.read(), g_iTDI.read() );
        return;
    }
    if (p == TMS) {
        g_iTMS = val;
        return;
    }
    if (p == TDI) {
        g_iTDI = val;
        return;
    }
}

/* toggle tck LH.  No need to modify this code.  It is output via setPort. */
void pulseClock() {
    setPort(TCK,0);  /* set the TCK port to low  */
    setPort(TCK,1);  /* set the TCK port to high */
}

/* readByte:  Implement to source the next byte from your XSVF file location */
/* read in a byte of data from the prom */
void readByte(unsigned char *data) {
    /*
     * Modified for mbed.
     */
    *data   = (unsigned char)fgetc( work.fp );

    work.done++;
    if (work.cbfunc_progress != NULL) {
        work.cbfunc_progress(work.done, work.total);
    }
}

/* readTDOBit:  Implement to return the current value of the JTAG TDO signal.*/
/* read the TDO bit from port */
unsigned char readTDOBit() {
    /*
     * Modified for mbed.
     */
    return( (unsigned char) g_iTDO );
}

/* waitTime:  Implement as follows: */
/* REQUIRED:  This function must consume/wait at least the specified number  */
/*            of microsec, interpreting microsec as a number of microseconds.*/
/* REQUIRED FOR SPARTAN/VIRTEX FPGAs and indirect flash programming:         */
/*            This function must pulse TCK for at least microsec times,      */
/*            interpreting microsec as an integer value.                     */
/* RECOMMENDED IMPLEMENTATION:  Pulse TCK at least microsec times AND        */
/*                              continue pulsing TCK until the microsec wait */
/*                              requirement is also satisfied.               */
void waitTime(long microsec) {
    /*
     * Modified for mbed.
     */

    static long tckCyclesPerMicrosec    = 2; /* must be at least 1 */
    long        tckCycles   = microsec * tckCyclesPerMicrosec;
    long        i;

    if (work.cbfunc_waittime != NULL) {
        work.cbfunc_waittime(microsec);
    }

    /* This implementation is highly recommended!!! */
    /* This implementation requires you to tune the tckCyclesPerMicrosec
       variable (above) to match the performance of your embedded system
       in order to satisfy the microsec wait time requirement. */
    for ( i = 0; i < tckCycles; ++i ) {
        pulseClock();
    }
}
