
#ifndef _TEXT_LCD_SR4_H
#define _TEXT_LCD_SR4_H

#include "mbed.h"

/*
 * Modified for http://mbed.org/users/shintamainjp/notebook/textlcd_sr4_en/
 * Shinichiro Nakamura
 */

/**
 * A TextLCD_SR4 interface for driving 4-bit HD44780-based LCDs with shift register.
 *
 * Currently supports 16x2, 20x2 and 20x4 panels
 *
 * @code
 * #include "mbed.h"
 * #include "TextLCD_SR4.h"
 *
 * TextLCD_SR4 lcd(p29, p30, p28, p27); // rs, e, dat, clk
 *
 * int main() {
 *     lcd.printf("Hello World!\n");
 * }
 * @endcode
 */
class TextLCD_SR4 : public Stream {
public:
    /** LCD panel format */
    enum LCDType {
        LCD16x2     /**< 16x2 LCD panel (default) */
        , LCD16x2B  /**< 16x2 LCD panel alternate addressing */
        , LCD20x2   /**< 20x2 LCD panel */
        , LCD20x4   /**< 20x4 LCD panel */
    };
    
    /** Create a TextLCD_SR4 interface
     *
     * @param rs    Instruction/data control line
     * @param e     Enable line (clock)
     * @param dat   Data line
     * @param clk   Clock line
     */
    TextLCD_SR4(PinName rs, PinName e, PinName dat, PinName clk, LCDType type = LCD16x2);

#if DOXYGEN_ONLY
    /** Write a character to the LCD
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /** Write a formated string to the LCD
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formating the string.
     */
    int printf(const char* format, ...);
#endif

    /** Locate to a screen column and row
     *
     * @param column  The horizontal position from the left, indexed from 0
     * @param row     The vertical position from the top, indexed from 0
     */
    void locate(int column, int row);

    /** Clear the screen and locate to 0,0 */
    void cls();

    int rows();
    int columns();

protected:

    // Stream implementation functions
    virtual int _putc(int value);
    virtual int _getc();

    int address(int column, int row);
    void character(int column, int row, int c);
    void writeByte(int value);
    void writeCommand(int command);
    void writeData(int data);

    void writeToShiftRegister(int value);

    DigitalOut _rs, _e, _dat, _clk;
    LCDType _type;

    int _column;
    int _row;
};

#endif
