/**
 * StarBoard Orange - Example application No.4 (Version 0.0.1)
 * Make a GPS logger!
 *
 * See also ... http://mbed.org/users/shintamainjp/notebook/starboard_example4_ja/
 * See also ... http://mbed.org/users/shintamainjp/notebook/starboard_example4_en/
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

/*
 * Connection map.
 *
 * +---+----------------+---------+
 * |Pin|Target          |Direction|
 * +---+----------------+---------+
 * |
 * +---+----------------+---------+
 */

#include "mbed.h"
#include "TextLCD.h"
#include "SerialGPS.h"
#include "SDFileSystem.h"
#include "ReceiverIR.h"
#include "RemoteIR.h"

/*
 * Objects.
 */
TextLCD lcd(p24, p26, p27, p28, p29, p30);
BusOut led(LED4, LED3, LED2, LED1);
SerialGPS gps(p9, p10);
SDFileSystem sd(p5, p6, p7, p8, "sd");
ReceiverIR ir_rx(p15);

/*
 * Variables.
 */
static int display_mode = 0;

static const int SAMPLE_PER_FILE = 500;
typedef struct {
    int count;
    FILE *fp;
    bool enabled;
} log_t;

log_t logger = {
    .count = 0,
    .fp = NULL,
    .enabled = false
};

/*
 * Functions.
 */

/**
 * A ticker function.
 */
void loggle_led(void) {
    led = led + 1;
}

/**
 * A callback function for logging data.
 */
void cbfunc_log(char *s) {
    /*
     * Rotate a file for log.
     */
    if ((logger.count % SAMPLE_PER_FILE) == 0) {
        /*
         * Close a file.
         */
        if (logger.fp != NULL) {
            fclose(logger.fp);
        }
        /*
         * Open file for write.
         */
        char fname[32];
        sprintf(fname, "/sd/gps%05d.txt", logger.count / SAMPLE_PER_FILE);
        logger.fp = fopen(fname, "w");
        if (logger.fp == NULL) {
            logger.enabled = false;
        } else {
            logger.enabled = true;
        }
    }

    /*
     * Write a data to the file.
     */
    if (logger.enabled) {
        if ((fwrite(s, strlen(s), 1, logger.fp) != 1) || (fputc('\n', logger.fp) != '\n')) {
            logger.enabled = false;
            fclose(logger.fp);
            logger.fp = NULL;
        }
    }

    /*
     * Display statuses.
     */
    if (display_mode == 6) {
        lcd.locate(0, 1);
        if (logger.enabled) {
            lcd.printf("%06d:%04d/%04d", logger.count / SAMPLE_PER_FILE, (logger.count % SAMPLE_PER_FILE) + 1, SAMPLE_PER_FILE);
        } else {
            lcd.printf("Logger disabled!");
        }
    }

    /*
     * Count up.
     */
    logger.count++;
    
    loggle_led();
}

/**
 * A callback function for GGA.
 *
 * GGA - Global Positioning System Fixed Data.
 */
void cbfunc_gga(SerialGPS::gps_gga_t *p) {
    if (display_mode == 0) {
        lcd.locate(0, 1);
        lcd.printf("%02d:%02d:%02d(P%d,S%d)", p->hour, p->min, p->sec, p->position_fix, p->satellites_used);
    }

    if (display_mode == 1) {
        lcd.locate(0, 1);
        lcd.printf("%c=%10.4f", p->ns, p->latitude);
    }
    if (display_mode == 2) {
        lcd.locate(0, 1);
        lcd.printf("%c=%10.4f", p->ew, p->longitude);
    }
}

/**
 * A callback function for GSA.
 *
 * GSA - GNSS DOP and Active Satellites.
 */
void cbfunc_gsa(SerialGPS::gps_gsa_t *p) {
    if (display_mode == 3) {
        lcd.locate(0, 1);
        lcd.printf("SEL:%c FIX:%d", p->selmode, p->fix);
    }
}

/**
 * A callback function for GSV.
 *
 * GSV - GNSS Satellites in View.
 */
void cbfunc_gsv(SerialGPS::gps_gsv_t *p) {
    if (display_mode == 4) {
        lcd.locate(0, 1);
        lcd.printf("Satellites:%2d", p->satcnt);
    }
}

/**
 * A callback function for RMC.
 *
 * RMC - Recommended Minimum Specific GNSS Data.
 */
void cbfunc_rmc(SerialGPS::gps_rmc_t *p) {
    if (display_mode == 5) {
        lcd.locate(0, 1);
        lcd.printf("%02d:%02d:%02d(%c)", p->hour, p->min, p->sec, p->status);
    }
}

/**
 * Entry point.
 */
int main() {
    /*
     * Attach callback functions.
     */
    SerialGPS::gps_callback_t cb;
    cb.cbfunc_log = cbfunc_log;
    cb.cbfunc_gga = cbfunc_gga;
    cb.cbfunc_gsa = cbfunc_gsa;
    cb.cbfunc_gsv = cbfunc_gsv;
    cb.cbfunc_rmc = cbfunc_rmc;
    gps.attach(&cb);

    /*
     * Initial display.
     */
    lcd.cls();
    lcd.printf("GGA (Time)");

    /*
     * Application loop.
     */
    int irlen;
    uint8_t irbuf[32];
    RemoteIR::Format irfmt;
    while (1) {
        /*
         * GPS processing.
         */
        gps.processing();

        /*
         * Get a data from IR receiver.
         */
        irlen = ir_rx.getData(&irfmt, irbuf, sizeof(irbuf) * 8);
        if (0 < irlen) {
            uint64_t n = 0;
            for (int i = 0; i < irlen; i++) {
                if (irbuf[i / 8] & (1 << (i % 8))) {
                    n = n | (1 << i);
                }
            }
            if ((irlen == 12) && (irfmt == RemoteIR::SONY)) {
                switch (n) {
                    case 0x80:
                        display_mode = 0;
                        lcd.cls();
                        lcd.printf("GGA (Time)");
                        break;
                    case 0x81:
                        display_mode = 1;
                        lcd.cls();
                        lcd.printf("GGA (Latitude)");
                        break;
                    case 0x82:
                        display_mode = 2;
                        lcd.cls();
                        lcd.printf("GGA (Longitude)");
                        break;
                    case 0x83:
                        display_mode = 3;
                        lcd.cls();
                        lcd.printf("GSA (Statuses)");
                        break;
                    case 0x84:
                        display_mode = 4;
                        lcd.cls();
                        lcd.printf("GSV (Satellites)");
                        break;
                    case 0x85:
                        display_mode = 5;
                        lcd.cls();
                        lcd.printf("RMC (Time)");
                        break;
                    case 0x86:
                        display_mode = 6;
                        lcd.cls();
                        lcd.printf("LOGGER (Status)");
                        break;
                    default:
                        break;
                }
            }
        }
    }
}
