/**
 * StarBoard Orange - Example application No.3 (Version 0.0.1)
 * Drive a CHORO Q HYBRID with wii nunchuk
 *
 * See also ... http://mbed.org/users/shintamainjp/notebook/starboard_example3_ja/
 * See also ... http://mbed.org/users/shintamainjp/notebook/starboard_example3_en/
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

/*
 * Connection map.
 *
 * +---+----------------+---------+
 * |Pin|Target          |Direction|
 * +---+----------------+---------+
 * |p21|IR transmitter  |OUT      |
 * +---+----------------+---------+
 */

/*
 * Include files.
 */

#include <mbed.h>
#include <algorithm>
#include <ChoroQ.h>
#include <I2CConfig.h>
#include <WiiNunchuckReader.h>
#include <TextLCD.h>
#include "appconf.h"

/*
 * Objects.
 */

WiiNunchuckReader wn(I2CPort_A::SDA, I2CPort_A::SCL);
ChoroQ cq(p21);
TextLCD lcd(p24, p26, p27, p28, p29, p30);
BusOut led(LED4, LED3, LED2, LED1);

/**
 * Display a splash screen.
 */
void splash(void) {
    lcd.cls();
    lcd.locate(0, 0);
    lcd.printf("StarBoard Orange");
    lcd.locate(0, 1);
    lcd.printf("mbed NXP LPC1768");
    wait(3);

    lcd.cls();
    lcd.locate(0, 0);
    lcd.printf("Example app No.3");
    lcd.locate(0, 1);
    lcd.printf(" CHORO Q HYBRID ");
    wait(3);
}

/**
 * Get an action from a coordinate.
 *
 * @param x X axis.
 * @param y Y axis.
 * @param dash State of dash.
 *
 * @return An action.
 */
ChoroQ::Action getAction(const int x, const int y, bool dash) {
    static const int MAX_X = 200;
    static const int MIN_X = 20;
    static const int MAX_Y = 200;
    static const int MIN_Y = 20;
    int px = ((x - MIN_X) * 100) / (MAX_X - MIN_X);
    int py = ((y - MIN_Y) * 100) / (MAX_Y - MIN_Y);
    px = std::max(0, std::min(100, px)) - 50;   // Range of a value is -50 to +50
    py = std::max(0, std::min(100, py)) - 50;   // Range of a value is -50 to +50

    if ((std::abs(px) <= 10) && (std::abs(py) <= 10)) {
        return ChoroQ::Stop;
    }

    if ((x == 0) && (y == 0)) {
        return ChoroQ::Stop;
    }

    if (std::abs(px) < 10) {
        if (py < 0) {
            if (dash) {
                return ChoroQ::DownDash;
            } else {
                return ChoroQ::Down;
            }
        } else {
            if (dash) {
                return ChoroQ::UpDash;
            } else {
                return ChoroQ::Up;
            }
        }
    }
    if (std::abs(py) < 10) {
        if (px < -20) {
            return ChoroQ::Left;
        }
        if (20 < px) {
            return ChoroQ::Right;
        }
    }
    if ((10 < px) && (10 < py)) {
        if (dash) {
            return ChoroQ::UpRightDash;
        } else {
            return ChoroQ::UpRight;
        }
    }
    if ((px < -10) && (10 < py)) {
        if (dash) {
            return ChoroQ::UpLeftDash;
        } else {
            return ChoroQ::UpLeft;
        }
    }
    if ((px < -10) && (py < -10)) {
        if (dash) {
            return ChoroQ::DownLeftDash;
        } else {
            return ChoroQ::DownLeft;
        }
    }
    if ((10 < px) && (py < -10)) {
        if (dash) {
            return ChoroQ::DownRightDash;
        } else {
            return ChoroQ::DownRight;
        }
    }
    return ChoroQ::Stop;
}

/**
 * Entry point.
 */
int main() {
    /*
     * Splash.
     */
    splash();
    lcd.cls();

    /*
     * Setup a configuration.
     */
    appconf_t conf;
    appconf_init(&conf);
    appconf_read(&conf);

    /*
     * Application loop.
     */
    while (true) {
        wn.RequestRead();

        ChoroQ::Action ac = getAction(wn.getJoyX(), wn.getJoyY(), (wn.getButtonZ() == 1) ? true : false);
        cq.execute(conf.channel, ac);

        lcd.locate(0, 0);
        lcd.printf("JS  :(%3d,%3d) %s", wn.getJoyX(), wn.getJoyY(), ((wn.getButtonZ() == 1) ? "D" : " "));

        char *stattext = "";
        switch (ac) {
            case ChoroQ::Undef:
                stattext = "Undef";
                break;
            case ChoroQ::Up:
                stattext = "Up";
                break;
            case ChoroQ::Down:
                stattext = "Down";
                break;
            case ChoroQ::Left:
                stattext = "Left";
                break;
            case ChoroQ::Right:
                stattext = "Right";
                break;
            case ChoroQ::UpDash:
                stattext = "Up       +D";
                break;
            case ChoroQ::UpLeft:
                stattext = "UpLeft";
                break;
            case ChoroQ::UpRight:
                stattext = "UpRight";
                break;
            case ChoroQ::UpRightDash:
                stattext = "UpRight  +D";
                break;
            case ChoroQ::UpLeftDash:
                stattext = "UpLeft   +D";
                break;
            case ChoroQ::DownLeft:
                stattext = "DownLeft";
                break;
            case ChoroQ::DownRight:
                stattext = "DownRight";
                break;
            case ChoroQ::DownDash:
                stattext = "Down     +D";
                break;
            case ChoroQ::DownLeftDash:
                stattext = "DownLeft +D";
                break;
            case ChoroQ::DownRightDash:
                stattext = "DownRight+D";
                break;
            case ChoroQ::Stop:
                stattext = "Stop";
                break;
        }
        lcd.locate(0, 1);
        lcd.printf("Ch.%c:%-11.11s", 'A' + (int)conf.channel, stattext);

        wait_ms(50);
        led = led + 1;
    }
}
