/**
 * StarBoard Orange - Example application No.2 (Version 0.0.4)
 * Remote IR receiver with StarBoard Orange
 *
 * See also ... http://mbed.org/users/shintamainjp/notebook/starboard_example2_ja/
 * See also ... http://mbed.org/users/shintamainjp/notebook/starboard_example2_en/
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

#include <mbed.h>

#include "ReceiverIR.h"
#include "TransmitterIR.h"
#include "TextLCD.h"

ReceiverIR ir_rx(p15);
TransmitterIR ir_tx(p21);
TextLCD lcd(p24, p26, p27, p28, p29, p30);
BusOut led(LED4, LED3, LED2, LED1);
Ticker ledTicker;

/**
 * Display a splash screen.
 */
void splash(void) {
    lcd.cls();
    lcd.locate(0, 0);
    lcd.printf("StarBoard Orange");
    lcd.locate(0, 1);
    lcd.printf("mbed NXP LPC1768");
    wait(3);

    lcd.cls();
    lcd.locate(0, 0);
    lcd.printf("Example app No.2");
    lcd.locate(0, 1);
    lcd.printf("    Remote IR   ");
    wait(3);
}

/**
 * Receive.
 *
 * @param format Pointer to a format.
 * @param buf Pointer to a buffer.
 * @param bufsiz Size of the buffer.
 *
 * @return Bit length of the received data.
 */
int receive(RemoteIR::Format *format, uint8_t *buf, int bufsiz, int timeout = 1000) {
    int cnt = 0;
    while (ir_rx.getState() != ReceiverIR::Received) {
        wait_ms(1);
        cnt++;
        if (timeout < cnt) {
            return -1;
        }
    }
    return ir_rx.getData(format, buf, bufsiz * 8);
}

/**
 * Transmit.
 *
 * @param format Format.
 * @param buf Pointer to a buffer.
 * @param bitlength Bit length of the data.
 *
 * @return Bit length of the received data.
 */
int transmit(RemoteIR::Format format, uint8_t *buf, int bitlength, int timeout = 1000) {
    int cnt = 0;
    while (ir_tx.getState() != TransmitterIR::Idle) {
        wait_ms(1);
        cnt++;
        if (timeout < cnt) {
            return -1;
        }
    }
    return ir_tx.setData(format, buf, bitlength);
}

/**
 * Display a current status.
 */
void display_status(char *status) {
    lcd.locate(0, 0);
    lcd.printf("%-12.12s", status);
}

/**
 * Display a format of a data.
 */
void display_format(RemoteIR::Format format) {
    lcd.locate(12, 0);
    switch (format) {
        case RemoteIR::UNKNOWN:
            lcd.printf("????");
            break;
        case RemoteIR::NEC:
            lcd.printf(" NEC");
            break;
        case RemoteIR::AEHA:
            lcd.printf("AEHA");
            break;
        case RemoteIR::SONY:
            lcd.printf("SONY");
            break;
    }
}

/**
 * Display a data.
 *
 * @param buf Pointer to a buffer.
 * @param bitlength Bit length of a data.
 */
void display_data(uint8_t *buf, int bitlength) {
    lcd.locate(0, 1);
    const int n = bitlength / 8 + (((bitlength % 8) != 0) ? 1 : 0);
    for (int i = 0; i < n; i++) {
        lcd.printf("%02X", buf[i]);
    }
    for (int i = 0; i < 8 - n; i++) {
        lcd.printf("--");
    }
}

void ledfunc(void) {
    led = led + 1;
}

/**
 * Entry point.
 */
int main(void) {

    ledTicker.attach(&ledfunc, 0.5);

    /*
     * Splash.
     */
    splash();

    /*
     * Initialize.
     */
    led = 0;
    lcd.cls();
    lcd.locate(0, 0);
    lcd.printf("Press a button  ");
    lcd.locate(0, 1);
    lcd.printf("on a controller.");
    wait_ms(2000);

    lcd.cls();

    /*
     * Execute.
     */
    while (1) {
        uint8_t buf1[32];
        uint8_t buf2[32];
        int bitlength1;
        int bitlength2;
        RemoteIR::Format format;

        memset(buf1, 0x00, sizeof(buf1));
        memset(buf2, 0x00, sizeof(buf2));

        {
            display_status("1:RX>  >  ");
            bitlength1 = receive(&format, buf1, sizeof(buf1));
            if (bitlength1 < 0) {
                continue;
            }
            display_data(buf1, bitlength1);
            display_format(format);
        }

#if 0
        wait_ms(1000);

        {
            display_status("2:RX>TX>  ");
            bitlength1 = transmit(format, buf1, bitlength1);
            if (bitlength1 < 0) {
                continue;
            }
            display_data(buf1, bitlength1);
            display_format(format);
        }

        wait_ms(100);

        {
            display_status("3:RX>TX>RX");
            bitlength2 = receive(&format, buf2, sizeof(buf2));
            if (bitlength2 < 0) {
                continue;
            }
            display_data(buf2, bitlength2);
            display_format(format);
        }

        wait_ms(100);

        {
            for (int i = 0; i < sizeof(buf1); i++) {
                if (buf1[i] != buf2[i]) {
                    display_status("Compare err.");
                    wait(1);
                    continue;
                }
            }
        }
#endif
    }
}