/**
 * SG12864A Graphics LCD module driver class (Version 0.0.1)
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 *
 * See also ...
 * http://mbed.org/users/shintamainjp/notebook/sg12864asunlike-display-graphics-lcd-driver/
 */

#ifndef _SG12864A_H_
#define _SG12864A_H_

#include "mbed.h"

/**
 * SG12864A Graphics LCD module driver class.
 */
class SG12864A {
public:

    /**
     * Create.
     *
     * @param di D-/I (H:Instruction, L:Data)
     * @param rw R/W- (H:Read, L:Write)
     * @param en Enable signal
     * @param db0 Data bus line bit-0.
     * @param db1 Data bus line bit-1.
     * @param db2 Data bus line bit-2.
     * @param db3 Data bus line bit-3.
     * @param db4 Data bus line bit-4.
     * @param db5 Data bus line bit-5.
     * @param db6 Data bus line bit-6.
     * @param db7 Data bus line bit-7.
     * @param cs1 Chip select signal for IC1.
     * @param cs2 Chip select signal for IC2.
     * @param res Reset signal.
     */
    SG12864A(
        PinName di,
        PinName rw,
        PinName en,
        PinName db0,
        PinName db1,
        PinName db2,
        PinName db3,
        PinName db4,
        PinName db5,
        PinName db6,
        PinName db7,
        PinName cs1,
        PinName cs2,
        PinName res);
    /**
     * Destroy.
     */
    ~SG12864A();

    /**
     * Target of a chip.
     */
    enum Target {
        CS1,
        CS2
    };

    /*
     * =======================================================
     * High Level Interfaces.
     * =======================================================
     */

    /**
     * Push images from a internal buffer.
     */
    void bufferPush(void);

    /**
     * Pull images to a internal buffer.
     */
    void bufferPull(void);

    /**
     * Clear a internal buffer images.
     *
     * @param reverse True if images are reversed.
     */
    void bufferClear(bool reverse = false);

    /**
     * Draw a line to a internal buffer.
     *
     * @param x1 Starting point at x-axis.
     * @param y1 Starting point at y-axis.
     * @param x2 Ending point at x-axis.
     * @param y2 Ending point at y-axis.
     * @param reverse True if images are reversed.
     */
    void bufferDrawLine(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, bool reverse = false);

    /**
     * Draw a box to a internal buffer.
     *
     * @param x1 Starting point at x-axis.
     * @param y1 Starting point at y-axis.
     * @param x2 Ending point at x-axis.
     * @param y2 Ending point at y-axis.
     * @param reverse True if images are reversed.
     */
    void bufferDrawBox(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, bool reverse = false);

    /**
     * Fill a box to a internal buffer.
     *
     * @param x1 Starting point at x-axis.
     * @param y1 Starting point at y-axis.
     * @param x2 Ending point at x-axis.
     * @param y2 Ending point at y-axis.
     * @param reverse True if images are reversed.
     */
    void bufferFillBox(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, bool reverse = false);

    /**
     * Draw a text string to a internal buffer.
     * The font size is 5x7 dots.
     *
     * @param x Starting point at x-axis.
     * @param y Starting point at y-axis.
     * @param str Text string. (NULL terminate.)
     * @param reverse True if images are reversed.
     */
    void bufferDrawString(uint8_t x, uint8_t y, char * str, bool reverse = false);

    /**
     * Draw a character to a internal buffer.
     * The font size is 5x7 dots.
     *
     * @param x Starting point at x-axis.
     * @param y Starting point at y-axis.
     * @param c A character.
     * @param reverse True if images are reversed.
     */
    void bufferDrawChar(uint8_t x, uint8_t y, char c, bool reverse = false);

    /**
     * Draw a checkbox to a internal buffer.
     *
     * @param x1 Starting point at x-axis.
     * @param y1 Starting point at y-axis.
     * @param x2 Ending point at x-axis.
     * @param y2 Ending point at y-axis.
     * @param state True if state is checked.
     * @param reverse True if images are reversed.
     */
    void bufferDrawCheckbox(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, bool state, bool reverse = false);

    /**
     * Draw a progressbar to a internal buffer.
     *
     * @param x1 Starting point at x-axis.
     * @param y1 Starting point at y-axis.
     * @param x2 Ending point at x-axis.
     * @param y2 Ending point at y-axis.
     * @param min Minimum value on a scale.
     * @param max Maximum value on a scale.
     * @param value Current value on a scale.
     * @param reverse True if images are reversed.
     */
    void bufferDrawProgressbar(uint16_t x1, uint16_t y1, uint16_t x2, uint16_t y2, int min, int max, int value, bool reverse = false);

    /*
     * High Level Interfaces.
     */

    /**
     * Reset module.
     */
    void reset(void);

    /**
     * Clear display.
     */
    void clear(void);

    /*
     * =======================================================
     * Middle Level Interfaces.
     * =======================================================
     */

    /**
     * Set display ON/OFF.
     *
     * @param t Target.
     * @param on True if the display is ON.
     */
    void setDisplayOnOff(Target t, bool on);
    
    /**
     * Set display start line.
     *
     * @param t Target.
     * @param displayStartLine Start line number.
     */
    void setDisplayStartLine(Target t, uint8_t displayStartLine);
    
    /**
     * Set page address.
     *
     * @param t Target.
     * @param addr Address.
     */
    void setPageAddress(Target t, uint8_t addr);

    /**
     * Set column address.
     *
     * @param t Target.
     * @param addr Address.
     */
    void setColumnAddress(Target t, uint8_t addr);

    /**
     * Read status.
     *
     * @param t Target.
     * @param c Status.
     */
    void readStatus(Target t, uint8_t *c);

    /**
     * Write data.
     *
     * @param t Target.
     * @param c Status.
     */
    void writeData(Target t, uint8_t c);

    /**
     * Read data.
     *
     * @param t Target.
     * @param c Status.
     */
    void readData(Target t, uint8_t *c);

    static const int PIXEL_X = 128;
    static const int PIXEL_Y = 64;
    static const uint8_t FONT_X = 5;
    static const uint8_t FONT_Y = 7;
private:
    static const int PAGES = 8;
    static const int COLUMNS = 64;
    static const uint16_t FONT_MIN_CODE = 0x20;
    static const uint16_t FONT_MAX_CODE = 0x7F;
    static const uint8_t font5x7_data[];
    uint8_t buffer[PAGES * COLUMNS * 2];
    DigitalOut ioDI;
    DigitalOut ioRW;
    DigitalOut ioEN;
    BusInOut ioDB;
    DigitalOut ioCS1;
    DigitalOut ioCS2;
    DigitalOut ioRES;

    /**
     * Mode of a write data.
     */
    enum Mode {
        Data,
        Instruction
    };

    /*
     * =======================================================
     * Low Level Interfaces.
     * =======================================================
     */

    /**
     * Set I/O direction to read.
     */
    void setDirectionToRead();

    /**
     * Set I/O direction to write.
     */
    void setDirectionToWrite();

    /**
     * Write data.
     *
     * @param t Target.
     * @param m Mode.
     * @param c Data.
     */
    void write(Target t, Mode m, uint8_t c);

    /**
     * Read data.
     *
     * @param t Target.
     * @param m Mode.
     * @param c Data.
     */
    void read(Target t, Mode m, uint8_t *c);

    /**
     * Reset.
     *
     * @param b True if reset active.
     */
    void reset(bool b);
};

#endif
