/*---------------------------------------------------------------*/
/* Petit FAT file system module test program R0.02 (C)ChaN, 2009 */
/*---------------------------------------------------------------*/

/*
 * Ported by Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 * 2010/06/01
 */

#include <string.h>
#include "mbed.h"
#include "libpff/diskio.h"
#include "libpff/pff.h"

Serial pc(USBTX, USBRX);

char Line[128];        /* Console input buffer */

static
void put_rc (FRESULT rc) {
    const char *p;
    static const char str[] =
        "OK\0" "DISK_ERR\0" "NOT_READY\0" "NO_FILE\0" "NO_PATH\0"
        "NOT_OPENED\0" "NOT_ENABLED\0" "NO_FILE_SYSTEM\0";
    uint8_t i;

    for (p = str, i = 0; i != (uint8_t)rc && p; i++) {
        while (p++);
    }
    pc.printf("rc=%u FR_%S\n", (WORD)rc, p);
}



static
void put_drc (BYTE res) {
    pc.printf("rc=%d\n", res);
}



static
void get_line (char *buff, BYTE len) {
    BYTE c, i;

    i = 0;
    for (;;) {
        c = pc.getc();
        if (c == '\r') break;
        if ((c == '\b') && i) {
            i--;
            pc.putc('\b');
            pc.putc(' ');
            pc.putc('\b');
        }
        if ((c >= ' ') && (i < len - 1)) {
            buff[i++] = c;
            pc.putc(c);
        }
    }
    buff[i] = 0;
    pc.putc('\n');
}



static
void put_dump (const BYTE *buff, DWORD ofs, int cnt) {
    BYTE n;

    pc.printf("%08x", ofs);
    pc.putc(' ');
    for (n = 0; n < cnt; n++) {
        pc.putc(' ');
        pc.printf("%02x", buff[n]);
    }
    pc.printf("  ");
    for (n = 0; n < cnt; n++) {
        pc.putc(((buff[n] < 0x20)||(buff[n] >= 0x7F)) ? '.' : buff[n]);
    }
    pc.putc('\n');
}



/*-----------------------------------------------------------------------*/
/* Main                                                                  */


int main (void) {
    char *ptr;
    long p1, p2;
    BYTE res;
    WORD s1, s2, s3, ofs, cnt, w;
    FATFS fs;            /* File system object */
    FATDIR dir;            /* Directory object */
    FILINFO fno;        /* File information */

    pc.baud(19200);
    pc.printf("\nPFF test monitor\n");

    for (;;) {
        pc.putc('>');
        get_line(Line, sizeof(Line));
        ptr = Line;

        switch (*ptr++) {
            case '?' :
                pc.printf("di                - Initialize physical drive\n");
                pc.printf("dd <sector> <ofs> - Dump partial secrtor 128 bytes\n");
                pc.printf("fi                - Mount the volume\n");
                pc.printf("fo <file>         - Open a file\n");
                pc.printf("fd                - Read the file 128 bytes and dump it\n");
                pc.printf("fw <len> <val>    - Write data to the file\n");
                pc.printf("fp                - Write console input to the file\n");
                pc.printf("fe <ofs>          - Move file pointer of the file\n");
                pc.printf("fl [<path>]       - Directory listing\n");
                break;
            case 'd' :
                switch (*ptr++) {
                    case 'i' :    /* di - Initialize physical drive */
                        res = disk_initialize();
                        put_drc(res);
                        break;
                    case 'd' :    /* dd <sector> <ofs> - Dump partial secrtor 128 bytes */
                        if (2 != sscanf(ptr, "%x %x", &p1, &p2)) {
                            break;
                        }
                        s2 = p2;
                        res = disk_readp((BYTE*)Line, p1, s2, 128);
                        if (res) {
                            put_drc(res);
                            break;
                        }
                        s3 = s2 + 128;
                        for (ptr = Line; s2 < s3; s2 += 16, ptr += 16, ofs += 16) {
                            s1 = (s3 - s2 >= 16) ? 16 : s3 - s2;
                            put_dump((BYTE*)ptr, s2, s1);
                        }
                        break;
                }
                break;
            case 'f' :
                switch (*ptr++) {

                    case 'i' :    /* fi - Mount the volume */
                        put_rc(pf_mount(&fs));
                        break;

                    case 'o' :    /* fo <file> - Open a file */
                        while (*ptr == ' ') ptr++;
                        put_rc(pf_open(ptr));
                        break;
#if _USE_READ
                    case 'd' :    /* fd - Read the file 128 bytes and dump it */
                        ofs = fs.fptr;
                        res = pf_read(Line, sizeof(Line), &s1);
                        if (res != FR_OK) {
                            put_rc((FRESULT)res);
                            break;
                        }
                        ptr = Line;
                        while (s1) {
                            s2 = (s1 >= 16) ? 16 : s1;
                            s1 -= s2;
                            put_dump((BYTE*)ptr, ofs, s2);
                            ptr += 16;
                            ofs += 16;
                        }
                        break;

#endif
#if _USE_WRITE
                    case 'w' :    /* fw <len> <val> - Write data to the file */
                        if (2 != sscanf(ptr, "%x %x", &p1, &p2)) {
                            break;
                        }
                        for (s1 = 0; s1 < sizeof(Line); Line[s1++] = (BYTE)p2) ;
                        p2 = 0;
                        while (p1) {
                            if ((UINT)p1 >= sizeof(Line)) {
                                cnt = sizeof(Line);
                                p1 -= sizeof(Line);
                            } else {
                                cnt = (WORD)p1;
                                p1 = 0;
                            }
                            res = pf_write(Line, cnt, &w);    /* Write data to the file */
                            p2 += w;
                            if (res != FR_OK) {
                                put_rc((FRESULT)res);
                                break;
                            }
                            if (cnt != w) break;
                        }
                        res = pf_write(0, 0, &w);        /* Finalize the write process */
                        put_rc((FRESULT)res);
                        if (res == FR_OK)
                            pc.printf("%lu bytes written.\n", p2);
                        break;
                    case 'p' :    /* fp - Write console input to the file */
                        pc.printf("Type any line to write. A blank line finalize the write operation.\n");
                        for (;;) {
                            get_line(Line, sizeof(Line));
                            if (!Line[0]) break;
                            strcat(Line, "\r\n");
                            res = pf_write(Line, strlen(Line), &w);    /* Write a line to the file */
                            if (res) break;
                        }
                        res = pf_write(0, 0, &w);        /* Finalize the write process */
                        put_rc((FRESULT)res);
                        break;
#endif
#if _USE_LSEEK
                    case 'e' :    /* fe <ofs> - Move file pointer of the file */
                        if (1 != sscanf(ptr, "%x", &p1)) {
                            break;
                        }
                        res = pf_lseek(p1);
                        put_rc((FRESULT)res);
                        if (res == FR_OK) {
                            pc.printf("fptr = %lu(0x%lX)\n", fs.fptr, fs.fptr);
                        }
                        break;
#endif
#if _USE_DIR
                    case 'l' :    /* fl [<path>] - Directory listing */
                        while (*ptr == ' ') ptr++;
                        res = pf_opendir(&dir, ptr);
                        if (res) {
                            put_rc((FRESULT)res);
                            break;
                        }
                        s1 = 0;
                        for (;;) {
                            res = pf_readdir(&dir, &fno);
                            if (res != FR_OK) {
                                put_rc((FRESULT)res);
                                break;
                            }
                            if (!fno.fname[0]) break;
                            if (fno.fattrib & AM_DIR)
                                pc.printf("   <DIR>   %s\n", fno.fname);
                            else
                                pc.printf("%9lu  %s\n", fno.fsize, fno.fname);
                            s1++;
                        }
                        pc.printf("%u item(s)\n", s1);
                        break;
#endif
                }
                break;
        }
    }

}



