/*
 * ===============================================================
 *  Natural Tiny Shell (NT-Shell) Application example.
 *  Version 0.0.6
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "mbed.h"
#include "ntshell.h"
#include "ntopt.h"
#include "cmd.h"

Serial serial(USBTX, USBRX);
ntshell_t ntshell;

typedef struct {
    char *command;
    char *description;
    void (*func)(int argc, char **argv);
} command_table_t;

const command_table_t cmdlist[] = {
    {"foo", "foo command", cmd_foo },
    {"bar", "bar command", cmd_bar },
    {"baz", "baz command", cmd_baz },
    {NULL, NULL, NULL}
};

int func_read(char *buf, int cnt);
int func_write(const char *buf, int cnt);
int func_cb_ntshell(const char *text);
void func_cb_ntopt(int argc, char **argv);

/**
 * Serial read function.
 */
int func_read(char *buf, int cnt) {
    for (int i = 0; i < cnt; i++) {
        buf[i] = serial.getc();
    }
    return 0;
}

/**
 * Serial write function.
 */
int func_write(const char *buf, int cnt) {
    for (int i = 0; i < cnt; i++) {
        serial.putc(buf[i]);
    }
    return 0;
}

/**
 * Callback function for ntshell module.
 */
int func_cb_ntshell(const char *text) {
    return ntopt_parse(text, func_cb_ntopt);
}

/**
 * Callback function for ntopt module.
 */
void func_cb_ntopt(int argc, char **argv) {
    if (argc == 0) {
        return;
    }    
    int execnt = 0;
    const command_table_t *p = &cmdlist[0];
    while (p->command != NULL) {
        if (strcmp(argv[0], p->command) == 0) {
            p->func(argc, argv);
            execnt++;
        }
        p++;
    }
    if (execnt == 0) {
        printf("Command not found.\n");
    }
    wait_ms(250);
}

/**
 * Application code.
 */
int main() {
    ntshell_execute(&ntshell, func_read, func_write, func_cb_ntshell);
}
