/**
 * Test program.
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

/*
 * Include files.
 */

#include "mbed.h"
#include "CameraC328.h"
#include "SDFileSystem.h"

/*
 * Definitions.
 */
#define USE_JPEG_HIGH_RESOLUTION  1
#define USE_SD_CARD 0

/*
 * Variables.
 */
static const int CAPTURE_FRAMES = 5;
static const int RAWIMG_X = 80;
static const int RAWIMG_Y = 60;
static char buf[RAWIMG_X * RAWIMG_Y * 2];
static FILE *fp_jpeg;

/*
 * Modules.
 */
#if USE_SD_CARD
SDFileSystem sd(p5, p6, p7, p8, "fs");
#else
LocalFileSystem fs("fs");
#endif
CameraC328 camera(p9, p10, CameraC328::Baud19200);

/**
 * A callback function for uncompressed images.
 * Please do NOT block this callback function.
 * Because the camera module transmit image datas continuously.
 *
 * @param done a done number of packets.
 * @param total a total number of packets.
 * @param c received data.
 */
void uncompressed_callback(size_t done, size_t total, char c) {
    buf[done - 1] = c;
}

/**
 * A callback function for jpeg images.
 * You can block this function until saving the image datas.
 *
 * @param buf A pointer to the image buffer.
 * @param siz A size of the image buffer.
 */
void jpeg_callback(char *buf, size_t siz) {
    for (int i = 0; i < (int)siz; i++) {
        fprintf(fp_jpeg, "%c", buf[i]);
    }
}

/**
 * Synchronizing.
 */
void sync(void) {
    CameraC328::ErrorNumber err = CameraC328::NoError;

    err = camera.sync();
    if (CameraC328::NoError == err) {
        printf("[ OK ] : CameraC328::sync\n");
    } else {
        printf("[FAIL] : CameraC328::sync (Error=%02X)\n", (int)err);
    }
}

/**
 * A test function for uncompressed snapshot picture.
 */
void test_uncompressed_snapshot_picture(void) {
    CameraC328::ErrorNumber err = CameraC328::NoError;

    err = camera.init(CameraC328::Color16bit, CameraC328::RawResolution80x60, CameraC328::JpegResolution160x128);
    if (CameraC328::NoError == err) {
        printf("[ OK ] : CameraC328::init\n");
    } else {
        printf("[FAIL] : CameraC328::init (Error=%02X)\n", (int)err);
    }

    for (int i = 0; i < CAPTURE_FRAMES; i++) {
        err = camera.getUncompressedSnapshotPicture(uncompressed_callback);
        if (CameraC328::NoError == err) {
            printf("[ OK ] : CameraC328::getUncompressedSnapshotPicture\n");
        } else {
            printf("[FAIL] : CameraC328::getUncompressedSnapshotPicture (Error=%02X)\n", (int)err);
        }

        char fname[64];
        snprintf(fname, sizeof(fname), "/fs/ucss%04d.ppm", i);
        FILE *fp = fopen(fname, "w");
        fprintf(fp, "P3\n");
        fprintf(fp, "%d %d\n", RAWIMG_X, RAWIMG_Y);
        fprintf(fp, "%d\n", 255);
        for (int y = 0; y < RAWIMG_Y; y++) {
            for (int x = 0; x < RAWIMG_X; x++) {
                int adrofs = y * (RAWIMG_X * 2) + (x * 2);
                uint16_t dat = (buf[adrofs + 0] << 8) | (buf[adrofs + 1] << 0);
                uint8_t r = ((dat >> 11) & 0x1f) << 3;
                uint8_t g = ((dat >> 5) & 0x3f) << 2;
                uint8_t b = ((dat >> 0) & 0x1f) << 3;
                fprintf(fp,"%d %d %d\n", r, g, b);
            }
        }
        fclose(fp);
    }
}

/**
 * A test function for uncompressed preview picture.
 */
void test_uncompressed_preview_picture(void) {
    CameraC328::ErrorNumber err = CameraC328::NoError;

    err = camera.init(CameraC328::Color16bit, CameraC328::RawResolution80x60, CameraC328::JpegResolution160x128);
    if (CameraC328::NoError == err) {
        printf("[ OK ] : CameraC328::init\n");
    } else {
        printf("[FAIL] : CameraC328::init (Error=%02X)\n", (int)err);
    }

    for (int i = 0; i < CAPTURE_FRAMES; i++) {
        err = camera.getUncompressedPreviewPicture(uncompressed_callback);
        if (CameraC328::NoError == err) {
            printf("[ OK ] : CameraC328::getUncompressedPreviewPicture\n");
        } else {
            printf("[FAIL] : CameraC328::getUncompressedPreviewPicture (Error=%02X)\n", (int)err);
        }

        char fname[64];
        snprintf(fname, sizeof(fname), "/fs/ucpv%04d.ppm", i);
        FILE *fp = fopen(fname, "w");
        fprintf(fp, "P3\n");
        fprintf(fp, "%d %d\n", RAWIMG_X, RAWIMG_Y);
        fprintf(fp, "%d\n", 255);
        for (int y = 0; y < RAWIMG_Y; y++) {
            for (int x = 0; x < RAWIMG_X; x++) {
                int adrofs = y * (RAWIMG_X * 2) + (x * 2);
                uint16_t dat = (buf[adrofs + 0] << 8) | (buf[adrofs + 1] << 0);
                uint8_t r = ((dat >> 11) & 0x1f) << 3;
                uint8_t g = ((dat >> 5) & 0x3f) << 2;
                uint8_t b = ((dat >> 0) & 0x1f) << 3;
                fprintf(fp,"%d %d %d\n", r, g, b);
            }
        }
        fclose(fp);
    }
}

/**
 * A test function for jpeg snapshot picture.
 */
void test_jpeg_snapshot_picture(void) {
    CameraC328::ErrorNumber err = CameraC328::NoError;

#if USE_JPEG_HIGH_RESOLUTION
    err = camera.init(CameraC328::Jpeg, CameraC328::RawResolution80x60, CameraC328::JpegResolution640x480);
#else
    err = camera.init(CameraC328::Jpeg, CameraC328::RawResolution80x60, CameraC328::JpegResolution320x240);
#endif
    if (CameraC328::NoError == err) {
        printf("[ OK ] : CameraC328::init\n");
    } else {
        printf("[FAIL] : CameraC328::init (Error=%02X)\n", (int)err);
    }

    for (int i = 0; i < CAPTURE_FRAMES; i++) {
        char fname[64];
        snprintf(fname, sizeof(fname), "/fs/jpss%04d.jpg", i);
        fp_jpeg = fopen(fname, "w");

        err = camera.getJpegSnapshotPicture(jpeg_callback);
        if (CameraC328::NoError == err) {
            printf("[ OK ] : CameraC328::getJpegSnapshotPicture\n");
        } else {
            printf("[FAIL] : CameraC328::getJpegSnapshotPicture (Error=%02X)\n", (int)err);
        }

        fclose(fp_jpeg);
    }
}

/**
 * A test function for jpeg preview picture.
 */
void test_jpeg_preview_picture(void) {
    CameraC328::ErrorNumber err = CameraC328::NoError;

#if USE_JPEG_HIGH_RESOLUTION
    err = camera.init(CameraC328::Jpeg, CameraC328::RawResolution80x60, CameraC328::JpegResolution640x480);
#else
    err = camera.init(CameraC328::Jpeg, CameraC328::RawResolution80x60, CameraC328::JpegResolution320x240);
#endif
    if (CameraC328::NoError == err) {
        printf("[ OK ] : CameraC328::init\n");
    } else {
        printf("[FAIL] : CameraC328::init (Error=%02X)\n", (int)err);
    }

    for (int i = 0; i < CAPTURE_FRAMES; i++) {
        char fname[64];
        snprintf(fname, sizeof(fname), "/fs/jppv%04d.jpg", i);
        fp_jpeg = fopen(fname, "w");

        err = camera.getJpegPreviewPicture(jpeg_callback);
        if (CameraC328::NoError == err) {
            printf("[ OK ] : CameraC328::getJpegPreviewPicture\n");
        } else {
            printf("[FAIL] : CameraC328::getJpegPreviewPicture (Error=%02X)\n", (int)err);
        }

        fclose(fp_jpeg);
    }
}

/**
 * A entry point.
 */
int main() {
    printf("\n");
    printf("==========\n");
    printf("CameraC328\n");
    printf("==========\n");

    sync();
    test_uncompressed_snapshot_picture();
    test_uncompressed_preview_picture();
    test_jpeg_preview_picture();
    test_jpeg_snapshot_picture();

    return 0;
}
