/**
 * =============================================================================
 * Aitendo OLED device driver library. (Version 0.0.2)
 *
 * [The product]
 * OLED-2P-095BWNN-SPI : http://www.aitendo.co.jp/product/2099
 * ALO-095BWNN-J9      : http://www.aitendo.co.jp/product/1449
 *
 * [References]
 * http://serdisplib.sourceforge.net/ser/doc/Treiber_IC-SSD1332_OLED_96x64_COLOR.pdf
 * =============================================================================
 * Copyright (c) 2010 Shinichiro Nakamura (CuBeatSystems)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * =============================================================================
 */
#ifndef AITENDO_OLED_H
#define AITENDO_OLED_H

#include "mbed.h"

/**
 * OLED.
 */
class AitendoOLED {
public:
    /**
     * Create.
     *
     * @param pin_mosi MOSI.
     * @param pin miso MISO.
     * @param pin_sclk SCLK.
     * @param pin_res RES.
     * @param pin_cs CS.
     * @param pin_dc DC.
     */
    AitendoOLED(
        PinName pin_mosi,
        PinName pin_miso,
        PinName pin_sclk,
        PinName pin_res,
        PinName pin_cs,
        PinName pin_dc);

    /**
     * Dispose.
     */
    ~AitendoOLED();

    typedef struct color {
        uint8_t r;
        uint8_t g;
        uint8_t b;
    } Color;

    /**
     * Draw pixel.
     *
     * @param x X.
     * @param y Y.
     * @param c Color.
     */
    void drawPixel(int x, int y, Color c);

    /**
     * Draw line.
     *
     * @param x1 X1.
     * @param y1 Y1.
     * @param x2 X2.
     * @param y2 Y2.
     * @param c Color.
     */
    void drawLine(int x1, int y1, int x2, int y2, Color c);

    /**
     * Fill box.
     *
     * @param x1 X1.
     * @param y1 Y1.
     * @param x2 X2.
     * @param y2 Y2.
     * @param c1 Color1.
     * @param c2 Color2.
     */
    void fillBox(int x1, int y1, int x2, int y2, Color c1, Color c2);

    /**
     * Copy.
     *
     * @param x1 X1.
     * @param y1 Y1.
     * @param x2 X2.
     * @param y2 Y2.
     * @param nx X of the destination.
     * @param ny Y of the destination.
     */
    void copy(int x1, int y1, int x2, int y2, int nx, int ny);

    /**
     * Darker.
     *
     * @param x1 X1.
     * @param y1 Y1.
     * @param x2 X2.
     * @param y2 Y2.
     */
    void darker(int x1, int y1, int x2, int y2);

    /**
     * Clear.
     *
     * @param x1 X1.
     * @param y1 Y1.
     * @param x2 X2.
     * @param y2 Y2.
     */
    void clear(int x1, int y1, int x2, int y2);
private:
    AitendoOLED();

    SPI io_spi;
    DigitalOut io_res;
    DigitalOut io_cs;
    DigitalOut io_dc;

    /**
     * Reset.
     */
    void reset();

    /**
     * Write data.
     *
     * @param data Data.
     */
    void writeData(uint8_t data);

    /**
     * Set mode.
     *
     * @param fillrect Fill the rectangle.
     * @param revcopy Reverse copy.
     */
    void setMode(bool fillrect, bool revcopy);

    /**
     * Set column address.
     *
     * @param start Start address. (0-95)
     * @param end End address. (0-95)
     */
    void setColumnAddress(int start, int end);

    /**
     * Set row address.
     *
     * @param start Start address. (0-63)
     * @param end End address. (0-63)
     */
    void setRowAddress(int start, int end);

    /**
     * Set contrast for Color A.
     *
     * @param contrast Contrast. (Default:0x80)
     */
    void setContrastForColorA(int contrast);

    /**
     * Set contrast for Color B.
     *
     * @param contrast Contrast. (Default:0x80)
     */
    void setContrastForColorB(int contrast);

    /**
     * Set contrast for Color C.
     *
     * @param contrast Contrast. (Default:0x80)
     */
    void setContrastForColorC(int contrast);

    /**
     * Set master current control value.
     *
     * @param current Current value. (0x00-0x0F)
     */
    void setMasterCurrentControl(int current);

    /**
     * Set remapping mode and the data format.
     *
     * @param mode See the document.
     */
    void setRemapAndDataFormat(int mode);

    /**
     * Set display start line.
     *
     * @param line Start line number. (0-63)
     */
    void setDisplayStartLine(int line);

    /**
     * Set display offset line.
     *
     * @param offset Offset line number. (0-63)
     */
    void setDisplayOffset(int offset);

    typedef enum {
        NormalDisplay,
        EntireDisplayOn,
        EntireDisplayOff,
        InverseDisplay
    } DisplayMode;

    /**
     * Set display mode.
     *
     * @param mode Display mode.
     */
    void setDisplayMode(DisplayMode mode);

    /**
     * Set multiplex ratio.
     *
     * @param ratio Ratio.
     */
    void setMultiplexRatio(int ratio);

    /**
     * Set display on/off.
     *
     * @param on On.
     */
    void setDisplayOnOff(bool on);

    typedef struct gray_scale_table {
        char data[32];
    } gray_scale_table_t;

    /**
     * Set gray scale table.
     *
     * @param p A pointer to the look up table.
     */
    void setGrayScaleTable(gray_scale_table_t *p);

    /**
     * NOP.
     */
    void nop();

    /**
     * Set power saving mode.
     *
     * @param value Value. (0x00:None, 0x12:power saving)
     */
    void setPowerSavingMode(int value);

    /**
     * Set phase period.
     *
     * @param value Value. (Default:0x74)
     */
    void setPhasePeriod(int value);

    /**
     * Set display clock divide ratio.
     *
     * @param value Value. (Default:0x00)
     */
    void setDisplayClockDivideRatio(int value);

    /**
     * Set second pre-charge speed for color A.
     *
     * @param value Value.
     */
    void setSecondPreChargeSpeedForColorA(int value);

    /**
     * Set second pre-charge speed for color B.
     *
     * @param value Value.
     */
    void setSecondPreChargeSpeedForColorB(int value);

    /**
     * Set second pre-charge speed for color C.
     *
     * @param value Value.
     */
    void setSecondPreChargeSpeedForColorC(int value);

    /**
     * Set pre charge level for color A.
     *
     * @param value The value.
     */
    void setPreChargeLevelForColorA(int value);

    /**
     * Set pre charge level for color B.
     *
     * @param value The value.
     */
    void setPreChargeLevelForColorB(int value);

    /**
     * Set pre charge level for color C.
     *
     * @param value The value.
     */
    void setPreChargeLevelForColorC(int value);

    /**
     * Set VCOMH.
     *
     * @param value VCOMH value. (0x00:0.43 * Vref, 0x3F:0x83 * Vref)
     */
    void setVCOMH(int value);

    /**
     * Read the status register.
     *
     * @return the value.
     */
    uint8_t readStatusRegister();
};

#endif
