#include "mbed.h"
#include "FlashManager.h"
extern RawSerial pc;


//FlashManager::FlashManager(SPI spi, PinName cs, Serial* _pc) : m_mx25r(spi, cs), pc(_pc) { }

FlashManager::FlashManager(SPI spi, PinName cs) : m_mx25r(spi, cs) {
    //spi.frequency(16000000);
}

//FlashManager::FlashManager(PinName mosi, PinName miso, PinName sclk, PinName cs) : _spi(mosi,miso,sclk),m_mx25r(_spi,cs){}


/*FlashManager::~FlashManager()
{
    pc = NULL;
}*/


int min(int a, int b)
{
    return a < b ? a : b ;
}

//vector<uint32_t>をvector<uint8_t>に収めて返す
//サイズは32/8=4倍になる
vector<uint8_t> type32to8(const vector<uint32_t>& data)
{
    vector<uint8_t> temp;
    for(int i = 0; i < data.size()*4; i++) {
        temp.push_back((uint8_t)((data[i/4] >> (24 - (i%4)*8)) & 0xFF));
    }
    return temp;
}

//vector<uint8_t>をvector<uint32_t>に収めて返す
//サイズは8/32=1/4倍になる
vector<uint32_t> type8to32(const vector<uint8_t>& data)
{
    vector<uint32_t> temp;
    for(int i = 0; i < data.size()/4; i++) {
        uint32_t val = 0;
        for(int j = 0; j < 4; j++) {
            val += (uint32_t)data[i*4+j] << (24 - j*8);
        }
        temp.push_back(val);
    }
    return temp;
}

//addrからSPI_MX25R.cppに定義されているread8()を使って、1byte分のデータを読み取り
uint8_t FlashManager::read(int addr)
{
    return m_mx25r.read8(addr);
}

//addr~addr+0x3の4byte分のデータを読み取り、返す
uint32_t FlashManager::read4byte(int addr)
{
    uint32_t temp = 0;
    vector<uint8_t> data = m_mx25r.readFREAD(addr, addr+0x3); 
    for(int i = 0; i < 4; i++) {
        temp += (uint32_t)data[i] << (24 - i*8);
    }
    return temp;
}


vector<uint8_t> FlashManager::readSector(int sector)
{
    return m_mx25r.readFREAD(sector * SECTOR_SIZE, (sector + 1) * SECTOR_SIZE - 0x01);
}


vector<uint8_t> FlashManager::read(int start_addr, int end_addr)
{
    return m_mx25r.readFREAD(start_addr, end_addr);
}


vector<uint32_t> FlashManager::read4byte(int start_addr, int end_addr)
{
    return type8to32(m_mx25r.readFREAD(start_addr, end_addr));
}


void FlashManager::write(int addr, const vector<uint8_t>& data)
{
    // 書き換えが必要なsectorの計算
    int start_sector = addr / SECTOR_SIZE;
    int end_sector = (addr + data.size() - 0x1) / SECTOR_SIZE;

    for(int sector = start_sector; sector <= end_sector; sector++) {
        // bufferに一時退避させ，sector削除
        vector<uint8_t> write_buf = readSector(sector);
        sectorErase(sector);

        // write_bufの書き換え
        for(int i = 0; i < write_buf.size(); i++) {
            int data_index = sector * SECTOR_SIZE + i - addr;
            if(0 <= data_index && data_index < data.size()) {
                write_buf[i] = data[data_index];
            }
        }

        // 変更したwrite_bufをページ単位でメモリ書き込み
        for(int page = 0; page < (write_buf.size() + PAGE_SIZE - 1) / PAGE_SIZE; page++) {
            int length = min(write_buf.size() - page*PAGE_SIZE , PAGE_SIZE);
            m_mx25r.writeEnable(); // send WREN 1st
            m_mx25r.programPage(sector * SECTOR_SIZE + page*PAGE_SIZE, &write_buf[page*PAGE_SIZE], length);
            do {
                wait_ms(TIME_PP);
            } while(isBusyInWriting());
        }
    }
}


void FlashManager::write(int addr, uint8_t data[], int size)
{
    write(addr, vector<uint8_t>(data, data + size));
}


void FlashManager::write4byte(int addr, const vector<uint32_t>& data)
{
    write(addr, type32to8(data));
}


void FlashManager::sectorErase(int sector)
{
    m_mx25r.writeEnable() ; // send WREN 1st
    m_mx25r.sectorErase(sector * SECTOR_SIZE) ;
    do {
        wait_ms(TIME_SE);
    } while(isBusyInWriting());                  // end poll
}


void FlashManager::chipErase()
{
    m_mx25r.writeEnable() ; // send WREN 1st
    m_mx25r.chipErase() ;
    do {
        wait_ms(TIME_CE);
    } while(isBusyInWriting());                  // end poll
}


void FlashManager::reset()
{
    m_mx25r.resetEnable();
    m_mx25r.noOperation();
    m_mx25r.reset();
}


bool FlashManager::isBusyInWriting()
{
    return m_mx25r.readStatus() & 0x01 == 0x01;
}


bool FlashManager::isValidRange(int addr)
{
    return ADDR_MIN <= addr && addr <= ADDR_MAX;
}

//--------------double用に追加した関数--------------//
//addr~addr+0x7に格納されている数値をdoubleとみて返す
double FlashManager::readdouble(int addr){
    vector<uint8_t> datavec = read(addr,addr+0x8-1);
    return *(double*) &datavec[0];
}

//readdoubleをsize分繰り返し、vectorに格納して返す
vector<double> FlashManager::readdouble(int start_addr, int size){
    vector<double> datavec;
    for(int current_addr = start_addr; current_addr <= start_addr + 8*(size-1); current_addr += 8){
        datavec.push_back(readdouble(current_addr));
    }
    return datavec;
}


void FlashManager::writedouble(int addr,double value){
    vector<uint8_t> datavec(8); //8byte(=double)分のデータ構造を用意
    *(double*) &datavec[0] = value;
    write(addr,datavec);
}

void FlashManager::writedouble(int addr, const vector<double>& data){
    vector<uint8_t> uint8data;
    uint8_t temp[8];
    for(int i = 0; i < data.size(); i++){
        *(double*) temp = data[i];
        for(int j=0; j < 8; j++){
            uint8data.push_back(temp[j]);
        }
    }
    write(addr,uint8data);
}
